// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file devel/ProteinInterfaceDesign/design_utils.cc
/// @brief various utilities for interface design.
/// @author Sarel Fleishman (sarelf@u.washington.edu)

// Project Headers
#include <protocols/ProteinInterfaceDesign/read_patchdock.hh>
#include <core/pose/Pose.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/conformation/Conformation.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/options/util.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/util/Tracer.hh>

#include <numeric/random/random.hh>

// Utility Headers
#include <utility/vector1.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>
#include <utility/io/izstream.hh>

// Unit Headers
#include <protocols/ProteinInterfaceDesign/design_utils.hh>

// C++ headers
#include <map>
#include <stdlib.h>
#include <math.h>

// option key includes
#include <core/options/option.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/parser.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>


using namespace core;
using namespace core::scoring;

static core::util::Tracer TR( "protocols.ProteinInterfaceDesign.read_patchdock" );
static numeric::random::RandomGenerator RG( 15031972 ); // <- Magic number, do not change it!!!

struct Transformation
{
	typedef core::Real Real;

	Real alpha, beta, gamma; // Euler angles
	numeric::xyzVector< Real > translation; // translation
};

namespace protocols {
namespace ProteinInterfaceDesign {

///@detailed how many entries are there in the patchdock file?
core::Size
number_of_patchdock_entries( std::string const fname )
{
  utility::io::izstream data( fname );
  if ( !data ) {
    TR << "Cannot open patchdock file: " << fname << std::endl;
    runtime_assert( data );
		return 0;
  }

  std::string line;
	bool entries_found( false );
	core::Size curr_entry( 0 );
  while ( getline( data, line ) ) {
		using namespace std;

		istringstream line_stream( line );
    string first_field;
    line_stream >> first_field;

		if( first_field == "#" ) { entries_found = true; continue; }
		if( !entries_found ) continue;

		core::Size const wheres_pipe( line.find_first_of( "|" ) );
		if( wheres_pipe == string::npos ) break;; // no longer reading entries
		runtime_assert( wheres_pipe == 5 );
		curr_entry = atoi( first_field.c_str() );
	}
	runtime_assert( curr_entry );
	return( curr_entry );
}


///@detailed read a rigid-body transformation from a patchdock file
Transformation
read_patchdock_entry( std::string const fname, core::Size const entry )
{
	Transformation t;
	t.alpha = t.beta = t.gamma = 0;
	t.translation.zero() ;

  utility::io::izstream data( fname );
  if ( !data ) {
    TR << "Cannot open patchdock file: " << fname << std::endl;
    runtime_assert( data );
		return( t );
  }

  std::string line;
	bool entries_found( false );
  while ( getline( data, line ) ) {
		using namespace std;

		istringstream line_stream( line );
    string first_field;
		core::Size curr_entry( 0 );
    line_stream >> first_field;

		if( first_field == "#" ) { entries_found = true; continue; }
		if( !entries_found ) continue;

		curr_entry = atoi( first_field.c_str() );
		if( curr_entry != entry ) continue;

		core::Size const transformation_begin( line.find_last_of( "||" ) + 2 );
		std::istringstream transData( line.substr( transformation_begin, 10000) );
		core::Real x,y,z;
		transData >> t.alpha >> t.beta >> t.gamma >> x >> y >> z;
		if( transData.fail() ) {
			TR<<"Error parsing transformation data in line\n"<<line<<std::endl;
			runtime_assert( !transData.fail() );
		}
		t.translation.assign( x, y, z );
		TR<<"Patchdock transformation: "<<t.alpha<<" "<<t.beta<<" "<<t.gamma<<" "<<x<<" "<<y<<" "<<z<<'\n';
		return( t );
	}
	return( t ); // control shouldn't reach this point, but compiler doesn't like function with no return
}

//@detailed transform a chain within the pose according to t. The transformation computed here is
//based on patchdock's transOutput.pl and pdb_trans
void
transform_pose( core::pose::Pose & pose, core::Size const chain, Transformation const & t )
{
	core::Size const chain_begin( pose.conformation().chain_begin( chain ) );
	core::Size const chain_end( pose.conformation().chain_end( chain ) );

	numeric::xyzMatrix< core::Real > rotation;
	{ //compute rotation matrix (taken from RBSMover), but here expecting radian rather than degrees
	  core::Real const sa ( std::sin( t.alpha ));
  	core::Real const ca ( std::cos( t.alpha ));
  	core::Real const sb ( std::sin( t.beta  ));
  	core::Real const cb ( std::cos( t.beta  ));
  	core::Real const sg ( std::sin( t.gamma ));
  	core::Real const cg ( std::cos( t.gamma ));
// Adapted from code sent by Dina Schneidman of the Wolfson lab (Tel-Aviv U)
		rotation.xx( cg * cb ); rotation.xy( -sb*sa*cg - sg * ca ); rotation.xz(  -sb*ca*cg + sg * sa );
		rotation.yx( sg * cb ); rotation.yy(  -sb*sa*sg + ca*cg ); rotation.yz( -sb*ca*sg - sa*cg );
		rotation.zx( sb );            rotation.zy( cb*sa );            rotation.zz(  cb*ca );
	}//compute rotation

//rotate each atom around the geometric centre of the chain
	for( core::Size residue=chain_begin; residue<=chain_end; ++residue ) {
		core::Size const atom_begin( 1 );
		core::Size const atom_end( pose.residue( residue ).natoms() );

		numeric::xyzVector< core::Real > localX, localRX;
		for( core::Size atom=atom_begin; atom<=atom_end; ++atom ) {
			id::AtomID const id( atom, residue );

			localX = pose.xyz( id );
			localRX = rotation * localX;
			pose.set_xyz( id, localRX );
		}
	}

//translate
	for( core::Size residue=chain_begin; residue<=chain_end; ++residue ) {
		core::Size const atom_begin( 1 );
		core::Size const atom_end( pose.residue( residue ).natoms() );

		for( core::Size atom=atom_begin; atom<=atom_end; ++atom ) {
			id::AtomID const id( atom, residue );

			numeric::xyzVector< core::Real > const new_pos( pose.xyz( id ) + t.translation );
			pose.set_xyz( id, new_pos );
		}
	}
	// detect disulfides
	using core::options::option;
	using namespace core::options::OptionKeys;
	if ( option[ in::detect_disulf ].user() ?
			option[ in::detect_disulf ]() : // detect_disulf true
			pose.is_fullatom() // detect_disulf default but fa pose
		)
	{
		pose.conformation().detect_disulfides();
	}
}

void
read_poses( core::pose::Pose & input_pose, std::string & input_tag )
{
	core::pose::Pose dummy_pose;
	std::string dummy_tag( input_tag );

	read_poses( input_pose, dummy_pose, input_tag, dummy_tag );
}

//@detailed transform an input pdb according to patchdock parameters
//if patchdock option is not set, return as is
void
read_poses( core::pose::Pose & input_pose, core::pose::Pose & native_pose, std::string & input_tag, std::string & native_tag )
{
	using namespace core::options;
	using namespace core::options::OptionKeys;

	core::Size patchdock_entrynum( 0 );

	bool const patchdock( option[ parser::patchdock ].user() );
	std::string patchdock_fname( "" );
	if( patchdock ) {
		patchdock_fname = option[ parser::patchdock ]();
		if( patchdock_fname == "" ) { // use default patchdock fname ( 1jjj_2xxx.pdb.gz -> 1jjj_2xxx.patchdock )
			core::Size const filename_end( input_tag.find_first_of( "." ) );

			patchdock_fname = input_tag.substr( 0, filename_end ) + ".patchdock";
		}
		bool const patchdock_random_entry( option[ parser::patchdock_random_entry ].user() );
		core::Size const number_of_entries( number_of_patchdock_entries( patchdock_fname ) );
		if( number_of_entries == 0 )
			utility_exit_with_message_status( "No patchdock entries found. Aborting", 0 );
		if( patchdock_random_entry ) {
			utility::vector1< core::Size > entry_num_extrema = option[ parser::patchdock_random_entry ]();

			runtime_assert( entry_num_extrema.size() == 2 );
			runtime_assert( entry_num_extrema[ 1 ] <= entry_num_extrema[ 2 ] );
			runtime_assert( entry_num_extrema[ 1 ] > 0 );

			TR<<number_of_entries<<" entries in patchdock file "<<patchdock_fname<<std::endl;
			entry_num_extrema[ 2 ] = std::min( entry_num_extrema[ 2 ], number_of_entries );
			TR<<"sampling a number between "<<entry_num_extrema[ 1 ]<<" and "<<entry_num_extrema[ 2 ]<<std::endl;

			patchdock_entrynum = ( core::Size ) floor( RG.uniform() * ( entry_num_extrema[ 2 ] - entry_num_extrema[ 1 ] ) ) + entry_num_extrema[ 1 ];
			runtime_assert( patchdock_entrynum <= entry_num_extrema[ 2 ] );
			runtime_assert( patchdock_entrynum >= entry_num_extrema[ 1 ] );
			std::stringstream ss;
			ss << "." << patchdock_entrynum;
			option[ out::user_tag ].value( ss.str() ); // to set the output tag
		}
		else{
			core::Size const entrynum_begin( input_tag.find_first_of( "." ) );
			core::Size const entrynum_end( input_tag.find_last_of( "." ) );
	  	std::stringstream ss( input_tag.substr( entrynum_begin+1, entrynum_end - entrynum_begin ) );
			ss >> patchdock_entrynum;
			if( patchdock_entrynum > number_of_entries ){
				TR<<"number of patchdock entries exceeded. You've asked for entry "<< patchdock_entrynum<<" but only "<<number_of_entries<<" entries were found"<<std::endl;
				utility_exit_with_message_status("aborting.", 0 );
			}
			if( input_tag == native_tag ){
				input_tag.replace( entrynum_begin, entrynum_end - entrynum_begin + 1, "." );
				native_tag.replace( entrynum_begin, entrynum_end - entrynum_begin + 1, "." );
			}
		}
	}
	if ( option[ in::file::centroid_input ].user() ) {
		core::io::pdb::centroid_pose_from_pdb( input_pose, input_tag );
		core::io::pdb::centroid_pose_from_pdb( native_pose,  native_tag );
	} else {
		core::chemical::ResidueTypeSetCAP rsd_set;
		rsd_set = core::chemical::ChemicalManager::get_instance()->residue_type_set( "fa_standard" );

		core::io::pdb::pose_from_pdb( input_pose, *rsd_set, input_tag );
		core::io::pdb::pose_from_pdb( native_pose, *rsd_set, native_tag );
	}

	if( option[ in::file::fold_tree ].user() ){
		std::string const fold_tree_fname( option[ in::file::fold_tree ]() );
		utility::io::izstream data( fold_tree_fname );
		if ( !data ) {
  		TR << "Cannot open  file: " << fold_tree_fname << std::endl;
  		runtime_assert( data );
		}
		std::string line;
		bool ft_found( false );
		while( getline( data, line ) ){
			if( line.substr(0,10) == "FOLD_TREE " ){
				std::istringstream line_stream( line );
				kinematics::FoldTree f;
				line_stream >> f;
				input_pose.fold_tree( f );
				native_pose.fold_tree( f );
				ft_found = true;
				TR<<"Using user-defined fold-tree:\n"<<input_pose.fold_tree()<<std::endl;
				break;
			}//IF FOLD_TREE
		}//getline
		runtime_assert( ft_found );
	}//option foldtree


	if( !patchdock ) return; // no need for transformations
	runtime_assert( patchdock_entrynum );
	TR<<"Reading patchdock entry "<<patchdock_entrynum<<" from file: "<<patchdock_fname<<std::endl;
	Transformation t( read_patchdock_entry( patchdock_fname, patchdock_entrynum ) );

	transform_pose( input_pose, 2/*chain*/, t );
	transform_pose( native_pose, 2, t );
	TR.flush();
}

}//ProteinInterfaceDesign
}//devel

