// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file relax_protocols
/// @brief protocols that are specific to relax
/// @detailed
/// @author Mike Tyka, Monica Berrondo

#include <protocols/jobdist/JobDistributors.hh> // keep first
#include <protocols/jobdist/Jobs.hh>

#include <protocols/relax_protocols.hh>
#include <protocols/ScoreMap.hh>

#include <core/chemical/util.hh>
#include <core/scoring/rms_util.hh>
#include <protocols/evaluation/RmsdEvaluator.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/scoring/ScoringManager.fwd.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/scoring/constraints/util.hh>

#include <core/io/pdb/pose_io.hh>

#include <core/pose/Pose.hh>
#include <core/pose/util.hh>

#include <protocols/moves/ShakeStructureMover.hh>
#include <protocols/moves/RampingMover.hh>
#include <protocols/moves/BackboneMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/MoverContainer.hh>
#include <protocols/moves/RotamerTrialsMover.hh>
#include <protocols/moves/JumpOutMover.hh>
#include <protocols/moves/RepeatMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/jumping/Dssp.hh>
#include <protocols/moves/TrialMover.hh>


#include <protocols/abinitio/GunnCost.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/options/option.hh>
#include <core/options/keys/relax.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/after_opts.hh>
#include <core/fragment/ConstantLengthFragSet.hh>
#include <protocols/moves/WobbleMover.hh>
#include <protocols/checkpoint/Checkpoint.hh>
#include <utility/file/file_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <protocols/jobdist/standard_mains.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>

#ifdef BOINC_GRAPHICS
#include <protocols/boinc/boinc.hh>
#endif

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.relax.ClassicRelax");

using namespace core;
using io::pdb::dump_pdb;
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace protocols {
namespace relax {

SequenceRelax::SequenceRelax(
	core::scoring::ScoreFunctionOP scorefxn_in,
	const std::string &          sequence_file
) :
	parent("SequenceRelax"),
	scorefxn_(scorefxn_in),
	checkpoints_("SequenceRelax")
{
	set_to_default();
	read_sequence_file( sequence_file );
}

SequenceRelax::SequenceRelax(	SequenceRelax const & other ) :
	parent( other ),
	scorefxn_( other.scorefxn_ ),
	checkpoints_( other.checkpoints_ ),
	sequence_( other.sequence_ )
{}

SequenceRelax::~SequenceRelax() {}


protocols::moves::MoverOP
SequenceRelax::clone() const {
	return new SequenceRelax(*this);
}



////////////////////////////////////////////////////////////////////////////////////////////////////
void SequenceRelax::set_to_default( )
{
}

////////////////////////////////////////////////////////////////////////////////////////////////////
void SequenceRelax::apply( core::pose::Pose & pose ){
	using namespace core::options;
	using namespace core::options::OptionKeys;
	std::cerr << "================== SequenceRelax: " << sequence_.size() << " ===============================" << std::endl;

	// not used for some reason.
	//int max_accepts = option[ OptionKeys::relax::sequence_max_accept ]();

	/// SETUP
	core::Real start_rep_weight = scorefxn_->get_weight( scoring::fa_rep );


	core::pack::task::PackerTaskOP task_;
	task_ = pack::task::TaskFactory::create_packer_task( pose ); //!!!!!!!!!!!
	bool const repack = core::options::option[ core::options::OptionKeys::relax::chi_move]();
	utility::vector1<bool> allow_repack( pose.total_residue(), repack);
	task_->initialize_from_command_line().restrict_to_repacking().restrict_to_residues(allow_repack);
	task_->or_include_current( true );
	moves::PackRotamersMoverOP pack_full_repack_ = new moves::PackRotamersMover( scorefxn_, task_ );
	(*scorefxn_)( pose );

	moves::MinMoverOP min_mover;
	core::kinematics::MoveMapOP movemap = get_movemap()->clone();
	//	movemap->set_bb ( true );
	//	movemap->set_chi( true );
	initialize_movemap( pose, *movemap );

	pose::Pose start_pose=pose;
	pose::Pose best_pose=pose;
	core::Real best_score=100000000.0;
	core::Size accept_count = 0;


	int repeat_step=0;
	int repeat_count=-1;

	core::pose::PoseCOP native_pose = get_native_pose();
	/// RUN


	std::vector< core::Real > best_score_log;
	std::vector< core::Real > curr_score_log;




	for ( core::Size ncmd = 0; ncmd < sequence_.size(); ncmd ++ ){

		int command_starttime = time(NULL);

		SequenceCommand cmd = sequence_[ncmd];

		if( cmd.command == "repeat" ){
			repeat_count = (int) cmd.param1;
			repeat_step = ncmd;
		} else

		if( cmd.command == "repeat_A" ){
			repeat_count = option[ OptionKeys::relax::sequence_repeat_A ]();
			repeat_step = ncmd;
		} else

		if( cmd.command == "endrepeat" ){
			std::cerr << "CMD:  Repeat: " << repeat_count << std::endl;
			repeat_count -- ;
			if( repeat_count <= 0 ){}
			else{
				ncmd = repeat_step;
			}
		} else
		if( cmd.command == "dump" ){
			if( cmd.nparams < 1 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }
			pose.dump_pdb( "dump_" + right_string_of( (int) cmd.param1, 4, '0' ) );
		}	else

		if( cmd.command == "shake" ){
			if( cmd.nparams < 2 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }

			protocols::moves::ShakeStructureMover ssm; //don't initialize with a scorefunction,
			ssm.set_sc_min(true); //sc min after perturbing backbone
			ssm.set_nrounds( (int)cmd.param1 );
			ssm.set_mc_temperature( cmd.param2 );
			ssm.apply(pose);

		}	else

		if( cmd.command == "repweight" ){
			if( cmd.nparams < 1 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }
			scorefxn_->set_weight( scoring::fa_rep, start_rep_weight * cmd.param1 );
		}	else

		if( cmd.command == "repack" ){
			//if( cmd.nparams < 0 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }
			pack_full_repack_->apply( pose );
		}	else

		if( cmd.command == "min" ){
			if( cmd.nparams < 1 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }
			min_mover = new moves::MinMover( movemap, scorefxn_, "dfpmin_armijo_nonmonotone", cmd.param1, true );
			min_mover->apply( pose );
		}	else


		if( cmd.command == "ramp_repack_min" ){
			if( cmd.nparams < 2 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }
			scorefxn_->set_weight( scoring::fa_rep, start_rep_weight * cmd.param1 );
			pack_full_repack_->apply( pose );
			min_mover = new moves::MinMover( movemap, scorefxn_, "dfpmin_armijo_nonmonotone", cmd.param2, true );
			min_mover->apply( pose );
		}	else


		if( cmd.command == "accept_to_best" ){
			// grab the score and remember the pose if the score is better then ever before.
			core::Real score = (*scorefxn_)( pose );
			if( ( score < best_score) || (accept_count == 0) ){
				best_score = score;
				best_pose = pose;
			}
	 		core::Real rms =  protocols::evaluation::native_CA_rmsd( *native_pose , best_pose );
	 		core::Real irms =  protocols::evaluation::native_CA_rmsd( start_pose , best_pose );
			TR << "MRP: " << accept_count << "  " << score << "  " << best_score << "  "
									<< rms << "  "
									<< irms << "  "
									<< std::endl;

			best_score_log.push_back( best_score );
			curr_score_log.push_back( score );

			accept_count++;
			if ( static_cast< int > ( accept_count ) >
				option[ OptionKeys::relax::sequence_max_accept ]()
			) {
				break;
			}
		}	else

		if( cmd.command == "load_best" ){
			pose = best_pose;
		}	else
		if( cmd.command == "load_start" ){
			pose = start_pose;
		}	else

		if( cmd.command == "exit" ){
			utility_exit_with_message( "EXIT INVOKED" );
		}	else

		{
			utility_exit_with_message( "Unknown command: " + cmd.command );
		}


	 	core::Real rms =  protocols::evaluation::native_CA_rmsd( *native_pose , pose );
	 	core::Real irms =  protocols::evaluation::native_CA_rmsd( start_pose , pose );
		int command_endtime = time(NULL);
		TR << "CMD: " <<  cmd.command << "  "
			<< command_endtime - command_starttime << "  "
			<< (*scorefxn_)( pose ) << "  "
			<< rms << "  "
			<< irms << "  "
			<< scorefxn_->get_weight( scoring::fa_rep	)
			<< std::endl;


	}

	pose = best_pose;
	(*scorefxn_)( pose );

	for( Size j = 0; j < best_score_log.size(); j++ )
		core::pose::setPoseExtraScores( pose, "B" + right_string_of(j,3,'0'), best_score_log[j]);

	for( Size j = 0; j < curr_score_log.size(); j ++ )
		core::pose::setPoseExtraScores( pose, "S" + right_string_of(j,3,'0'), curr_score_log[j] );

	checkpoints_.clear_checkpoints();
}

void SequenceRelax::read_sequence_file( const std::string &sequence_file ){
	using namespace ObjexxFCL;
	sequence_.clear();
	std::ifstream infile( sequence_file.c_str() );

	std::cerr << "================== Reading sequence file: ==================" << std::endl;
	if (!infile.good()) {
		utility_exit_with_message( "[ERROR] Error opening sequence file '" + sequence_file + "'" );
	}
	std::string line;
	int linecount=0;

	while( getline(infile,line) ) {
		std::cerr << line << std::endl;
		linecount++;
		std::vector< std::string > tokens ( utility::split( line ) );

		if ( tokens.size() > 0 ) {
			SequenceCommand newcmd;
			newcmd.command = tokens[0];

			if (tokens.size() > 1) {newcmd.param1 = atof(tokens[1].c_str()); newcmd.nparams = 1;}
			if (tokens.size() > 2) {newcmd.param2 = atof(tokens[2].c_str()); newcmd.nparams = 2;}
			if (tokens.size() > 3) {newcmd.param3 = atof(tokens[3].c_str()); newcmd.nparams = 3;}
			if (tokens.size() > 4) {newcmd.param4 = atof(tokens[4].c_str()); newcmd.nparams = 4;}

			sequence_.push_back( newcmd );
		}
	}


}


} // namespace protocols


