// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file KinematicTaskCenter
/// @brief  this class will be handled to a SampleProtocol as a control instance
/// @detailed responsibilities:
///           know which chainbreaks to penalize and close
///           know which jumps to use during sampling, which (if any) to keep after loop-closing
///           supply a JumpMover if jumps should be moved
///           supply a MoveMap
///           supply a "StrictMoveMap": the protocol should not move anything that is dissallowed in strict_movemap(),
///                      it should try to move just stuff in movemap()
/// should this class also know how to ramp score terms ?
/// handle the titration of constraints ?
/// @author Oliver Lange

#ifdef BOINC_GRAPHICS
	#include <protocols/boinc/boinc.hh>
#endif
// Unit Headers
#include <protocols/abinitio/AbrelaxMover.hh>

// Package Headers
#include <protocols/abinitio/ResolutionSwitcher.hh>

// Project Headers
#include <core/pose/Pose.hh>

#include <core/kinematics/util.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/id/NamedAtomID.hh>

#include <core/conformation/ResidueFactory.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/options/option.hh>

#include <core/conformation/util.hh> //idealize

#include <core/util/datacache/BasicDataCache.hh>

#include <protocols/loops/util.hh>
#include <protocols/loops/Exceptions.hh>
#include <protocols/jumping/util.hh>

#include <protocols/relax_protocols.hh> //for the setPoseExtraScores
//#include <core/scoring/constraints/ConstraintSet.hh> //for pose.constraint_set( NULL ) in final_clean_relax
// ObjexxFCL Headers
//#include <ObjexxFCL/formatted.o.hh>
//#include <ObjexxFCL/string.functions.hh>

// headers for default setup
#include <protocols/loops/WidthFirstSlidingWindowLoopClosure.hh>
#include <protocols/idealize/idealize.hh>
#include <protocols/abinitio/FragmentSampler.hh>
#include <protocols/abinitio/ConstraintFragmentSampler.hh>
#include <protocols/topology_broker/TopologyBroker.hh>
#include <protocols/topology_broker/util.hh>

#include <core/options/option.hh>
#include <core/options/keys/OptionKeys.hh>
#include <core/options/keys/abinitio.OptionKeys.gen.hh>
#include <core/options/keys/score.OptionKeys.gen.hh>
#include <core/options/keys/loops.OptionKeys.gen.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/broker.OptionKeys.gen.hh>

// Utility headers
#include <numeric/random/random.hh> //for checkpointing
//#include <utility/io/izstream.hh>
//#include <utility/io/ozstream.hh>
//#include <utility/io/util.hh>
#include <core/util/Tracer.hh>


//#include <core/options/keys/constraints.OptionKeys.gen.hh>
//#include <core/options/keys/evaluation.OptionKeys.gen.hh>
//#include <core/options/keys/filters.OptionKeys.gen.hh>
//#include <core/options/keys/frags.OptionKeys.gen.hh>



#include <core/options/option_macros.hh>

#include <protocols/viewer/viewers.hh>

//// C++ headers
#include <fstream>

static core::util::Tracer tr("protocols.general_abinitio",core::util::t_info);
static numeric::random::RandomGenerator RG(1981221134);

namespace protocols {
namespace abinitio {

using namespace core;
void AbrelaxMover::clear() {
	topology_broker_ = NULL;
	sampling_protocol_ = NULL;
	loop_closure_protocol_ = NULL;
	relax_protocol_ = NULL;
	post_loop_closure_protocol_ = NULL;
}

void AbrelaxMover::set_defaults() {
	using namespace core::options;
	b_return_unrelaxed_fullatom_ = false;

	topology_broker_ = new topology_broker::TopologyBroker();
	topology_broker::add_claims_from_file( *topology_broker_, option[ OptionKeys::broker::setup ]() );
	// the new Abrelax Mover
	AbrelaxMover* abrelax = this; //don't use an OP here ... damn that gives a weird BUG

	//add basic abinitio sampler
	FragmentSamplerOP sampler = new ConstraintFragmentSampler( topology_broker_ );
	abrelax->sampling_protocol( sampler );

	bool bIdeal( true );

	//loop closing
	if ( option[ OptionKeys::abinitio::close_loops ]() ) {
		protocols::loops::SlidingWindowLoopClosureOP closure_protocol =
			new protocols::loops::SlidingWindowLoopClosure;

		if ( option[ OptionKeys::loops::alternative_closure_protocol ]() ) {
			closure_protocol =
				new protocols::loops::WidthFirstSlidingWindowLoopClosure;
		}

		bIdeal = !option[ OptionKeys::loops::non_ideal_loop_closing ]();

		// set options here if you like
		// closure_protocol-> ... and write the setters/accessors, too, if you have to
		closure_protocol->scored_frag_cycle_ratio( option[ OptionKeys::loops::scored_frag_cycles ]() );
		closure_protocol->short_frag_cycle_ratio( option[ OptionKeys::loops::short_frag_cycles ]() );
		closure_protocol->set_bIdealLoopClosing( bIdeal );
		closure_protocol->set_chainbreak_max( option[ OptionKeys::loops::chainbreak_max_accept ]() );

		//add closure protocol to abrelax
		abrelax->closure_protocol( closure_protocol );

		//////////////////////////////////////////////////////////////////////////////////////
		////  Maybe idealize the structure before relax ?
		if ( option[ OptionKeys::loops::idealize_after_loop_close ].user() ){
			protocols::idealize::IdealizeMoverOP idealizer(
				new protocols::idealize::IdealizeMover
			);
			idealizer->fast( false );
			abrelax->post_loop_closure_protocol( idealizer );
			bIdeal = true;
		}
	}

	//yipee a bad hack using global variables
	if ( !bIdeal ) option[ options::OptionKeys::out::file::silent_struct_type ].def( "binary");

	//add relax protocol to abrelax
	//cst_fa_weight and cst_weight are properly interpreted by this utility function
	abrelax->relax_protocol( relax::generate_relax_from_cmd( true /*null if no relax flag*/ ) );
}

//@brief basic apply for the generalized protocol:
void AbrelaxMover::apply( pose::Pose &pose ) {
	using namespace core::options; //yes there are some direct option dependencies here

	runtime_assert( sampling_protocol() );
	runtime_assert( topology_broker() );

	tr.Info << "AbrelaxMover: " << get_current_tag() << std::endl;

	// kidnap sampling_protocols's checkpointer - this could ultimately be a singleton i guess
	checkpoint::CheckPointer &checkpoints = sampling_protocol()->get_checkpoints();

	// need to save RG states such that choices for constraints and fold-tree are the same.
	if( ! checkpoints.recover_checkpoint( pose, get_current_tag(), "rg_state") ){
		checkpoints.checkpoint( pose, get_current_tag(), "rg_state");
	}
	checkpoints.debug( get_current_tag(), "rg_state", RG.uniform() );
	if ( sampling_protocol() ) sampling_protocol()->set_current_tag( get_current_tag() );
	if ( closure_protocol() )  closure_protocol()->set_current_tag( get_current_tag() );
	if ( relax_protocol() )    relax_protocol()->set_current_tag( get_current_tag() );

	// --------------------------------------------------------------------------------
	//
	//   setup -- e.g., sequence --> pose
	//
	topology_broker()->apply( pose ); //creates pose and a state in the topology_broker needed for the whole run
	protocols::viewer::add_conformation_viewer( pose.conformation(), "start_pose" );  //add viewer
#ifdef BOINC_GRAPHICS
	// attach boinc graphics pose observer
	protocols::boinc::Boinc::attach_graphics_current_pose_observer( pose );
#endif

	// --------------------------------------------------------------------------------
	//
	//   abinitio
	//
	sampling_protocol()->topology_broker( topology_broker() );
#ifdef BOINC_GRAPHICS
	// attach boinc graphics pose observer
	protocols::boinc::Boinc::attach_graphics_current_pose_observer( pose );
#endif
	sampling_protocol()->apply( pose );



	// --------------------------------------------------------------------------------
	//
	//   filters
	//

	bool loop_success = true;

#ifdef BOINC_GRAPHICS
	// attach boinc graphics pose observer
	protocols::boinc::Boinc::attach_graphics_current_pose_observer( pose );
#endif

	scoring::ScoreFunction last_scorefxn = sampling_protocol()->current_scorefxn();

	// Make sure score columns always the same.

	relax::ClassicRelax().setPoseExtraScores( pose );
	relax::SimpleMultiRelax::setPoseExtraScores( pose );
	protocols::loops::SlidingWindowLoopClosure::setPoseExtraScores( pose );



	// --------------------------------------------------------------------------------
	//
	//   loop closing
	//
	if ( closure_protocol() ) {

		// Did we already close the loops successfully ?
		if ( checkpoints.recover_checkpoint( pose, get_current_tag(), "loops_S", false /*fullatom*/, true /*foldtree*/))  {
			checkpoints.debug( get_current_tag(), "close_loops", (last_scorefxn)(pose), (core::Real) true );
			loop_success = true;
		} else
		 // No ? Have we already tried but failed ?
		if ( checkpoints.recover_checkpoint( pose, get_current_tag(), "loops_C", false /*fullatom*/, true /*foldtree*/))  {
			checkpoints.debug( get_current_tag(), "close_loops", (last_scorefxn)(pose), (core::Real) false );
			loop_success = false;
		} else {
		// No ? Then evidently we havn't even tried yet

		tr << "AbrelaxMover: start loops" << std::endl;
		kinematics::MoveMapOP movemap = new kinematics::MoveMap;
		closure_protocol()->scorefxn( new scoring::ScoreFunction( sampling_protocol()->current_scorefxn() ) );
		closure_protocol()->fragments( topology_broker()->loop_frags( *movemap ) ); //get frags and movemap from broker
		closure_protocol()->movemap( movemap ); //this is the movemap sanctioned by the broker ... see line above

		try {
			jumping::close_chainbreaks(
						closure_protocol(),
						pose,
						sampling_protocol()->get_checkpoints(),
						get_current_tag(),
						topology_broker()->final_fold_tree()
			);
		} catch ( loops::EXCN_Loop_not_closed& excn ) {
			set_current_tag( "C_"+get_current_tag().substr(std::min(2,(int)get_current_tag().size())) );
			set_last_move_status( moves::FAIL_RETRY );
			loop_success = false;
		}

		topology_broker()->apply_filter( pose, LOOP_CLOSURE, 1 );

		if ( post_loop_closure_protocol_ ) {
			post_loop_closure_protocol_->apply( pose );
		}

		// to know this we'd have to catch the Exception EXCN_Loop_not_closed
		if ( loop_success ) checkpoints.checkpoint( pose, get_current_tag(), "loops_S", true /*foldtree*/ );
		else           checkpoints.checkpoint( pose, get_current_tag(), "loops_C", true /*foldtree*/ );

		checkpoints.debug( get_current_tag(), "loops", (last_scorefxn)(pose), (core::Real) loop_success );

		}
	}

	// --------------------------------------------------------------------------------
	//
	//   Fullatom switch
	//
	if ( relax_protocol() || b_return_unrelaxed_fullatom_ ) {
		tr << "AbrelaxMover: switch to fullatom" << std::endl;
		topology_broker()->switch_to_fullatom( pose );
		// we're now in fullatom mode - so upate the score function.
		if (( (option[ OptionKeys::score::weights ]() == "score0") ||
					(option[ OptionKeys::score::weights ]() == "score2") ||
					(option[ OptionKeys::score::weights ]() == "score3") ||
					(option[ OptionKeys::score::weights ]() == "score5") ) ) {
					utility_exit_with_message(
						"Cannot proceed - you chose a centroid score function for fullatom mode"
					);
		}
		last_scorefxn = *relax_protocol()->get_scorefxn();
	}

	// --------------------------------------------------------------------------------
	//
	//   Relax
	//
	if ( ( loop_success || option[ OptionKeys::abinitio::relax_failures ]() ) &&
				relax_protocol()
	) {
		tr << "AbrelaxMover: relax " << std::endl;
		if ( !checkpoints.recover_checkpoint( pose, get_current_tag(), "relax", true, true) ) {
			relax_protocol()->apply( pose );
			checkpoints.checkpoint( pose, get_current_tag(), "relax", true ); //since relax_protocol throws away its checkpoints right here
		}
		checkpoints.debug( get_current_tag(), "relax", (last_scorefxn)( pose ) );
	}

	topology_broker()->apply_filter( pose, RELAX, 1 );

	// --------------------------------------------------------------------------------
	//
	//   Final clean relax
	//
	if ( ( loop_success || option[ OptionKeys::abinitio::relax_failures ]() ) &&
	       relax_protocol() &&
				 option[ OptionKeys::abinitio::final_clean_relax ]() ) {
		tr << "AbrelaxMover: final relax " << std::endl;
		pose.constraint_set( NULL );
		if ( !checkpoints.recover_checkpoint( pose, get_current_tag(), "finalrelax", true, true) ) {
			relax_protocol()->apply( pose );
			checkpoints.checkpoint( pose, get_current_tag(), "finalrelax", true ); //since relax_protocol throws away its checkpoints right here
		}
		checkpoints.debug( get_current_tag(), "finalrelax", (last_scorefxn)( pose ) );
	}

	if ( option[ OptionKeys::abinitio::clear_pose_cache ]() ) {
		tr.Debug << "\n******************************************************** \n"
						 << "              CLEAR POSE CACHE                             \n"
						 << "*************************************************************" << std::endl;
		pose.data().clear();
	}

	if ( sampling_protocol() ) sampling_protocol()->get_checkpoints().clear_checkpoints();

	if ( !b_return_unrelaxed_fullatom_ ) last_scorefxn( pose );
}

}
}
