// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @brief Application level code for relax-type protocols
/// @detailed
///
/// use AbrelaxApplication in the following way:
///
/// AbrelaxAppliaction::register_options();
/// core::init
/// AbrelaxAppliaction my_app;
/// my_app.add_evaluation( new MySpecialEvaluator );
/// ...
/// my_app.run();
///
/// ---------------------------------------
/// control flow:
/// the run method calls
/// setup()
/// and then either fold() or rerun() (depending on option -rerun )
///
/// each decoy is evaluated by process_decoy() and results are written to the score-file (if specified) or
/// to the silent_output file ( if applicable ).
/// the score file is a silent-file without structural data ( just SCORE lines )
///
/// rerun(): run thru structures in in:file:silent and call process_decoy for each
/// fold(): produce structures with an Abinitio-type protocol and call process_decoy
///
/// options specific to AbrelaxApplication can be found by using -help at the command-line.
/// if you add new options please follow the scheme in the static method register options
///
/// the behaviour of AbrelaxApplication is controlled by comman-line-options. Refer to -help (usage) and the code
///
/// information that is not always present is stored as xxxOP, and the NULL-pointer is interpreted that the respective
/// behaviour is not present. (i.e., native_pose_ is either pointing to the native pose (-native) or to NULL.
/// when you use such pointers ask if they are non-NULL.
///
///
///
/// @author Oliver Lange

// keep these headers first for compilation with Visual Studio C++
#include <protocols/jobdist/JobDistributors.hh>
#include <protocols/jobdist/Jobs.hh>

// Unit Headers


// Package Headers
#include <core/kinematics/util.hh>

#include <protocols/abinitio/FragmentMover.hh>
#include <protocols/abinitio/SmoothFragmentMover.hh>
#include <protocols/abinitio/GunnCost.hh>


//#include <protocols/Protocol.hh>
#include <protocols/relax/util.hh>
#include <protocols/abinitio/AbrelaxMover.hh>
#include <protocols/jumping/SecondaryStructure.hh>
// Project Headers
#include <core/types.hh>

#include <core/pose/Pose.hh>
#include <core/pose/util.hh>
#include <core/init.hh>
#include <core/util/MetricValue.hh>

#include <core/options/option.hh>
#include <core/options/keys/abinitio.OptionKeys.gen.hh>
#include <core/options/keys/constraints.OptionKeys.gen.hh>
#include <core/options/keys/evaluation.OptionKeys.gen.hh>
#include <core/options/keys/filters.OptionKeys.gen.hh>
#include <core/options/keys/frags.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/jumps.OptionKeys.gen.hh>
#include <core/options/keys/loopfcst.OptionKeys.gen.hh>
#include <core/options/keys/loops.OptionKeys.gen.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>
#include <core/options/keys/templates.OptionKeys.gen.hh>
#include <core/options/keys/chunk.OptionKeys.gen.hh>
#include <core/options/keys/broker.OptionKeys.gen.hh>

#include <core/options/after_opts.hh>
#include <core/options/option_macros.hh>


#include <core/fragment/ConstantLengthFragSet.hh>
#include <core/fragment/BBTorsionSRFD.hh>
#include <core/fragment/util.hh>
#include <core/fragment/FragmentIO.hh>

#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/util.hh>

#include <core/conformation/util.hh>
#include <core/io/silent/SilentFileData.hh>

#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/ConstraintIO.hh>
//#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/scoring/constraints/util.hh>
#include <core/scoring/constraints/NamedAtomPairConstraint.hh>
#include <core/scoring/ScoreFunction.fwd.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/constraints/BoundConstraint.hh>

#include <protocols/toolbox/PoseMetricCalculators/ClashCountCalculator.hh>

#include <core/io/pdb/pose_io.hh>
#include <core/io/silent/silent.fwd.hh>
#include <core/io/silent/SilentStructFactory.hh>
#include <core/io/silent/SilentFileData.hh>


#include <core/sequence/util.hh>

#include <protocols/evaluation/PoseEvaluator.hh>
#include <protocols/evaluation/RmsdEvaluator.hh>
#include <protocols/evaluation/JumpEvaluator.hh>
#include <protocols/evaluation/TimeEvaluator.hh>
#include <protocols/evaluation/PCA.hh>
#include <protocols/evaluation/PoseMetricEvaluator.hh>
#include <protocols/evaluation/ConstraintEvaluator.hh>
#include <protocols/evaluation/util.hh>
#include <protocols/evaluation/ChemicalShiftEvaluator.hh>

#include <protocols/loops/SlidingWindowLoopClosure.hh>
#include <protocols/loops/WidthFirstSlidingWindowLoopClosure.hh>
//#include <protocols/loops/ShortLoopClosure.hh>
//#include <protocols/loops/LoopClosure.hh>
#include <protocols/loops/LoopClass.hh>
#include <protocols/loops/LoopMover.hh>
#include <protocols/loops/util.hh>

#include <protocols/idealize/idealize.hh>


//#include <protocols/loops/looprelax_protocols.hh>

#include <protocols/filters/Filter.hh>
#include <protocols/filters/RGFilter.hh>
#include <protocols/filters/COFilter.hh>
#include <protocols/filters/SheetFilter.hh>

#include <protocols/jd2/JobDistributor.hh>
#include <protocols/jd2/JobOutputter.hh>
#include <protocols/jd2/Job.hh>

#include <protocols/moves/MoverStatus.hh>
#include <math.h>

//numeric headers
#include <numeric/random/random.hh>

// ObjexxFCL Headers

// Utility headers
#include <utility/io/izstream.hh>
#include <utility/vector1.hh>
#include <core/util/Tracer.hh>
#include <utility/io/util.hh>
#include <utility/exit.hh>
#include <utility/excn/Exceptions.hh>
// C++ headers
#include <cstdlib>
#include <string>
#include <vector>

#include <protocols/topology_broker/TopologyBroker.hh>
#include <protocols/topology_broker/util.hh>
#include <protocols/topology_broker/FragmentClaimer.hh>
#include <protocols/topology_broker/JumpClaimer.hh>
#include <protocols/topology_broker/weights/all.hh>
#include <protocols/topology_broker/SequenceClaimer.hh>
#include <protocols/topology_broker/RigidChunkClaimer.hh>
#include <protocols/topology_broker/ConstraintClaimer.hh>
#include <protocols/topology_broker/LoopFragmentClaimer.hh>
#include <protocols/topology_broker/CutBiasClaimer.hh>
#include <protocols/jumping/SecondaryStructure.hh>
#include <protocols/abinitio/FragmentSampler.hh>
#include <protocols/abinitio/ConstraintFragmentSampler.hh>
#include <protocols/abinitio/AbrelaxMover.hh>
#include <protocols/loops/LoopRelaxMover.hh>


static core::util::Tracer tr("protocols.abinitio.AbrelaxApplication");
static numeric::random::RandomGenerator RG(42322464);  // <- Magic number, do not change it!




namespace protocols  {
namespace abinitio {

using core::Size;
using namespace core;
using namespace fragment;
using namespace jumping;
using namespace evaluation;
using namespace core::options;
//using namespace OptionKeys;

////////////////////////////////////////////////////////////////////////////////////////////////////
///@details registering of options that are relevant for AbrelaxApplication
void register_options_broker() {
	using namespace core::options;
	using namespace OptionKeys;

	option.add_relevant(  chunk::pdb2 );
	option.add_relevant(  chunk::loop2 );
	option.add_relevant(  broker::setup );

	/// options from the main Option file that are relevant in this context ( and should appear in -help output )
	option.add_relevant(  in::file::native );
	option.add_relevant(  in::file::silent ); // input silent file
	option.add_relevant(  in::file::frag3 );
	option.add_relevant(  in::file::frag9 );
	option.add_relevant(  in::file::fasta );
	option.add_relevant(  in::file::native_exclude_res ); // list for residues to exclude

	option.add_relevant(  out::file::silent );
	option.add_relevant(  out::nstruct );

	option.add_relevant(  run::proc_id );
	option.add_relevant(  run::nproc );
	option.add_relevant(  run::condor );


	option.add_relevant(  OptionKeys::abinitio::fastrelax );
	option.add_relevant(  OptionKeys::abinitio::relax );
	option.add_relevant(  OptionKeys::abinitio::multifastrelax );
	option.add_relevant(  OptionKeys::abinitio::relax_with_jumps );
	option.add_relevant(  OptionKeys::abinitio::use_filters );
	option.add_relevant(  OptionKeys::abinitio::detect_disulfide_before_relax );
	option.add_relevant(  OptionKeys::abinitio::debug );
	option.add_relevant(  OptionKeys::abinitio::number_3mer_frags );
	option.add_relevant(  OptionKeys::abinitio::number_9mer_frags );
	option.add_relevant(  OptionKeys::abinitio::process_store );
	option.add_relevant(  OptionKeys::abinitio::fix_residues_to_native );
	option.add_relevant(  OptionKeys::abinitio::return_full_atom );
	option.add_relevant(  OptionKeys::abinitio::rerun );
	option.add_relevant(  OptionKeys::abinitio::jdist_rerun );
	// starting conditions
	option.add_relevant(  OptionKeys::abinitio::start_native );
	option.add_relevant(  OptionKeys::abinitio::perturb );
	option.add_relevant(  OptionKeys::abinitio::close_loops );

	// evaluation
	option.add_relevant(  OptionKeys::abinitio::rmsd_residues );
	option.add_relevant(  OptionKeys::abinitio::bGDT );
	option.add_relevant(  OptionKeys::run::no_prof_info_in_silentout );
	// use fragments from native structure
	option.add_relevant(  OptionKeys::abinitio::steal_3mers );
	option.add_relevant(  OptionKeys::abinitio::steal_9mers );
	option.add_relevant(  OptionKeys::abinitio::dump_frags );
	option.add_relevant(  OptionKeys::abinitio::no_write_failures );

	option.add_relevant(  loopfcst::use_general_protocol );
	option.add_relevant(  loopfcst::coord_cst_weight );
	option.add_relevant(  loopfcst::coord_cst_all_atom );
	option.add_relevant(  loopfcst::coord_cst_weight_array );
	option.add_relevant(  loopfcst::dump_coord_cst_weight_array );

	option.add_relevant(  OptionKeys::in::file::pca );
	option.add_relevant(  OptionKeys::out::sf );
	// jumping
	option.add_relevant(  jumps::fix_jumps );
	option.add_relevant(  jumps::jump_lib );
	option.add_relevant(  jumps::fix_chainbreak );
	option.add_relevant(  jumps::pairing_file );
	option.add_relevant(  jumps::sheets );
	option.add_relevant(  jumps::random_sheets );
	option.add_relevant(  jumps::evaluate );
	option.add_relevant(  jumps::extra_frags_for_ss );
	option.add_relevant(  jumps::loop_definition_from_file );
	option.add_relevant(  jumps::no_chainbreak_in_relax );
	option.add_relevant(  jumps::residue_pair_jump_file );
	option.add_relevant(  jumps::topology_file );
	//loop closure
	option.add_relevant(  OptionKeys::loops::loop_file );
	option.add_relevant(  OptionKeys::loops::alternative_closure_protocol );
	option.add_relevant(  OptionKeys::loops::short_frag_cycles );
	option.add_relevant(  OptionKeys::loops::scored_frag_cycles );
	option.add_relevant(  OptionKeys::loops::debug_loop_closure );
	option.add_relevant(  OptionKeys::loops::non_ideal_loop_closing );
	option.add_relevant(  OptionKeys::loops::chainbreak_max_accept );


	option.add_relevant(  OptionKeys::loops::extended );
	// constraints
	option.add_relevant(  constraints::cst_file );
	option.add_relevant(  constraints::forest_file );
	option.add_relevant(  constraints::compute_total_dist_cst );
	option.add_relevant(  constraints::no_linearize_bounded );
	option.add_relevant(  constraints::dump_cst_set );
	option.add_relevant(  constraints::no_cst_in_relax );
	option.add_relevant(  constraints::evaluate_max_seq_sep );
	option.add_relevant(  constraints::cull_with_native );
	option.add_relevant(  constraints::named );

	option.add_relevant(  constraints::viol );
	option.add_relevant(  constraints::viol_level );
	option.add_relevant(  constraints::viol_type );

	// homologs
	option.add_relevant(  templates::config );
	option.add_relevant(  templates::pairings );

	//large default number means all frags are used	if this option is not specified
	option.add_relevant(  templates::min_nr_large_frags );
	option.add_relevant(  templates::min_nr_small_frags );

	option.add_relevant(  templates::nr_large_copies );
	option.add_relevant(  templates::nr_small_copies );
	option.add_relevant(  templates::vary_frag_size );
	option.add_relevant(  templates::fix_aligned_residues );
	option.add_relevant(  templates::fix_margin );
	option.add_relevant(  templates::fix_frag_file );
	option.add_relevant(  templates::no_pick_fragments );
	option.add_relevant(  templates::pick_multiple_sizes );
	option.add_relevant(  templates::strand_constraint );
	option.add_relevant(  frags::nr_large_copies );
	option.add_relevant(  frags::annotate );


	option.add_relevant(  in::file::tags );
	// generalized protocol:


	// this adds all relevant options from the protocols
	// ClassicAbinitio, FoldConstraints, JumpingFoldConstraints
	//KinematicAbinitio::register_options();
	//Templates::register_options();
	//loops::WidthFirstSlidingWindowLoopClosure::register_options();
	// here we should have
	// ClassicRelax::register_options();
	// FastRelax::register_options(); etc.
	#ifdef BOINC
	std::cerr << "Registered extra options." << std::endl; std::cerr.flush();
	#endif
}
////////////////////////////////////////////////////////////////////////////////////////////////////

// little helper classes for evaluation of generated decoys:
////////////////////////////////////////////////////////////////////////////////////////////////////
// evaluates the PCA
class PcaEvaluator : public PoseEvaluator {
public:
	PcaEvaluator ( PCA_OP pca ) : pca_( pca ) {}
	virtual void apply( pose::Pose& pose, std::string tag, io::silent::SilentStruct &pss ) const;
private:
	PCA_OP pca_;
};

////////////////////////////////////////////////////////////////////////////////////////////////////
// evaluates the violations of atom-pairconstraints always with the full weight and full sequence separation
using namespace scoring::constraints;
class ShowViolation : public PoseEvaluator {
public:
	ShowViolation( ) : constraints_( NULL ) {}
	virtual void apply( pose::Pose& pose, std::string tag, io::silent::SilentStruct &pss ) const;
private:
	mutable ConstraintSetOP constraints_;
};
////////////////////////////////////////////////////////////////////////////////////////////////////
class ComputeTotalDistCst : public PoseEvaluator {
public:
	ComputeTotalDistCst( ) : constraints_( NULL ) {};
	virtual void apply( pose::Pose& pose, std::string tag, io::silent::SilentStruct &pss ) const;
private:
	mutable ConstraintSetOP constraints_;
};


////////////////////////////////////////////////////////////////////////////////////////////////////
///@detail called by ): read fragment libraries, I strongly suggest to use different options than A and B
/// if option[ steal ] fragments from the native structure are added to the set.
/// native structure needs to be idealized for this!
void setup_fragments( FragSetOP& fragset_large, FragSetOP& fragset_small ) { //needed for setup_broker_from_cmdline
	using namespace core::options;
	using namespace OptionKeys;

	std::string frag_large_file, frag_small_file;
	if (option[ in::file::fragA ].user()) {
		frag_large_file  = option[ in::file::fragA ]();
	} else {
		frag_large_file  = option[ in::file::frag9 ]();
	}

	if (option[ in::file::fragB ].user()) {
		frag_small_file  = option[ in::file::fragB ]();
	} else {
		frag_small_file  = option[ in::file::frag3 ]();
	}

//	fragset_large_ = FragmentIO().read( frag_large_file );
	fragset_large = FragmentIO(
		option[ OptionKeys::abinitio::number_9mer_frags ](),
		option[ OptionKeys::frags::nr_large_copies ](),
		option[ OptionKeys::frags::annotate ]()
	).read( frag_large_file );

	fragset_small = FragmentIO(
		option[ OptionKeys::abinitio::number_3mer_frags ],
		1, //nr_copies
		option[ OptionKeys::frags::annotate ]
	).read( frag_small_file );


	if ( option[ OptionKeys::abinitio::steal_3mers ]() || option[ OptionKeys::abinitio::steal_9mers ]() ) {
		// read native pose to get sequence
		pose::PoseOP native_pose = new pose::Pose;
		if ( option[ in::file::native ].user() ) {
			io::pdb::pose_from_pdb( *native_pose, option[ in::file::native ]() );
			pose::set_ss_from_phipsi( *native_pose );
		} else {
			utility_exit_with_message(" can't steal natie fragments without in:file:native " );
		}
		tr.Info << " stealing fragments from native pose: ATTENTION: native pose has to be IDEALIZED!!! " << std::endl;
		//		utility_exit_with_message(" stealing fragments from pose: currently not supported! ask Oliver " );
		if ( option[ OptionKeys::abinitio::steal_9mers ]() ) steal_frag_set_from_pose( *native_pose, *fragset_large,
			new FragData( new BBTorsionSRFD, fragset_large->max_frag_length() ) );
		if ( option[ OptionKeys::abinitio::steal_3mers ]() ) steal_frag_set_from_pose( *native_pose, *fragset_small,
			new FragData( new BBTorsionSRFD, fragset_small->max_frag_length() ) );
	}

	if ( option[ OptionKeys::abinitio::dump_frags ]() ) { //diagnosis
		utility::io::ozstream dump_frag_small( "fragset_small.dump" );
		for ( FrameIterator it=fragset_small->begin(), eit=fragset_small->end(); it!=eit; ++it ) {
			(*it)->show( dump_frag_small );
		}
		utility::io::ozstream dump_frag_large( "fragset_large.dump" );
		for ( FrameIterator it=fragset_large->begin(), eit=fragset_large->end(); it!=eit; ++it ) {
			(*it)->show( dump_frag_large );
		}
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
///@detail setup_fold() all initialization that is necessary to run abinitio production loop in fold()
/// read fragments, make pose from sequence, get constraints, set jumps, movemap .etc
/// the two parameters are OUTPUT:
///    extended_pose to run A) with ( might actually contain native starting structure (option!) )
///    prot_ptr: an initialized instance of ClassicAbinitio, FoldConstraints or JumpingFoldConstraints depending on
///   user settings.
void setup_broker_from_cmdline( topology_broker::TopologyBrokerOP broker ) {
	/* this is OBSOLET since it is much better done by initializing broker from INIT-file

		 keep it for two reasons:
		     - demo of how to setup or add Claimers out of your application
				 - easy use of abinitio for standard applications without writing input file

	*/
	using namespace core::options;
	using namespace OptionKeys;
	std::string sequence;
	// specify sequence -- from fasta file or native_pose
	if ( option[ in::file::fasta ].user() ) {
		sequence = core::sequence::read_fasta_file( option[ in::file::fasta ]()[1] )[1]->sequence();
		tr.Info << "read fasta sequence: " << sequence.size() << " residues\n"  << sequence << std::endl;
	} else if ( option[ in::file::native ].user() ) {
		pose::PoseOP native_pose = new pose::Pose;
		io::pdb::pose_from_pdb( *native_pose, option[ in::file::native ]() );
		sequence = native_pose->sequence();
	} else {
		utility_exit_with_message(
				"Error: can't read sequence! Use -in::file::fasta sequence.fasta or -in::file::native native.pdb!"
		);
	}

	using namespace protocols::topology_broker;
	FragSetOP fragset_small, fragset_large;
	setup_fragments( fragset_large, fragset_small );
	ClassicFragmentMoverOP bms, bml, sms;
	broker->add( new SequenceClaimer( sequence, core::chemical::CENTROID ) );
	broker->add( new FragmentClaimer( bml = new ClassicFragmentMover( fragset_large ), "LargeFrags", new weights::LargeFragWeight ) );
	broker->add( new FragmentClaimer( bms = new ClassicFragmentMover( fragset_small ), "SmallFrags", new weights::SmallFragWeight ) );
	jumping::SecondaryStructure ss_def( *fragset_small, 0, false /* use only central residue */ );
	broker->add( new CutBiasClaimer( ss_def ) );
	broker->add( new LoopFragmentClaimer( fragset_small ) );
	broker->add( new FragmentClaimer( sms =
																	 new SmoothFragmentMover( fragset_small, /*dummy -*/ new GunnCost ),
																	 "SmoothFrags",
																	 new weights::SmoothFragWeight )
	);

	bms->set_end_bias( option[ OptionKeys::abinitio::end_bias ] ); //default is 30.0
	bml->set_end_bias( option[ OptionKeys::abinitio::end_bias ] );
	sms->set_end_bias( option[ OptionKeys::abinitio::end_bias ] );

	//	if ( jump_def_ ) broker->add( new JumpClaimer( jump_def_, "beta-pair", new weights::LargeFragWeight( 0.2 ) ) );

	if ( option[ constraints::cst_file ].user() ) {
		broker->add( new ConstraintClaimer( core::scoring::constraints::get_cst_file_option(), "cst_file" ) );
	}


	if ( option[ OptionKeys::loops::input_pdb ].user() ) {
		core::pose::Pose chunk_pose;
		core::io::pdb::pose_from_pdb( chunk_pose,
			*core::chemical::ChemicalManager::get_instance()->residue_type_set( core::chemical::FA_STANDARD ),
			option[ OptionKeys::loops::input_pdb ]().name()
		);
		if ( !chunk_pose.is_fullatom() ) utility_exit_with_message(" this full-atom pose should be a full-atom pose, no? ");
			std::string filename( protocols::loops::get_loop_file_name() );
			protocols::loops::Loops loops_in;
			loops_in.read_loop_file( filename, false /*no strict looprlx file*/ );  // <== TODO: select these using density score
			broker->add( new RigidChunkClaimer( chunk_pose, loops_in ) );
	}

	if ( option[ OptionKeys::chunk::pdb2 ].user() ) {
			core::pose::Pose chunk_pose;
			core::io::pdb::pose_from_pdb( chunk_pose,
				*core::chemical::ChemicalManager::get_instance()->residue_type_set( core::chemical::FA_STANDARD ),
				option[ OptionKeys::chunk::pdb2 ]().name()
			);
			if ( !chunk_pose.is_fullatom() ) utility_exit_with_message(" this full-atom pose should be a full-atom pose, no? ");
			std::string filename( option[ OptionKeys::chunk::loop2 ]().name() );
			protocols::loops::Loops loops_in;
			loops_in.read_loop_file( filename, false /*no strict looprlx file*/ );  // <== TODO: select these using density score
			broker->add( new RigidChunkClaimer( chunk_pose, loops_in ) );
	}
}

/// setup some additional PoseEvaluators that we like to use.
void setup_private_evaluators() {
	using namespace OptionKeys;
	using namespace protocols::jd2;
	JobOutputterOP jo = JobDistributor::get_instance()->job_outputter();
	// run with homolog info? -- needed for setup_fragments, and rerun keep it upfront
	//setup_templates();

// //this doesn't work together with MPIBatchMode since this code is not called when new job-outputters are created
//    core::pose::metrics::PoseMetricCalculatorOP
// 		clash_calculator = new protocols::toolbox::PoseMetricCalculators::ClashCountCalculator( 2.0 );
// 	core::pose::metrics::CalculatorFactory::Instance().register_calculator( "clashes", clash_calculator );
// 	jo->add_evaluation( new PoseMetricEvaluator<core::Size>( "clashes", "total" ) );
// 	jo->add_evaluation( new PoseMetricEvaluator<core::Size>( "clashes", "bb" ) );

//	if ( option[ constraints::viol ]() ) jo->add_evaluation( new ShowViolation );
//	if ( option[ constraints::compute_total_dist_cst ] ) jo->add_evaluation( new ComputeTotalDistCst );

// 	// read PCA info
// 	if ( option[ OptionKeys::in::file::pca ].user() ) {
// 		pca_ = new PCA;
// 		pca_->read_eigvec_file( option[ OptionKeys::in::file::pca ](), *native_pose_, 2 );
// 		// if ( tr.Trace.visible() ) pca_->show( std::cout );
// 		add_evaluation( new PcaEvaluator( pca_ ) );
// 	}
}

/// look at job --- find out what your loop-file and rigid-chunk-pdb is and start sampling
class TemplateMover : public moves::Mover {
public:
private:
};

///@brief setup of the AbrelaxMover
///   uses AbinitioProtocol aka FragmentSampler
///   and adds according to cmd-line
///      -- loop-closing if requested
///      -- relax protocol if requested
//AbrelaxMoverOP setup_sampler( topology_broker::TopologyBrokerOP broker ) {

// 	// new Abrelax Mover
// 	AbrelaxMoverOP abrelax = new AbrelaxMover( broker );

// 	//add basic abinitio sampler
// 	FragmentSamplerOP sampler = new ConstraintFragmentSampler( broker );
// 	abrelax->sampling_protocol( sampler );

// 	bool bIdeal( true );

// 	//loop closing
// 	if ( option[ OptionKeys::abinitio::close_loops ]() ) {
// 		protocols::loops::SlidingWindowLoopClosureOP closure_protocol =
// 			new protocols::loops::SlidingWindowLoopClosure;

// 		if ( option[ OptionKeys::loops::alternative_closure_protocol ]() ) {
// 			closure_protocol =
// 				new protocols::loops::WidthFirstSlidingWindowLoopClosure;
// 		}

// 		bIdeal = !option[ OptionKeys::loops::non_ideal_loop_closing ]();

// 		// set options here if you like
// 		// closure_protocol-> ... and write the setters/accessors, too, if you have to
// 		closure_protocol->scored_frag_cycle_ratio( option[ OptionKeys::loops::scored_frag_cycles ]() );
// 		closure_protocol->short_frag_cycle_ratio( option[ OptionKeys::loops::short_frag_cycles ]() );
// 		closure_protocol->set_bIdealLoopClosing( bIdeal );
// 		closure_protocol->set_chainbreak_max( option[ OptionKeys::loops::chainbreak_max_accept ]() );

// 		//add closure protocol to abrelax
// 		abrelax->closure_protocol( closure_protocol );

// 		//////////////////////////////////////////////////////////////////////////////////////
// 		////  Maybe idealize the structure before relax ?
// 		if ( option[ OptionKeys::loops::idealize_after_loop_close ].user() ){
// 			protocols::idealize::IdealizeMoverOP idealizer(
// 				new protocols::idealize::IdealizeMover
// 			);
// 			idealizer->fast( false );
// 			abrelax->post_loop_closure_protocol( idealizer );
// 			bIdeal = true;
// 		}
// 	}

// 	if ( !bIdeal ) option[ options::OptionKeys::out::file::silent_struct_type ].def( "binary");

// 	//add relax protocol to abrelax
// 	abrelax->relax_protocol( relax::generate_relax_from_cmd( true /*null if no relax flag*/ ) );


// 	return abrelax;
//}


void Broker_main(){
	using namespace OptionKeys;

	if ( option[ constraints::no_linearize_bounded ] ) {
		tr.Info << "use fully harmonic potential for BOUNDED " << std::endl;
		ConstraintIO::get_func_factory().add_type("BOUNDED", new BoundFunc(0,0,0,1000,"dummy") );
	}
	if ( option[ constraints::named ] ) {
		tr.Info << "use named constraints in AtomPairConstraint to avoid problems with cutpoint-variants " << std::endl;
		ConstraintIO::get_cst_factory().add_type( new scoring::constraints::NamedAtomPairConstraint( id::NamedAtomID(), id::NamedAtomID(), NULL) );
	}

	//	setup_private_evaluators();

	// 	using namespace protocols::topology_broker;
	// 	TopologyBrokerOP broker( new TopologyBroker() );
	// 	if ( option[ broker::setup ].user() ) { //read from file
	// 		add_claims_from_file( *broker, option[ broker::setup ]() );
	// 	} else { //direct via C++
	// 		setup_broker_from_cmdline( broker );
	// 	}

	// 	AbrelaxMoverOP abrelax = setup_sampler( broker );

	AbrelaxMoverOP abrelax = new AbrelaxMover; //its doing its setup in Abrelax::set_defaults now.

	protocols::jd2::JobDistributor::get_instance()->go( abrelax );
}





void Loopbuild_Threading_main(){
	using namespace OptionKeys;

	if ( option[ constraints::no_linearize_bounded ] ) {
		tr.Info << "use fully harmonic potential for BOUNDED " << std::endl;
		ConstraintIO::get_func_factory().add_type("BOUNDED", new BoundFunc(0,0,0,1000,"dummy") );
	}
	if ( option[ constraints::named ] ) {
		tr.Info << "use named constraints in AtomPairConstraint to avoid problems with cutpoint-variants " << std::endl;
		ConstraintIO::get_cst_factory().add_type( new scoring::constraints::NamedAtomPairConstraint( id::NamedAtomID(), id::NamedAtomID(), NULL) );
	}
	setup_private_evaluators();

	protocols::loops::LoopRelaxThreadingMoverOP lrt = new protocols::loops::LoopRelaxThreadingMover();
	lrt->setup();

	protocols::jd2::JobDistributor::get_instance()->go( lrt );
}




}
}


