// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file IterativeAbrelax
/// @brief iterative protocol starting with abinitio and getting progressively more concerned with full-atom relaxed structures
/// @detailed
///
///
/// @author Oliver Lange


// Unit Headers
#include <protocols/abinitio/IterativeAbrelax.hh>
#include <protocols/jd2/archive/ArchiveManager.hh>

// Package Headers

// Project Headers
#include <core/types.hh>
#include <core/pose/Pose.hh>

#include <core/io/silent/SilentStruct.hh>
#include <core/io/silent/SilentFileData.hh>

#include <core/fragment/ConstantLengthFragSet.hh>
#include <core/fragment/FragmentIO.hh>
#include <core/fragment/util.hh>

#include <protocols/toolbox/DecoySetEvaluation.hh>
#include <core/scoring/ScoreFunction.fwd.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

// #include <core/scoring/ScoreType.hh>
// //#include <core/kinematics/MoveMap.hh>
// #include <core/types.hh>
// #include <core/scoring/rms_util.hh> //for ConvergenceCheck
// //#include <core/pack/task/PackerTask.fwd.hh>
// #include <core/scoring/constraints/ConstraintSet.hh>
// //only needed because of temporary output_debug_structure ...

// #include <core/io/silent/SilentStructFactory.hh>
// #include <core/options/keys/out.OptionKeys.gen.hh>
// #include <protocols/jd2/util.hh>
// //#include <protocols/moves/Mover.hh>
// //#include <protocols/moves/MoverContainer.hh>
// #include <protocols/moves/TrialMover.hh>
// #include <protocols/moves/RepeatMover.hh>
// //#include <protocols/moves/WhileMover.hh>
// #include <protocols/checkpoint/Checkpoint.hh>

// ObjexxFCL Headers
// //#include <ObjexxFCL/string.functions.hh>

// Utility headers
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/file/FileName.hh>

// #include <utility/exit.hh>
// #include <utility/vector1.fwd.hh>
// #include <utility/pointer/ReferenceCount.hh>
// #include <utility/file/file_sys_util.hh>
// #include <numeric/numeric.functions.hh>
// #include <core/util/prof.hh>
// #include <core/util/Tracer.hh>
// #include <core/options/option.hh>

// Option Headers
#include <core/options/keys/abinitio.OptionKeys.gen.hh>
// #include <core/options/keys/run.OptionKeys.gen.hh>
// //#include <core/options/keys/templates.OptionKeys.gen.hh>

//// C++ headers
#include <cstdlib>
#include <string>
#include <ctime>
#include <iterator>

// Utility headers
#include <core/options/option_macros.hh>

static core::util::Tracer tr("protocols.iterative");

using core::Real;
using namespace core;
using namespace util;
using namespace core::options;
using namespace OptionKeys;

OPT_1GRP_KEY( Boolean, iterative, fullatom )

namespace protocols {
namespace abinitio {
using namespace jd2::archive;

bool IterativeAbrelax::options_registered_( false );


void protocols::abinitio::IterativeAbrelax::register_options() {
	if ( !options_registered_ ) {
		NEW_OPT( iterative::fullatom, "sample fullatom structures in iterative protocol", false );
		options_registered_ = true;
	}
	IterativeBase::register_options();
	IterativeFullatom::register_options();
}

IterativeAbrelax::IterativeAbrelax( jd2::archive::ArchiveManagerAP ptr )
	: AbstractArchiveBase( ptr ),
		centroid_archive_( ptr ),
		fullatom_archive_( ptr ),
		fullatom_( false )
{
	fullatom_ = option[ iterative::fullatom ];

	//not needed anymore: if I initialize base with this finish_stage
	if ( !fullatom_ ) centroid_archive_.set_finish_stage( IterativeBase::CEN2FULLATOM );
}

void IterativeAbrelax::read_structures( core::io::silent::SilentFileData& sfd, Batch const& batch ) {
	if ( sfd.begin() != sfd.end() ) {
		pose::Pose pose;
		sfd.begin()->fill_pose( pose );
		if ( pose.is_fullatom() ) {
			runtime_assert( fullatom_ );
			fullatom_archive_.read_structures( sfd, batch );
		} else {
			centroid_archive_.read_structures( sfd, batch );
		}
	}
}

bool IterativeAbrelax::finished() const {
	return ( centroid_archive_.finished() && !fullatom_ ) || fullatom_archive_.finished();
}

bool IterativeAbrelax::still_interested( Batch& batch ) const {
	return Parent::still_interested( batch )
		&& ( centroid_archive_.still_interested( batch ) && ( fullatom_ && fullatom_archive_.still_interested( batch ) ) );
}

///@details ready for new batch .... if queue is empty batch will be generated any way, but otherwise we only generate if this yields true.
///  logic here: new batch at beginning, but only if we are in startup phase ( not a reload of a full archive )
///              otherwise make new batch if sufficiently many structures have been accepted since last batch
bool IterativeAbrelax::ready_for_batch() const {
	return centroid_archive_.ready_for_batch() || ( fullatom_ && fullatom_archive_.ready_for_batch() );
}

///@details generate new batch...
/// type of batch depends on stage_. we switch to next stage based on some convergence criteria:
/// right now it is how many decoys were accepted from last batch.. if this number drops sufficiently ---> next stage...
///    (maybe need to put a safeguard in here: ratio small but at least XXX decoys proposed since last batch... )
///
void IterativeAbrelax::generate_batch() {
	if ( fullatom_ && fullatom_archive_.ready_for_batch() ) {
		fullatom_archive_.generate_batch();
	} else centroid_archive_.generate_batch();
}

void IterativeAbrelax::idle() {
	centroid_archive_.idle();
	if ( fullatom_ ) fullatom_archive_.idle();
}

void IterativeAbrelax::save_to_file( std::string suffix ) {
	centroid_archive_.save_to_file( suffix );
	if ( fullatom_ ) fullatom_archive_.save_to_file( suffix );
}

void IterativeAbrelax::restore_from_file() {
	centroid_archive_.restore_from_file();
	if ( fullatom_ ) fullatom_archive_.restore_from_file();
}

} //abinitio
} //protocols
