// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file IterativeAbrelax
/// @brief iterative protocol starting with abinitio and getting progressively more concerned with full-atom relaxed structures
/// @detailed
///
///
/// @author Oliver Lange


// Unit Headers
#include <protocols/abinitio/IterativeBase.hh>
#include <protocols/jd2/archive/ArchiveManager.hh>

// Package Headers

// Project Headers
#include <core/types.hh>
#include <core/pose/Pose.hh>
#include <core/io/pdb/pose_io.hh>

#include <core/io/silent/SilentStruct.hh>
#include <core/io/silent/SilentFileData.hh>

#include <core/fragment/ConstantLengthFragSet.hh>
#include <core/fragment/FragmentIO.hh>
#include <core/fragment/util.hh>

#include <protocols/toolbox/DecoySetEvaluation.hh>

#include <core/scoring/ScoreFunction.fwd.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/ScoreType.hh>

#include <core/scoring/constraints/util.hh>

#include <protocols/evaluation/JumpEvaluator.hh>
#include <protocols/evaluation/RmsdEvaluator.hh>
#include <protocols/evaluation/ScoreEvaluator.hh>
// //#include <core/kinematics/MoveMap.hh>
// #include <core/types.hh>
// #include <core/scoring/rms_util.hh> //for ConvergenceCheck
// //#include <core/pack/task/PackerTask.fwd.hh>
// #include <core/scoring/constraints/ConstraintSet.hh>
// //only needed because of temporary output_debug_structure ...

// #include <core/io/silent/SilentStructFactory.hh>
// #include <core/options/keys/out.OptionKeys.gen.hh>
// #include <protocols/jd2/util.hh>
// //#include <protocols/moves/Mover.hh>
// //#include <protocols/moves/MoverContainer.hh>
// #include <protocols/moves/TrialMover.hh>
// #include <protocols/moves/RepeatMover.hh>
// //#include <protocols/moves/WhileMover.hh>
// #include <protocols/checkpoint/Checkpoint.hh>

// ObjexxFCL Headers
// //#include <ObjexxFCL/string.functions.hh>

// Utility headers
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/file/FileName.hh>

// #include <utility/exit.hh>
// #include <utility/vector1.fwd.hh>
// #include <utility/pointer/ReferenceCount.hh>
// #include <utility/file/file_sys_util.hh>
// #include <numeric/numeric.functions.hh>
// #include <core/util/prof.hh>
// #include <core/util/Tracer.hh>
// #include <core/options/option.hh>

// Option Headers
#include <core/options/keys/abinitio.OptionKeys.gen.hh>
#include <core/options/keys/constraints.OptionKeys.gen.hh>
#include <core/options/keys/jumps.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
// #include <core/options/keys/run.OptionKeys.gen.hh>
// //#include <core/options/keys/templates.OptionKeys.gen.hh>

//// C++ headers
#include <cstdlib>
#include <string>
#include <ctime>
#include <iterator>

// Utility headers
#include <core/options/option_macros.hh>

static core::util::Tracer tr("protocols.iterative");

using core::Real;
using namespace core;
using namespace util;
using namespace core::options;
using namespace OptionKeys;

OPT_2GRP_KEY( File, iterative, enumerate, broker )
OPT_2GRP_KEY( Integer, iterative, enumerate, Naccept )
OPT_2GRP_KEY( Boolean, iterative, enumerate, skip_half )
OPT_1GRP_KEY( Integer, iterative, pool_size )
OPT_1GRP_KEY( Real, iterative, turnover_rate )
OPT_1GRP_KEY( Integer, iterative, rmsf_nstruct )
OPT_1GRP_KEY( File, iterative, cen_score )
OPT_1GRP_KEY( File, iterative, cen_score_patch )

OPT_1GRP_KEY( File, iterative, fa_score )
OPT_1GRP_KEY( File, iterative, fa_score_patch )

OPT_1GRP_KEY( IntegerVector, iterative, max_nstruct )
OPT_1GRP_KEY( Real, iterative, perturb_resampling )
OPT_1GRP_KEY( Boolean, iterative, mix_frags )
OPT_1GRP_KEY( Real, iterative, min_core_fraction_to_score )
OPT_1GRP_KEY( RealVector, iterative, min_diversity )
OPT_1GRP_KEY( RealVector, iterative, accept_ratio )
OPT_1GRP_KEY( Boolean, iterative, copy_pool_for_convergence_check )

bool protocols::abinitio::IterativeBase::options_registered_( false );

//Mike: when you want to remove these Macros... leave them at least here as comment - since they provide documentation
void protocols::abinitio::IterativeBase::register_options() {
	Parent::register_options();
	if ( !options_registered_ ) {
		NEW_OPT( iterative::enumerate::Naccept, "use enumerated pairings until Naccept decoys were added to archive", 5000 );
		NEW_OPT( iterative::enumerate::broker, "broker-file for enumerated_pairings", "" );
		NEW_OPT( iterative::enumerate::skip_half, "run half of the batches without enumerated pairings -- even before Naccept is reached", false );
		NEW_OPT( iterative::pool_size, "number of structures maintained in pool", 2000 );
		NEW_OPT( iterative::turnover_rate, "exchange of X percent of archive before new batch is started", 0.1 );
		NEW_OPT( iterative::accept_ratio, "switch to new stage if accept_ratio is lower than", 0.2 );
		NEW_OPT( iterative::rmsf_nstruct, "how many structures of pool used for computations of cores", 50 );
		NEW_OPT( iterative::cen_score, "energy function for centroid pool", "score3" );
		NEW_OPT( iterative::cen_score_patch, "patch of centroi_pool energy function", "NOPATCH" );
		NEW_OPT( iterative::fa_score, "energy function for centroid pool", "score13_env_hb" );
		NEW_OPT( iterative::fa_score_patch, "patch of centroi_pool energy function", "NOPATCH" );
		NEW_OPT( iterative::max_nstruct, "give maximum numbers of structures generated for a given stage before switch -- 0 for infinite, -1 for skip stage", 0);

		NEW_OPT( iterative::mix_frags, "mix frags with original fragset", false );
		NEW_OPT( iterative::perturb_resampling, "perturb resample_stage2 start structures by this amount", 0.0 );
		NEW_OPT( iterative::min_core_fraction_to_score, "use core (2, 3, 4A) for discriminiation if more than X% of residues", 2.0); //>1 == off
		NEW_OPT( iterative::min_diversity, "don't accept structures that are closer than X to one of the pool", 1.5 );
		NEW_OPT( iterative::copy_pool_for_convergence_check, "make a copy of the pool for each batch that uses the convergence check", true );
		options_registered_ = true;
	}
}

namespace protocols {
namespace abinitio {
using namespace jd2::archive;

IterativeBase::IterativeBase( jd2::archive::ArchiveManagerAP ptr, std::string name )
	: EvaluatedArchive( ptr ),
		beta_topol_( NULL ),
		bStartedInitBatch_( false ),
		nstruct_accept_before_rerun_( Size( 1.0*option[ iterative::pool_size ]() * option[ iterative::turnover_rate ]() ) ),
		bEnumeratedLastTime_( false ),
		reference_pose_( NULL ),
		last_accepted_decoys_in_idle_( 0 ),
		fa_score_( option[ iterative::fa_score ]() ),
		fa_score_patch_( option[ iterative::fa_score_patch ]() ),
		stage_( ENUMERATION ),
		first_batch_this_stage_ ( 1 ),
		finish_stage_( CEN2FULLATOM )
{
	set_name( name );
	set_nstruct( option[ iterative::pool_size ]() );
	if ( evaluate_local() ) {
		core::scoring::ScoreFunctionOP scorefxn =
			core::scoring::ScoreFunctionFactory::create_score_function( option[ iterative::cen_score](), option[ iterative::cen_score_patch ]() );
		//		scorefxn->set_weight( scoring::atom_pair_constraint, 1.0 );
		//constraints are done via cmdline_cst if local evaluation ( see EvaluatedArchive )
		scorefxn->set_weight( scoring::linear_chainbreak, 4.0/3.0 );
		scorefxn->set_weight( scoring::overlap_chainbreak, 1.0 );
		set_scorefxn( scorefxn );
		set_weight( "score", 1.0 );
		if ( option[ constraints::cst_file ].user() )
			set_weight( "cmdline_cst", option[ constraints::cst_weight ] );

		add_evaluation( new evaluation::JumpNrEvaluator );
	} else {
		//		core::scoring::constraints::add_constraints_from_cmdline_to_scorefxn( *scorefxn );
		// in this case constraint is already included in score_final
		set_weight( "score_final", 1.0 );
	}
	set_weight( "nrjump_weighted_chainbreaks", 1.0 );
	/// is not doing anything anymore:	setup_default_evaluators();

	max_nstruct_list_ = option[ iterative::max_nstruct ]();
	if ( max_nstruct_list_.size() != ( FINISHED-1 ) ) {
		throw EXCN_Archive("wrong number of values for max_nstruct -- needs exactly "+string_of( FINISHED - 1 )+" values.");
	}
	max_nstruct_list_.push_back( 0 );

	min_diversity_list_ = option[ iterative::min_diversity ]();
	if ( min_diversity_list_.size() != ( FINISHED-1 ) ) {
		throw EXCN_Archive("wrong number of values for min_diversity -- needs exactly "+string_of( FINISHED - 1 )+" values.");
	}
	min_diversity_list_.push_back( 0 );

	target_accept_ratio_ = option[ iterative::accept_ratio ]();
	if ( target_accept_ratio_.size() != ( FINISHED-1 ) ) {
		throw EXCN_Archive("wrong number of values for accept_ratio -- needs exactly "+string_of( FINISHED - 1 )+" values.");
	}
	target_accept_ratio_.push_back( 1 );

	if ( option[ in::file::native ].user() ) {
		core::pose::PoseOP native_pose = new core::pose::Pose;
		io::pdb::pose_from_pdb( *native_pose, option[ in::file::native ]() );
		reference_pose_ = native_pose;
	}


}

///@details ready for new batch .... if queue is empty batch will be generated any way, but otherwise we only generate if this yields true.
///  logic here: new batch at beginning, but only if we are in startup phase ( not a reload of a full archive )
///              otherwise make new batch if sufficiently many structures have been accepted since last batch
bool IterativeBase::ready_for_batch() const {
	if ( !bStartedInitBatch_ && decoys().size() <= nstruct() ) return true;
	if ( decoys().size() >= nstruct() && accepts_since_last_batch() > nstruct_accept_before_rerun_ && proposed_since_last_batch() > 500 ) return true;
	return false;
}

void IterativeBase::count_structure( Batch const& batch, bool accepted ) {
	if ( batch.id() >= first_batch_this_stage_ ) {
		Parent::count_structure( batch, accepted );
	}
}

bool IterativeBase::still_interested( Batch& batch ) const {
	return Parent::still_interested( batch ) && batch.id() >= first_batch_this_stage_;
}

void IterativeBase::increment_stage() {
	if ( stage_ >= finish_stage_ ) return;
	save_to_file( "_stage" + ObjexxFCL::string_of( stage_ ) );
	max_nstruct_list_[ stage_ ] = total_proposed();
	stage_ = IterationStage( 1 + (int) stage_ );
	while ( max_nstruct_list_[ stage_ ] < 0 && stage_ < finish_stage_ ) {
		stage_ = IterationStage( 1 + (int) stage_ );
	}
	tr.Info << "switched to stage: " << stage_ << std::endl;
	save_to_file();
	reset_accept_counter();
	first_batch_this_stage_ = manager().last_batch_id()+1;
}

bool IterativeBase::add_structure( core::io::silent::SilentStructOP pss, core::io::silent::SilentStructOP from_batch ) {
	if ( min_diversity_list_[ stage() ] > 0 && !from_batch->has_energy( "pool_converged_tag" ) ) return false;
	if ( min_diversity_list_[ stage() ] == 0 ) return Parent::add_structure( pss, from_batch );

	runtime_assert( from_batch->has_energy( "pool_converged_rmsd" ) );
	Real const rmsd_to_pool( from_batch->get_energy( "pool_converged_rmsd" ) );

	if ( rmsd_to_pool < min_diversity_list_[ stage() ] ) { //structure is close in RMSD to one of archive
		std::string const tag( from_batch->get_string_value( "pool_converged_tag" ) );
		SilentStructs::iterator it;
		for ( it = decoys().begin(); it != decoys().end(); ++it ) {
			if ( (*it)->decoy_tag() == tag ) break;
		}
		if ( it != decoys().end() && select_score( pss ) < select_score( *it ) ) {
			tr.Debug << "swap " << pss->decoy_tag() << " for " << (*it)->decoy_tag() << std::endl;
			*it = pss;
			return true; //swapped structures
		}
		tr.Trace << "decoy declined because of min_diversity limit" << std::endl;
		return false; //declined ... either tag not found or score not improved...
	} else { //structure is sufficiently different -- add via score
		return Parent::add_structure( pss, from_batch );
	}
	return false; //should never get here
}


void IterativeBase::test_for_stage_end() {
	//switch to next stage ? want to have some significance to this ratio --- hence at least 1000 proposals
	tr.Info << "current accept ratio: " << current_acceptance_ratio() << " this is "
					<< ( current_acceptance_ratio() < target_accept_ratio() ? "" : "not" )
					<< " lower than " << target_accept_ratio()
					<< "\n"
					<< proposed_since_last_batch() << " proposed decoys since last batch "
					<< ( proposed_since_last_batch() <= 500 ? " ...postpone decision" : "" ) << std::endl;


	int last_stage_N( 0 );
	if ( stage_ > 1 ) last_stage_N = max_nstruct_list_[ stage_ - 1 ];
	if ( max_nstruct_list_[ stage_ ] && ( (int) total_proposed() - last_stage_N ) > max_nstruct_list_[ stage_ ] ) {
		tr.Info << "maximum number of " << max_nstruct_list_[ stage_ ]
						<< " decoys for stage " << stage_ << " is reached...switching!" << std::endl;
		increment_stage();
	}

	if ( current_acceptance_ratio() < target_accept_ratio() && proposed_since_last_batch() > 500 ) {
		increment_stage();
	}
	if ( proposed_since_last_batch() > 500 ) {
		reset_accept_counter();
	}
}

void IterativeBase::read_structures( core::io::silent::SilentFileData& sfd, Batch const& batch ) {
	Parent::read_structures( sfd, batch );
	test_for_stage_end();
}

///@details generate new batch...
/// type of batch depends on stage_. we switch to next stage based on some convergence criteria:
/// right now it is how many decoys were accepted from last batch.. if this number drops sufficiently ---> next stage...
///    (maybe need to put a safeguard in here: ratio small but at least XXX decoys proposed since last batch... )
///
void IterativeBase::generate_batch() {

	//start new batch --e
	Batch& batch( manager().start_new_batch() );
	tr.Info << "\ngenerate batch from " <<name() << " " << batch.batch() << "\n";

	batch.set_intermediate_structs();

	//enumerate pairings ( has its own switch )
	if ( (int) stage_ < (int) PURE_TOPO_RESAMPLING ) gen_enumerate_pairings( batch );

	if ( stage_ == TOPO_RESAMPLING || stage_ == PURE_TOPO_RESAMPLING ) gen_resample_topologies( batch );
	if ( stage_ == STAGE2_RESAMPLING ) {
		gen_resample_stage2( batch );
		gen_resample_fragments( batch );
	}
	if ( stage_ == CEN2FULLATOM ) {
		gen_cen2fullatom( batch );
		gen_evaluation_output( batch, true /*fullatom*/ );
		batch.set_intermediate_structs( false ); //otherwise the intermediate (centroid) structures will be scored by score_13_envhb
	} else {
		gen_evaluation_output( batch, false /*fullatom*/ );
	}

	tr.Info << std::endl;
	//finalize
	manager().finalize_batch( batch );
	// don't want to reset counters too often... if we run out of steam the QUEUE EMPTY pathway will make sure that we do more runs
	if ( proposed_since_last_batch() > 500 ) {
		reset_accept_counter();
	}
	bStartedInitBatch_ = true;

}

void IterativeBase::idle() {
	if ( last_accepted_decoys_in_idle_ != ( accepts_since_last_batch() + total_accepts() ) ) {
		last_accepted_decoys_in_idle_ = accepts_since_last_batch() + total_accepts();
		compute_cores();
	}
}

/// ============================================================================
/// -----------           methods to make new batches               ------------
/// -----          each method should only "append" to broker and flags    -----
/// ============================================================================


void IterativeBase::gen_evaluation_output( Batch& batch, bool fullatom ) {
	utility::io::ozstream flags( batch.flag_file(), std::ios::app );
	flags << "-evaluation:jump_nr" << std::endl; //add JumpNrEvaluator
	if ( !fullatom ) {
		flags << "-evaluation:extra_score " << option[ iterative::cen_score]() << std::endl
					<< "-evaluation:extra_score_column _final" << std::endl;
		flags << "-evaluation:extra_score_patch " << option[ iterative::cen_score_patch ]() << std::endl;
	} else {
		flags << "-evaluation:extra_score " << option[ iterative::fa_score]() << std::endl
					<< "-evaluation:extra_score_column _final" << std::endl;
		flags << "-evaluation:extra_score_patch " << option[ iterative::fa_score_patch ]() << std::endl;

	}

	if ( min_diversity_list_[ stage() ] > 0 ) {
		if ( option[ iterative::copy_pool_for_convergence_check ]() ) {
			io::silent::SilentFileData sfd;
			for ( SilentStructs::const_iterator it = decoys().begin(); it != decoys().end(); ++it ) {
				sfd.add_structure( *it ); //only add OP to sfd
			}
			sfd.write_all( batch.dir()+"/pool.in" );

			//			char buf[500];
			//			sprintf(buf,"cp %s/decoys.out %s/pool.in", name().c_str(), batch.dir().c_str() );
			//			system(buf);
			flags << "-mc:known_structures " << batch.dir() << "/pool.in" << std::endl;
		} else 	flags << "-mc:known_structures " << name() << "/decoys.out" << std::endl;
		flags << "-mc:max_rmsd_against_known_structures " << std::max( 0.0, min_diversity_list_[ stage() ] - 0.25 ) << std::endl;
	}


}

void IterativeBase::gen_resample_topologies( Batch& batch) {
	tr.Info << "resample topologies\n";
	compute_beta_topology();

	utility::io::ozstream broker( batch.broker_file(), std::ios::app );
	broker << "\nCLAIMER TemplateJumpClaimer \n"
		     << "NO_USE_INPUT_POSE\n"
				 << "topol_file "<< batch.dir() << "beta.top\n"
				 << "END_CLAIMER\n\n" << std::endl;
	broker.close();

	utility::io::ozstream file( batch.dir() + "beta.top" );
	file << *beta_topol_ << std::endl;

}

void IterativeBase::gen_resample_stage2( jd2::archive::Batch& batch ) {
	tr.Info << "resample_stage2 \n";
	typedef std::map< std::string, utility::vector1< std::string > > SourceFiles;
	SourceFiles sources;
	Size ct_in( 0 );
	for ( const_decoy_iterator it = decoys().begin(); it != decoys().end(); ++it ) {
		runtime_assert( (*it)->has_comment( TAG_IN_FILE ) );
		std::string tag( (*it)->get_comment( TAG_IN_FILE ) );
		utility::file::FileName file( (*it)->get_comment( SOURCE_FILE ) );
		std::string stage2_file( file.path()+file.base() + "_stage2." + file.ext() );
		sources[ stage2_file ].push_back( tag );
		++ct_in;
	}

	Size ct_read( 0 );
	io::silent::SilentStructOPs start_decoys;
	for ( SourceFiles::const_iterator it = sources.begin(); it != sources.end(); ++it ) {
		io::silent::SilentFileData sfd;
		sfd.read_file( it->first, it->second );
		if ( sfd.size() != it->second.size() ) {
			tr.Warning << "[WARNING] multiple decoys with same tag detected in file " << it->first << std::endl;
		}
		copy( sfd.begin(), sfd.end(), std::back_inserter( start_decoys ) );
		ct_read += sfd.size();
	}

	tr.Debug << "structures from pool" << ct_in << " structure retrieved from stage2-files " << ct_read << " start structs: " << start_decoys.size() << std::endl;
	if ( start_decoys.size() != nstruct() ) tr.Warning << "[WARNING] why do we have a different number of decoys in pool and start_decoys ? " << std::endl;

	if ( start_decoys.size() ) {
		batch.set_has_silent_in();
		core::io::silent::SilentFileData sfd;
		for ( core::io::silent::SilentStructOPs::const_iterator
						it = start_decoys.begin(); it != start_decoys.end(); ++it ) {
			sfd.add_structure( **it );
		}
		sfd.write_all( batch.silent_in() );


		utility::io::ozstream broker( batch.broker_file(), std::ios::app );
		broker << "\nUSE_INPUT_POSE\n"
					 << "CLAIMER StartStructClaimer\n"
			     << "PERTURB " << option[ iterative::perturb_resampling ] << std::endl
					 << "END_CLAIMER\n\n"
					 << "CLAIMER JumpClaimer\n"
					 << "END_CLAIMER\n\n" << std::endl;


		utility::io::ozstream flags( batch.flag_file(), std::ios::app );
		flags << "-abinitio::skip_stages 1 " << std::endl;
	} else {
		tr.Warning << "[WARNING] no stage2 decoys found ! " << std::endl;
	}
	batch.nstruct() = std::max( 1, int( 1.0*batch.nstruct() / ( 1.0*start_decoys.size() ) ) );
}

void IterativeBase::gen_enumerate_pairings( Batch& batch ) {
	if ( !option[ iterative::enumerate::broker ].user() ) return;
	if ( option[ iterative::enumerate::Naccept ]() < (int) total_accepts() ) return;
	if ( option[ iterative::enumerate::skip_half ]() && bEnumeratedLastTime_ ) {
		bEnumeratedLastTime_ = false;
		return;
	}
	tr.Info << "enumerate pairings\n ";
	utility::io::ozstream broker( batch.broker_file(), std::ios::app );
	runtime_assert( broker.good() );
	utility::io::izstream enum_broker( option[ iterative::enumerate::broker ]() );
	if ( !enum_broker.good() ) throw ( utility::excn::EXCN_FileNotFound
															 ( "-iterative::enumerate::broker: File "
																 +std::string( option[ iterative::enumerate::broker ]())+" not found! ") );

	std::string line;
	while ( getline( enum_broker, line ) ) broker << line << std::endl;

	bEnumeratedLastTime_ = true;
}

void IterativeBase::gen_resample_fragments( Batch& batch ) {
	using namespace core::fragment;
	ConstantLengthFragSet frags_9mer( 9 ); //steal good old 9mers
	ConstantLengthFragSet frags_3mer( 3 ); //steal good old 9mers

	Size ct( 1 );
	Size const max_frags( 500 );
	if ( option[ iterative::mix_frags ]() ) {
		FragSetOP frags_l = FragmentIO(
		   option[ OptionKeys::abinitio::number_9mer_frags ]()
	  ).read( option[ in::file::frag9 ] );
		FragSetOP frags_s = FragmentIO(
		   option[ OptionKeys::abinitio::number_3mer_frags ]()
	  ).read( option[ in::file::frag3 ] );
		frags_9mer.add( *frags_l );//assuming we really have read 9mers and 3mers.
		frags_3mer.add( *frags_s );
	}
	for ( const_decoy_iterator it = decoys().begin(); it != decoys().end() && ct <= max_frags; ++it, ++ct ) {
		pose::Pose pose;
		std::string tag;// = it->decoy_tag();
		(*it)->fill_pose( pose );
		steal_constant_length_frag_set_from_pose( pose, frags_9mer );
		steal_constant_length_frag_set_from_pose( pose, frags_3mer );
	}
	std::string file_9mer( batch.dir() + "frags_9mer.dat.gz" );
	std::string file_3mer( batch.dir() + "frags_3mer.dat.gz" );
	FragmentIO().write( file_9mer, frags_9mer );
	FragmentIO().write( file_3mer, frags_3mer );

	utility::io::ozstream flags( batch.flag_file(), std::ios::app );
	flags << "-frag3 " << file_3mer << std::endl
				<< "-frag9 " << file_9mer << std::endl
				<< "-abinitio:number_3mer_frags 0" << std::endl
				<< "-abinitio:number_9mer_frags 0" << std::endl;

}

void IterativeBase::add_fullatom_flags( jd2::archive::Batch& batch ) {
	utility::io::ozstream flags( batch.flag_file(), std::ios::app );
	//these are many.... should there be some input file ?
	flags << "-relax:sequence" << std::endl
				<< "-abinitio:close_loops" << std::endl
				<< "-loops::idealize_after_loop_close" << std::endl
				<< "-abinitio::clear_pose_cache" << std::endl
				<< "-short_frag_cycles 1" << std::endl
				<< "-scored_frag_cycles 1" << std::endl
				<< "-non_ideal_loop_closing" << std::endl
				<< "-alternative_closure_protocol" << std::endl
				<< "-fast_loops:window_accept_ratio .01"<<std::endl
				<< "-fast_loops:nr_scored_sampling_passes 4" << std::endl
				<< "-fast_loops:min_breakout_good_loops 5" << std::endl
				<< "-fast_loops:min_breakout_fast_loops 80" << std::endl
				<< "-fast_loops:min_fast_loops 3" << std::endl
				<< "-fast_loops:min_good_loops 0" << std::endl
				<< "-fast_loops:nr_scored_fragments 20" << std::endl
				<< "-fast_loops:vdw_delta 0.5" << std::endl
				<< "-fast_loops:give_up 1000" << std::endl;



	if ( option[ constraints::cst_file ].user() ) {
		utility::io::ozstream broker( batch.broker_file(), std::ios::app );
		broker << "\nCLAIMER ConstraintClaimer \n"
					 << "CMD_FLAG\n"
					 << "FULLATOM\n"
			     << "END_CLAIMER\n"
					 << std::endl;
	}

}

void IterativeBase::gen_cen2fullatom( Batch& batch ) {
	utility::io::ozstream broker( batch.broker_file(), std::ios::app );
	broker << "\nUSE_INPUT_POSE\n"
				 << "CLAIMER StartStructClaimer\n"
				 << "END_CLAIMER\n\n"
				 << "CLAIMER JumpClaimer\n"
				 << "END_CLAIMER\n\n" << std::endl;


	utility::io::ozstream flags( batch.flag_file(), std::ios::app );
	flags << "-abinitio::skip_stages 1 2 3 4" << std::endl;

	add_fullatom_flags( batch );
	io::silent::SilentStructOPs start_decoys;
	std::copy( decoys().begin(), decoys().end(), std::back_inserter( start_decoys ) );

	if ( start_decoys.size() ) {
		batch.set_has_silent_in();
		core::io::silent::SilentFileData sfd;
		for ( core::io::silent::SilentStructOPs::const_iterator
						it = start_decoys.begin(); it != start_decoys.end(); ++it ) {
			sfd.add_structure( **it );
		}
		sfd.write_all( batch.silent_in() );
	}

	batch.nstruct() = std::max( 1, int( 1.0*batch.nstruct() / ( 1.0*start_decoys.size() ) ) );

}

/// ============================================================================
/// -----------           methods to compute stuff from archive     ------------
/// -----------             these may be called from idle()         ------------
/// ============================================================================

void IterativeBase::compute_beta_topology() {
	//use -jumps::max_strand_gap_allowed 10 -jumps:contact_score 0.2
	if ( !option[ jumps::max_strand_gap_allowed ].user() ) {
		option[ jumps::max_strand_gap_allowed ].def( 10 );
	}
	if ( !option[ jumps::contact_score ].user() ) {
		option[ jumps::contact_score ].def( 0.2 );
	}

	beta_topol_ = new PairingStatistics;
	PairingStatistics::ModelFreq model_freq;
	core::Size ct( 1 );
	for ( const_decoy_iterator it = decoys().begin(); it != decoys().end(); ++it, ++ct ) {
		pose::Pose pose;
		std::string tag;// = it->decoy_tag();
		(*it)->fill_pose( pose );

		// get strand pairings
		jumping::StrandPairingSet strand_pairings( pose );

		tag = right_string_of( ct, 4, '0');
		beta_topol_->add_topology( strand_pairings, tag );
		model_freq[ tag.substr(0, 4) ] += 1;

	} // for decoys
	// set score terms
	beta_topol_->compute_model_weights( model_freq );
}


void get_core( toolbox::DecoySetEvaluation& eval, core::Real cutoff, loops::Loops& rigid ) {
	ObjexxFCL::FArray1D_double weights( eval.n_atoms(), 1.0 );
	eval.superimpose();
	eval.wRMSD( cutoff, 0.00001, weights );

	utility::vector1< Real > result;
	eval.rmsf( result );

	rigid.clear();
	for ( Size i=1; i<=result.size(); ++i ) {
		if ( result[ i ] < cutoff ) rigid.add_loop( loops::Loop(  i, i ), 5 );
	}
	tr.Debug << "make rigid with cutoff " << cutoff << "  " << rigid << std::endl;
}


void IterativeBase::compute_cores() {
	core::Size const opt_nstruct( option[ iterative::rmsf_nstruct ]() );
	core::Size const nstruct( std::min( int( opt_nstruct ), int( decoys().size() ) ));
	toolbox::DecoySetEvaluation eval;
	eval.reserve( nstruct );
	Size ct( 1 );
	Size nres(1000);
	for ( const_decoy_iterator iss = decoys().begin(); iss != decoys().end(); ++iss, ++ct ) {
		if ( ct > nstruct ) break;
		pose::Pose pose;
		(*iss)->fill_pose( pose );
		eval.push_back( pose );
		nres = pose.total_residue();
	}

	loops::Loops old_core2 = core2_;
	loops::Loops old_core3 = core3_;
	loops::Loops old_core4 = core4_;

	get_core( eval, 2, core2_ );
	get_core( eval, 3, core3_ );
	get_core( eval, 4, core4_ );

	if ( old_core2 != core2_ ) add_core_evaluator( core2_, "_core2" );
	if ( old_core3 != core3_ ) add_core_evaluator( core3_, "_core3" );
	if ( old_core4 != core4_ ) add_core_evaluator( core4_, "_core4" );

	if ( evaluate_local() ) {
		set_weight( "score_core2", 0.0 );
		set_weight( "score_core3", 0.0 );
		set_weight( "score_core4", 0.0 );
		set_weight( "score", 0.0 );

		if ( ( 1.0*core2_.loop_size() / nres ) > option[ iterative::min_core_fraction_to_score ] ) {
			set_weight( "score_core2", 1.0);
			return;
		}

		if ( ( 1.0*core3_.loop_size() / nres ) > option[ iterative::min_core_fraction_to_score ] ) {
			set_weight( "score_core3", 1.0);
			return;
		}

		if ( ( 1.0*core4_.loop_size() / nres ) > option[ iterative::min_core_fraction_to_score ] ) {
			set_weight( "score_core4", 1.0);
			return;
		}

		set_weight( "score", 1.0 );
	}
}

void IterativeBase::add_core_evaluator( loops::Loops const& core, std::string const& core_tag ) {
	utility::vector1< Size> selection;
	core.get_residues( selection );
	if ( reference_pose_ ) add_evaluation( new evaluation::SelectRmsdEvaluator( reference_pose_, selection, core_tag ) );
	add_evaluation( new evaluation::TruncatedScoreEvaluator( core_tag, selection ) );
	core.write_loops_to_file( name()+"/"+core_tag+".rigid", "RIGID" ); //so we have them for other evaluations
}




// void IterativeBase::restore_from_file( std::string const& dirname ) {
// 	EvaluatedArchive::restore_from_file( dirname );
// 	utility::io::izstream stage( dirname+"/IterationStage" );
// 	int bla;
// 	stage >> bla;
// 	stage_ = IterationStage( bla );
// }

// void IterativeBase::save_to_file( std::string const& dirname ) {
// 	EvaluatedArchive::save_to_file( dirname );
// 	utility::io::ozstream stage( dirname+"/IterationStage" );
// 	stage << stage_;
// }

void IterativeBase::restore_status( std::istream& is ) {
	EvaluatedArchive::restore_status( is );
	int bla; std::string tag;
	is >> tag >> bla;
	runtime_assert( tag == "IterationStage:" );
	stage_ = IterationStage( bla );
	tr.Info << "restored iteration stage: " << stage_ << std::endl;
	is >> tag >> bla;
	runtime_assert( tag == "first_batch_this_stage:");
	first_batch_this_stage_ = bla;
	compute_cores();
}

void IterativeBase::save_status( std::ostream& os ) const {
	EvaluatedArchive::save_status( os );
	os << "IterationStage: " << stage_;
	os << "   first_batch_this_stage: " << first_batch_this_stage_;
	os << std::endl;
}


void IterativeBase::setup_default_evaluators() {
	Parent::setup_default_evaluators();
	add_evaluation( new evaluation::JumpNrEvaluator );
}

} //abinitio
} //protocols
