// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file AbrelaxMover
/// @brief  this class will be handled to a SampleProtocol as a control instance
/// @detailed responsibilities:
///           know which chainbreaks to penalize and close
///           know which jumps to use during sampling, which (if any) to keep after loop-closing
///           supply a JumpMover if jumps should be moved
///           supply a MoveMap
///           supply a "StrictMoveMap": the protocol should not move anything that is dissallowed in strict_movemap(),
///                      it should try to move just stuff in movemap()
/// should this class also know how to ramp score terms ?
/// handle the titration of constraints ?
/// @author Oliver Lange


#ifndef INCLUDED_protocols_abinitio_IterativeBase_HH
#define INCLUDED_protocols_abinitio_IterativeBase_HH

// Unit Headers
//#include <protocols/abinitio/IterativeAbrelax.fwd.hh>

// Package Headers
#include <protocols/jd2/archive/ArchiveBase.hh>
#include <protocols/jd2/archive/EvaluatedArchive.hh>

// Project Headers
#include <protocols/abinitio/PairingStatistics.hh>

#include <core/types.hh>
#include <core/pose/Pose.hh>

#include <protocols/loops/LoopClass.hh>

// ObjexxFCL Headers
//#include <ObjexxFCL/FArray1D.hh>
//#include <ObjexxFCL/FArray2D.hh>

// Utility headers
#include <utility/vector1.hh>
#include <utility/pointer/ReferenceCount.hh>

//// C++ headers
#include <cstdlib>
#include <string>


namespace protocols {
namespace abinitio {

class IterativeBase : public jd2::archive::EvaluatedArchive {
	typedef jd2::archive::EvaluatedArchive Parent;
public:
	enum IterationStage {
		ENUMERATION = 1,
    TOPO_RESAMPLING,
		PURE_TOPO_RESAMPLING,
    STAGE2_RESAMPLING,
		CEN2FULLATOM,
    FLEX_CORE_RESAMPLING,
		RIGID_CORE_RESAMPLING,
    FINISHED //keep last
	};

  IterativeBase( jd2::archive::ArchiveManagerAP ptr, std::string name );

  virtual bool finished() const { return stage_ >= finish_stage_; };
  virtual bool ready_for_batch() const;
 	virtual bool still_interested( jd2::archive::Batch& batch ) const;
  virtual void generate_batch();
  virtual void idle();

  static void register_options();

  //save Evaluator state ?
//   virtual void save_to_file( std::string const& dirname );
//   virtual void restore_from_file( std::string const& dirname );

  virtual void save_status( std::ostream& ) const;
	virtual void restore_status( std::istream& );

	//special convergence check
	virtual bool add_structure( core::io::silent::SilentStructOP, core::io::silent::SilentStructOP );

	///@brief set common evaluators: eg. ConstraintEvaluator if -cst_file is present
	void setup_default_evaluators();

	core::Real target_accept_ratio() const { return target_accept_ratio_[ stage_ ]; }

	virtual void read_structures( core::io::silent::SilentFileData&, jd2::archive::Batch const& batch );

protected:

  void gen_resample_topologies( jd2::archive::Batch& batch );
  void gen_enumerate_pairings( jd2::archive::Batch& batch );
  void gen_resample_stage2( jd2::archive::Batch& batch );
	void gen_resample_fragments( jd2::archive::Batch& batch );
	void gen_cen2fullatom( jd2::archive::Batch& batch );
	void add_fullatom_flags( jd2::archive::Batch& batch );
	void gen_evaluation_output( jd2::archive::Batch& batch, bool fullatom = false );

  void compute_beta_topology();
  void compute_cores();

	std::string const& fa_score() const {
		return fa_score_;
	}
	std::string const& fa_score_patch() const {
		return fa_score_patch_;
	}

	loops::Loops const& core( core::Size i ) {
		if ( i == 2 ) { return core2_; };
		if ( i == 3 ) { return core3_; };
		if ( i == 4 ) { return core4_; };
		runtime_assert( false );
		return core2_; //happy compiler
	}

	virtual void count_structure( jd2::archive::Batch const& batch, bool accepted  );

public:
	IterationStage stage() const {
		return stage_;
	}

	void set_finish_stage( IterationStage setting ) {
		finish_stage_ = setting;
	}

protected:
	void set_stage( IterationStage setting ) {
		stage_ = setting;
	}

private:
	void add_core_evaluator( loops::Loops const& core, std::string const& core_tag );
	void increment_stage();
	void test_for_stage_end();

  PairingStatisticsOP beta_topol_;

  bool bStartedInitBatch_;
  core::Size nstruct_accept_before_rerun_;

  bool bEnumeratedLastTime_;

  loops::Loops core2_;
  loops::Loops core3_;
  loops::Loops core4_;

	utility::vector1< int > max_nstruct_list_; //-1 --> skip stage, 0 infinite, N>0 make a maximum of N structures.
	utility::vector1< core::Real > min_diversity_list_;
	utility::vector1< core::Real > target_accept_ratio_;
	core::pose::PoseCOP reference_pose_;

	core::Size last_accepted_decoys_in_idle_;

	static bool options_registered_;

	std::string fa_score_;
	std::string fa_score_patch_;

  IterationStage stage_;
	core::Size first_batch_this_stage_;
	IterationStage finish_stage_;
};


}
}

#endif
