// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file IterativeAbrelax
/// @brief iterative protocol starting with abinitio and getting progressively more concerned with full-atom relaxed structures
/// @detailed
///
///
/// @author Oliver Lange


// Unit Headers
#include <protocols/abinitio/IterativeFullatom.hh>
#include <protocols/jd2/archive/ArchiveManager.hh>

// Package Headers

// Project Headers
#include <core/types.hh>
#include <core/pose/Pose.hh>

#include <core/io/silent/SilentStruct.hh>
#include <core/io/silent/SilentFileData.hh>

#include <core/fragment/ConstantLengthFragSet.hh>
#include <core/fragment/FragmentIO.hh>
#include <core/fragment/util.hh>

#include <protocols/toolbox/DecoySetEvaluation.hh>
#include <core/scoring/ScoreFunction.fwd.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/scoring/constraints/util.hh>
// #include <core/scoring/ScoreType.hh>
// //#include <core/kinematics/MoveMap.hh>
// #include <core/types.hh>
// #include <core/scoring/rms_util.hh> //for ConvergenceCheck
// //#include <core/pack/task/PackerTask.fwd.hh>
// #include <core/scoring/constraints/ConstraintSet.hh>
// //only needed because of temporary output_debug_structure ...

// #include <core/io/silent/SilentStructFactory.hh>
// #include <core/options/keys/out.OptionKeys.gen.hh>
// #include <protocols/jd2/util.hh>
// //#include <protocols/moves/Mover.hh>
// //#include <protocols/moves/MoverContainer.hh>
// #include <protocols/moves/TrialMover.hh>
// #include <protocols/moves/RepeatMover.hh>
// //#include <protocols/moves/WhileMover.hh>
// #include <protocols/checkpoint/Checkpoint.hh>

// ObjexxFCL Headers
// //#include <ObjexxFCL/string.functions.hh>

// Utility headers
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/file/FileName.hh>

// #include <utility/exit.hh>
// #include <utility/vector1.fwd.hh>
// #include <utility/pointer/ReferenceCount.hh>
// #include <utility/file/file_sys_util.hh>
// #include <numeric/numeric.functions.hh>
// #include <core/util/prof.hh>
// #include <core/util/Tracer.hh>
// #include <core/options/option.hh>

// Option Headers
#include <core/options/keys/abinitio.OptionKeys.gen.hh>
#include <core/options/keys/constraints.OptionKeys.gen.hh>
// #include <core/options/keys/run.OptionKeys.gen.hh>
// //#include <core/options/keys/templates.OptionKeys.gen.hh>

//// C++ headers
#include <cstdlib>
#include <string>
#include <ctime>
#include <iterator>

// Utility headers
#include <core/options/option_macros.hh>

static core::util::Tracer tr("protocols.iterative");

using core::Real;
using namespace core;
using namespace util;
using namespace core::options;
using namespace OptionKeys;


bool protocols::abinitio::IterativeFullatom::options_registered_( false );

void protocols::abinitio::IterativeFullatom::register_options() {
	IterativeBase::register_options();
	if ( !options_registered_ ) {
		options_registered_ = true;
	}
}

namespace protocols {
namespace abinitio {
using namespace jd2::archive;

IterativeFullatom::IterativeFullatom( jd2::archive::ArchiveManagerAP ptr )
	: IterativeBase( ptr, "fullatom_pool" )
{
	set_stage( CEN2FULLATOM );
	set_finish_stage( FINISHED );

	if ( evaluate_local() ) {
		 	core::scoring::ScoreFunctionOP scorefxn =
				core::scoring::ScoreFunctionFactory::create_score_function( fa_score(), fa_score_patch() );
			// core::scoring::constraints::add_fa_constraints_from_cmdline_to_scorefxn( *scorefxn );
			//			scorefxn->set_weight( scoring::atom_pair_constraint, 1.0 );
			// if local evaluation done by cmdline_cst evaluator
			if ( option[ constraints::cst_file ].user() )
				set_weight( "cmdline_cst", option[ constraints::cst_fa_weight ] );

			set_scorefxn( scorefxn );
	}

	perturb_start_structures_ = 2;
}

bool IterativeFullatom::ready_for_batch() const {
	return ( stage() > CEN2FULLATOM );
}

///@details generate new batch...
/// type of batch depends on stage_. we switch to next stage based on some convergence criteria:
/// right now it is how many decoys were accepted from last batch.. if this number drops sufficiently ---> next stage...
///    (maybe need to put a safeguard in here: ratio small but at least XXX decoys proposed since last batch... )
///
void IterativeFullatom::generate_batch() {

	//start new batch
	Batch& batch( manager().start_new_batch() );
	tr.Info << "\ngenerate batch from " <<name() << " " << batch.batch() << "\n";

	batch.set_intermediate_structs( false );

	gen_resample_fragments( batch );
	gen_evaluation_output( batch, true /*fullatom*/ );
	if ( stage() == FLEX_CORE_RESAMPLING ) {
		gen_resample_core( batch, true /*flex*/ );
	}
	if ( stage() == RIGID_CORE_RESAMPLING ) {
		gen_resample_core( batch, false /*flex*/ );
	}


	tr.Info << std::endl;
	//finalize
	manager().finalize_batch( batch );
	// don't want to reset counters too often... if we run out of steam the QUEUE EMPTY pathway will make sure that we do more runs
	if ( proposed_since_last_batch() > 500 ) {
		reset_accept_counter();
	}


}


/// ============================================================================
/// -----------           methods to make new batches               ------------
/// -----          each method should only "append" to broker and flags    -----
/// ============================================================================


void IterativeFullatom::gen_resample_core( Batch& batch, bool flex ) {
	//copy pool as input decoys
	batch.set_has_silent_in();
	io::silent::SilentFileData sfd;
	for ( SilentStructs::const_iterator it = decoys().begin(); it != decoys().end(); ++it ) {
		sfd.add_structure( *it ); //only add OP to sfd
	}
	sfd.write_all( batch.silent_in() );

	//take rigid-core definition that has most individual loops --- most jumps
	Size most_jumps( 0 ), nr_jumps( 0 );
	for ( Size i = 2; i<=4; ++i ) {
		if ( core( i ).num_loop() > nr_jumps ) {
			nr_jumps = core( i ).num_loop();
			most_jumps = i;
		}
	}

	utility::io::ozstream broker( batch.broker_file(), std::ios::app );
	broker << "\nUSE_INPUT_POSE" << std::endl;

	core( most_jumps ).write_loops_to_file( batch.dir()+"core.rigid", "RIGID" );
	broker << "\nCLAIMER RigidChunkClaimer \n"
				 << "REGION_FILE "<< batch.dir() << "core.rigid\n"
				 << ( flex ? "KEEP_FLEXIBLE\n" : "" )
				 << "END_CLAIMER\n\n" << std::endl;

	broker << "\nCLAIMER StartStructClaimer\n"
				 << "PERTURB " << perturb_start_structures_ << "\n"
				 << "END_CLAIMER\n\n" << std::endl;

	broker.close();

	utility::io::ozstream flags( batch.flag_file(), std::ios::app );
	flags << "-abinitio::skip_stages 1 2" << std::endl;

	add_fullatom_flags( batch );
	batch.nstruct() = std::max( 1, int( 1.0*batch.nstruct() / ( 1.0*decoys().size() ) ) );
}

} //abinitio
} //protocols
