// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief Helper class for FoldConstraints Protocol, filters constraints by sequence separation
/// @detailed
///
///
///
/// @author Oliver Lange


#ifndef INCLUDED_protocols_abinitio_MaxSeqSepConstraintSet_HH
#define INCLUDED_protocols_abinitio_MaxSeqSepConstraintSet_HH


// Unit Headers
#include <protocols/abinitio/MaxSeqSepConstraintSet.fwd.hh>

// Package Headers


// Project Headers
#include <core/pose/Pose.fwd.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/kinematics/ShortestPathInFoldTree.hh>
#include <core/types.hh>
#include <core/scoring/ScoreFunction.fwd.hh>
#include <core/scoring/ScoreType.hh>
#include <core/pack/task/PackerTask.fwd.hh>
#include <core/scoring/constraints/ConstraintSet.hh>

// ObjexxFCL Headers

// Utility headers
#include <utility/vector1.fwd.hh>

//// C++ headers
#include <cstdlib>
#include <string>


namespace protocols {
namespace abinitio {

class MaxSeqSepConstraintSet : public core::scoring::constraints::ConstraintSet {
  typedef ConstraintSet Parent;
  typedef core::scoring::constraints::ConstraintSetOP ConstraintSetOP;
public:
  MaxSeqSepConstraintSet( ConstraintSet const & other, core::kinematics::FoldTree const&f );

  MaxSeqSepConstraintSet( MaxSeqSepConstraintSet const &other );

protected:
  MaxSeqSepConstraintSet( ConstraintSet const &other, core::kinematics::ShortestPathInFoldTreeOP sp );

public:
  ConstraintSetOP clone() const {
    return new MaxSeqSepConstraintSet ( *this );
  };

	virtual ConstraintSetOP remapped_clone(
		core::pose::Pose const& src,
		core::pose::Pose const& dest,
		core::sequence::SequenceMappingCOP smap = NULL
	) const;

  Size max_seq_sep () const {
    return max_seq_sep_;
  };

  void set_max_seq_sep( Size setting ) {
    max_seq_sep_=setting;
    mark_revision_id_expired(); //force recompute of cached energies
  };

  ///
  void
  residue_pair_energy(
     Residue const & rsd1,
     Residue const & rsd2,
     Pose const & pose,
     core::scoring::ScoreFunction const & scorefxn,
     core::scoring::EnergyMap & emap
  ) const;

  bool too_far( int const pos1, int const pos2 ) const;

  Size
  largest_possible_sequence_sep( core::pose::Pose const& pose ) const;

  ///
  bool
  residue_pair_constraint_exists( int const pos1, int const pos2 ) const
  {
    if ( too_far( pos1, pos2 ) ) return false;
    return Parent::residue_pair_constraint_exists( pos1, pos2 );
  }

  Size
  show_violations( std::ostream& out, core::pose::Pose&, Size verbose_level, core::Real threshold = 1  );

  core::kinematics::ShortestPathInFoldTree const&
  shortest_path() const {
    return *shortest_path_;
  }

protected:
  ///
  void
  eval_atom_derivative_for_residue_pairs(
    core::id::AtomID const & atom_id,
    core::pose::Pose const & pose,
    core::scoring::ScoreFunction const &,
    core::scoring::EnergyMap const & weights,
    core::Vector & F1,
    core::Vector & F2
  ) const;


private:
  Size max_seq_sep_;
  core::kinematics::ShortestPathInFoldTreeOP shortest_path_;
};

} // abinitio
} // protocols

#endif
