// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Aroop Sircar

// Rosetta Headers
#include <core/id/AtomID_Map.Pose.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/pose/PDBInfo.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/rms_util.hh>
#include <core/types.hh>
#include <core/util/Tracer.hh>

#include <protocols/antibody/AntibodyClass.hh>
#include <protocols/loops/LoopClass.hh>

#include <utility/vector1.hh>

static core::util::Tracer TR("antibody");

namespace protocols{
	namespace antibody{

		/// default constructor
		Antibody::Antibody() {
			lfr_start_1 = 0;
			lfr_end_1 = 0;
			lfr_start_2 = 0;
			lfr_end_2 = 0;
			cdr_l1_start = 0;
			cdr_l1_end = 0;
			lfr_start_3 = 0;
			lfr_end_3 = 0;
			lfr_start_4 = 0;
			lfr_end_4 = 0;
			cdr_l2_start = 0;
			cdr_l2_end = 0;
			lfr_start_5 = 0;
			lfr_end_5 = 0;
			lfr_start_6 = 0;
			lfr_end_6 = 0;
			cdr_l3_start = 0;
			cdr_l3_end = 0;
			lfr_start_7 = 0;
			lfr_end_7 = 0;
			hfr_start_1 = 0;
			hfr_end_1 = 0;
			hfr_start_2 = 0;
			hfr_end_2 = 0;
			cdr_h1_start = 0;
			cdr_h1_end = 0;
			hfr_start_3 = 0;
			hfr_end_3 = 0;
			hfr_start_4 = 0;
			hfr_end_4 = 0;
			cdr_h2_start = 0;
			cdr_h2_end = 0;
			hfr_start_5 = 0;
			hfr_end_5 = 0;
			cdr_h3_start = 0;
			cdr_h3_end = 0;
			hfr_start_6 = 0;
			hfr_end_6 = 0;
			current_start = 0;
			current_end = 0;
			kinked_ = false;
			extended_ = false;
		}


		/// constructor with arguments
		Antibody::Antibody( core::pose::Pose& pose_in ) {
			Fv = pose_in;
			set_defaults();
			current_start = 0;
			current_end = 0;

			update_sequence();
			kinked_ = false;
			extended_ = false;
			detect_CDR_H3_stem_type();
		}

		/// constructor with arguments
		Antibody::Antibody(
		  core::pose::Pose& pose_in,
			std::string cdr_name ) {
			Fv = pose_in;
			if( cdr_name == "l1" ) {
				cdr_l1_start = Fv.pdb_info()->pdb2pose( 'L', 24 );
				cdr_l1_end = Fv.pdb_info()->pdb2pose( 'L', 34 );
				current_start = cdr_l1_start;
				current_end = cdr_l1_end;
			}
			else if( cdr_name == "l2" ) {
				cdr_l2_start = Fv.pdb_info()->pdb2pose( 'L', 50 );
				cdr_l2_end = Fv.pdb_info()->pdb2pose( 'L', 56 );
				current_start = cdr_l2_start;
				current_end = cdr_l2_end;
			}
			else if( cdr_name == "l3" ) {
				cdr_l3_start = Fv.pdb_info()->pdb2pose( 'L', 89 );
				cdr_l3_end = Fv.pdb_info()->pdb2pose( 'L', 97 );
				current_start = cdr_l3_start;
				current_end = cdr_l3_end;
			}
			else if( cdr_name == "h1" ) {
				cdr_h1_start = Fv.pdb_info()->pdb2pose( 'H', 26 );
				cdr_h1_end = Fv.pdb_info()->pdb2pose( 'H', 35 );
				current_start = cdr_h1_start;
				current_end = cdr_h1_end;
			}
			else if( cdr_name == "h2" ) {
				cdr_h2_start = Fv.pdb_info()->pdb2pose( 'H', 50 );
				cdr_h2_end = Fv.pdb_info()->pdb2pose( 'H', 65 );
				current_start = cdr_h2_start;
				current_end = cdr_h2_end;
			}
			else if( cdr_name == "h3" ) {
				cdr_h3_start = Fv.pdb_info()->pdb2pose( 'H', 95 );
				cdr_h3_end = Fv.pdb_info()->pdb2pose( 'H', 102 );
				current_start = cdr_h3_start;
				current_end = cdr_h3_end;
			}
			else {
				current_start = 0;
				current_end = 0;
			}
		} // constructor with arguments

		void
		Antibody::set_defaults() {
			lfr_start_1 = Fv.pdb_info()->pdb2pose( 'L', 4 );
			lfr_end_1 = Fv.pdb_info()->pdb2pose( 'L', 6 );
			lfr_start_2 = Fv.pdb_info()->pdb2pose( 'L', 10 );
			lfr_end_2 = Fv.pdb_info()->pdb2pose( 'L', 23 );
			cdr_l1_start = Fv.pdb_info()->pdb2pose( 'L', 24 );
			cdr_l1_end = Fv.pdb_info()->pdb2pose( 'L', 34 );
			lfr_start_3 = Fv.pdb_info()->pdb2pose( 'L', 35 );
			lfr_end_3 = Fv.pdb_info()->pdb2pose( 'L', 38 );
			lfr_start_4 = Fv.pdb_info()->pdb2pose( 'L', 45 );
			lfr_end_4 = Fv.pdb_info()->pdb2pose( 'L', 49 );
			cdr_l2_start = Fv.pdb_info()->pdb2pose( 'L', 50 );
			cdr_l2_end = Fv.pdb_info()->pdb2pose( 'L', 56 );
			lfr_start_5 = Fv.pdb_info()->pdb2pose( 'L', 57 );
			lfr_end_5 = Fv.pdb_info()->pdb2pose( 'L', 66 );
			lfr_start_6 = Fv.pdb_info()->pdb2pose( 'L', 71 );
			lfr_end_6 = Fv.pdb_info()->pdb2pose( 'L', 88 );
			cdr_l3_start = Fv.pdb_info()->pdb2pose( 'L', 89 );
			cdr_l3_end = Fv.pdb_info()->pdb2pose( 'L', 97 );
			lfr_start_7 = Fv.pdb_info()->pdb2pose( 'L', 98 );
			lfr_end_7 = Fv.pdb_info()->pdb2pose( 'L', 104 );
			hfr_start_1 = Fv.pdb_info()->pdb2pose( 'H', 5 );
			hfr_end_1 = Fv.pdb_info()->pdb2pose( 'H', 6 );
			hfr_start_2 = Fv.pdb_info()->pdb2pose( 'H', 10 );
			hfr_end_2 = Fv.pdb_info()->pdb2pose( 'H', 25 );
			cdr_h1_start = Fv.pdb_info()->pdb2pose( 'H', 26 );
			cdr_h1_end = Fv.pdb_info()->pdb2pose( 'H', 35 );
			hfr_start_3 = Fv.pdb_info()->pdb2pose( 'H', 36 );
			hfr_end_3 = Fv.pdb_info()->pdb2pose( 'H', 39 );
			hfr_start_4 = Fv.pdb_info()->pdb2pose( 'H', 46 );
			hfr_end_4 = Fv.pdb_info()->pdb2pose( 'H', 49 );
			cdr_h2_start = Fv.pdb_info()->pdb2pose( 'H', 50 );
			cdr_h2_end = Fv.pdb_info()->pdb2pose( 'H', 65 );
			hfr_start_5 = Fv.pdb_info()->pdb2pose( 'H', 66 );
			hfr_end_5 = Fv.pdb_info()->pdb2pose( 'H', 94 );
			cdr_h3_start = Fv.pdb_info()->pdb2pose( 'H', 95 );
			cdr_h3_end = Fv.pdb_info()->pdb2pose( 'H', 102 );
			hfr_start_6 = Fv.pdb_info()->pdb2pose( 'H', 103 );
			hfr_end_6 = Fv.pdb_info()->pdb2pose( 'H', 110 );

			cdr_h3_cut_ = cdr_h3_start + 1;

			populate_all_cdrs();
			all_cdr_fold_tree();
		} // set_defaults

		void
		Antibody::populate_all_cdrs() {

			core::Size begin(0), end(0), size(0), cut(0);

			std::string cdr_name[6] = { "l1", "l2", "l3", "h1", "h2", "h3" };

			core::Size cdr_total( 6 );

			for( core::Size i = 0; i < cdr_total; i++ ) {
				if( cdr_name[i] == "l1" ) {
					begin = cdr_l1_start;
					end = cdr_l1_end;
				}
				else if( cdr_name[i] == "l2" ) {
					begin = cdr_l2_start;
					end = cdr_l2_end;
				}
				else if( cdr_name[i] == "l3" ) {
					begin = cdr_l3_start;
					end = cdr_l3_end;
				}
				else if( cdr_name[i] == "h1" ) {
					begin = cdr_h1_start;
					end = cdr_h1_end;
				}
				else if( cdr_name[i] == "h2" ) {
					begin = cdr_h2_start;
					end = cdr_h2_end;
				}
				else if( cdr_name[i] == "h3" ) {
					begin = cdr_h3_start;
					end = cdr_h3_end + 1; // for the extra stem residue
				}

				//if( flank_relax && cdr_name[i] == "h3" ) {
				//	begin = begin - h3_flank;
				//	end = end + h3_flank;
				//}
				size = ( end - begin ) + 1;
				cut = begin + core::Size( size / 2 );
				if( cdr_name[i] == "h3" )
					cut = cdr_h3_cut_;
				all_cdr_loops.add_loop( begin, end, cut, 0, false);
			}
			all_cdr_loops.sequential_order();
		} // populate_all_cdrs

		void
		Antibody::update_sequence() {
			for( core::Size i = 1; i <= Fv.total_residue(); ++i )
				Fv_sequence_.push_back( Fv.residue(i).name1() );
		}

		void
		Antibody::detect_CDR_H3_stem_type() {

			core::Size cdr_h3_size = ( cdr_h3_end - cdr_h3_start ) + 1;
			bool is_H3( false );

			// extract single letter aa codes for the chopped loop residues
			utility::vector1< char > cdr_h3_sequence;
			for( core::Size ii = cdr_h3_start - 2; ii <= (cdr_h3_start - 2) +
						 cdr_h3_size + 3; ++ii )
				cdr_h3_sequence.push_back( Fv_sequence_[ii] );

			// Rule 1a for standard kink
			if( cdr_h3_sequence[ cdr_h3_sequence.size() - 2 ] != 'D') {
				kinked_ = true;
				is_H3 = true;
			}

			// Rule 1b for standard extended form
			if( ( cdr_h3_sequence[ cdr_h3_sequence.size() - 2 ] == 'D')
					&& ( (cdr_h3_sequence[2] != 'K') &&
							 (cdr_h3_sequence[2] != 'R') ) && (is_H3 != true)) {
				extended_ = true;
				is_H3 = true;
			}

			if( !is_H3 ) {
				// Rule 1b extension for special kinked form
				bool is_basic( false ); // Special basic residue exception flag
				for(core::Size ii = 3; ii <= core::Size(cdr_h3_sequence.size() - 4);
						ii++) {
					if( cdr_h3_sequence[ii] == 'R' || cdr_h3_sequence[ii] == 'K') {
						is_basic = true;
						break;
					}
				}

				if( !is_basic ) {
					core::Size L49_pose_number = Fv.pdb_info()->pdb2pose( 'L', 49 );
					char aa_code_L49 = Fv.residue( L49_pose_number ).name1();
					if( aa_code_L49 == 'R' || aa_code_L49 == 'K')
						is_basic = true;
				}
				if( is_basic ) {
					kinked_ = true;
					is_H3 = true;
				}
			}

			// Rule 1c for kinked form with salt bridge
			if( ( cdr_h3_sequence[ cdr_h3_sequence.size() - 2 ] == 'D') &&
					( (cdr_h3_sequence[2] == 'K') ||
						(cdr_h3_sequence[2] == 'R') ) &&
					( (cdr_h3_sequence[1] != 'K') &&
						(cdr_h3_sequence[1] != 'R') ) && (is_H3 != true) ) {
				kinked_ = true;
				is_H3 = true;
				if( !is_H3 ) {
					bool is_basic( false ); // Special basic residue exception flag
					core::Size L46_pose_number = Fv.pdb_info()->pdb2pose( 'L', 46 );
					char aa_code_L46 = Fv.residue( L46_pose_number ).name1();
					if( aa_code_L46 == 'R' || aa_code_L46 == 'K')
						is_basic = true;
					if( is_basic ) {
						extended_ = true;
						is_H3 = true;
					}
				}
			}

			// Rule 1d for extened form with salt bridge
			if( ( cdr_h3_sequence[ cdr_h3_sequence.size() - 2 ] == 'D') &&
					( ( cdr_h3_sequence[ 2 ] == 'K') ||
						(cdr_h3_sequence[2] == 'R')) &&
					( (cdr_h3_sequence[1] == 'K') ||
						(cdr_h3_sequence[1] == 'R') ) && (is_H3 != true) ) {
				extended_ = true;
				is_H3 = true;
			}
		} // detect_CDR_H3_stem_type()

		void
		Antibody::all_cdr_fold_tree() {
			using namespace core::kinematics;

			all_cdr_loops.sequential_order();

			FoldTree f;
			f.clear();

			core::Size jump_num = 0;
			for( loops::Loops::const_iterator it=all_cdr_loops.begin(),
						 it_end=all_cdr_loops.end(),
						 it_next; it < it_end; ++it ) {

				it_next = it;
				it_next++;

				if( it == all_cdr_loops.begin() )
					f.add_edge( 1, it->start()-1, Edge::PEPTIDE );

				jump_num++;
				f.add_edge( it->start()-1, it->stop()+1, jump_num );
				f.add_edge( it->start()-1, it->cut(),  Edge::PEPTIDE );
				f.add_edge( it->cut()+1, it->stop()+1, Edge::PEPTIDE );
				if( it == (it_end-1) )
					f.add_edge( it->stop()+1, Fv.total_residue(), Edge::PEPTIDE);
				else
					f.add_edge( it->stop()+1, it_next->start()-1, Edge::PEPTIDE );
			}

			f.reorder(1);
			Fv.fold_tree( f );

		} // all_cdr_fold_tree()

		void
		Antibody::align_to_native( antibody::Antibody & native ) {

			core::id::AtomID_Map< core::id::AtomID > atom_map;
			core::id::initialize( atom_map, Fv,	core::id::BOGUS_ATOM_ID );

			for( core::Size res_counter=hfr_start_1, nat_counter=native.hfr_start_1;
					 res_counter <= hfr_end_1; res_counter++, nat_counter++ ) {
				for( core::Size atm_counter=1; atm_counter <= 4; atm_counter++ ) {
					core::id::AtomID const id1( atm_counter, res_counter );
					core::id::AtomID const id2( atm_counter, nat_counter );
					atom_map[ id1 ] = id2;
				}
			}
			for( core::Size res_counter=hfr_start_2, nat_counter=native.hfr_start_2;
					 res_counter <= hfr_end_2; res_counter++, nat_counter++ ) {
				for( core::Size atm_counter=1; atm_counter <= 4; atm_counter++ ) {
					core::id::AtomID const id1( atm_counter, res_counter );
					core::id::AtomID const id2( atm_counter, nat_counter );
					atom_map[ id1 ] = id2;
				}
			}
			for( core::Size res_counter=hfr_start_3, nat_counter=native.hfr_start_3;
					 res_counter <= hfr_end_3; res_counter++, nat_counter++ ) {
				for( core::Size atm_counter=1; atm_counter <= 4; atm_counter++ ) {
					core::id::AtomID const id1( atm_counter, res_counter );
					core::id::AtomID const id2( atm_counter, nat_counter );
					atom_map[ id1 ] = id2;
				}
			}
			for( core::Size res_counter=hfr_start_4, nat_counter=native.hfr_start_4;
					 res_counter <= hfr_end_4; res_counter++, nat_counter++ ) {
				for( core::Size atm_counter=1; atm_counter <= 4; atm_counter++ ) {
					core::id::AtomID const id1( atm_counter, res_counter );
					core::id::AtomID const id2( atm_counter, nat_counter );
					atom_map[ id1 ] = id2;
				}
			}
			for( core::Size res_counter=hfr_start_5, nat_counter=native.hfr_start_5;
					 res_counter <= hfr_end_5; res_counter++, nat_counter++ ) {
				for( core::Size atm_counter=1; atm_counter <= 4; atm_counter++ ) {
					core::id::AtomID const id1( atm_counter, res_counter );
					core::id::AtomID const id2( atm_counter, nat_counter );
					atom_map[ id1 ] = id2;
				}
			}
			for( core::Size res_counter=hfr_start_6 + 1,
						 nat_counter=native.hfr_start_6 + 1; res_counter <= hfr_end_6;
					 res_counter++, nat_counter++ ) {
				for( core::Size atm_counter=1; atm_counter <= 4; atm_counter++ ) {
					core::id::AtomID const id1( atm_counter, res_counter );
					core::id::AtomID const id2( atm_counter, nat_counter );
					atom_map[ id1 ] = id2;
				}
			}

			core::scoring::superimpose_pose( Fv, native.Fv, atom_map );

		} // align_to_native()


	} // namespace antibody
} // namespace protocols

