// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @author Aroop Sircar ( aroopsircar@yahoo.com )

#include <protocols/jobdist/JobDistributors.hh> // SJF Keep first for mpi
#include <core/fragment/FragSet.hh>
#include <core/conformation/Residue.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/SilentStructFactory.hh>
#include <core/options/keys/antibody.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/Energies.hh>
#include <core/pose/util.hh>
#include <core/util/Tracer.hh>

#include <protocols/antibody/AntibodyClass.hh>
#include <protocols/antibody/AntibodyModeler.hh>
//#include <protocols/evaluation/PoseEvaluator.hh>
//#include <protocols/evaluation/RmsdEvaluator.hh>
#include <protocols/loops/loops_main.hh>
#include <protocols/moves/CDRH3Modeler.hh>
#include <protocols/moves/GraftMover.hh>
#include <protocols/moves/MoverContainer.hh>

#include <core/chemical/ChemicalManager.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.AntibodyModeler");

namespace protocols {
	namespace antibody {


		AntibodyModeler::AntibodyModeler() : Mover( "AntibodyModeler" )
		{
			set_default();
		}

		void AntibodyModeler::set_default() {
			using namespace core::options;
			using namespace OptionKeys;
			TR <<  "Setting up default settings" << std::endl;
			model_h3_ = option[ OptionKeys::antibody::model_h3 ]();
			snug_dock_ = option[ OptionKeys::antibody::snug_dock ]();
		}

		void AntibodyModeler::set_snugdock_foldtree( core::pose::Pose & pose_in ){
			TR << "setting up Snug Dock fold tree" << std::endl;
			TR << "Snug Dock Fold Tree: " << std::endl;
			TR << pose_in.fold_tree() << std::endl;
			return;
		}

		void AntibodyModeler::apply( core::pose::Pose & pose_in ){
			using namespace core::scoring;
			using namespace protocols::moves;

			antibody::Antibody antibody_in( pose_in );

			SequenceMoverOP model_antibody ( new SequenceMover() );

			GraftMoverOP graft_move( new GraftMover() );
			model_antibody->add_mover( graft_move );

			CDRH3ModelerOP model_cdrh3( new CDRH3Modeler() );
			model_antibody->add_mover( model_cdrh3 );

			model_antibody->apply( antibody_in.Fv );
			pose_in = antibody_in.Fv;

			/*
			if( snug_dock_ ) {
			set_snug_dock_fold_tree( pose );
			moves::RigidBodyMover dock_mover;
			movemap_ = new MoveMap();
			movemap_->set_true_bb_range( l1_start_, l1_end_ );

			movemap_->set_bb( false );
			for ( i ) {
			movemap_->set_bb( i, true );
			}
			// set movemap to allow:
			// sidechains: CDR & neighbors as well as docked interface
			// backbones: CDRs except H3
			// WE HAVE TO ENSURE THAT DURING MINIMIZATIONS INSIDE DOCKING
			// THE MOVEMAP IS THE ONE SPECIFIED HERE
			dock_mover->( pose );
			}

			// set movemap to allow:
			// sidechains: CDR & neighbors as well as docked interface
			// backbones: only CDR H3
			loop_mover->( pose, h3_start_, h3_end_ );
			}
			*/
		}// end apply

		int
		antibodyjob_main() {
			core::util::Tracer TR("protocols::antibodyjob_main_wrapper");

			using namespace core::options;
			using namespace core::options::OptionKeys;
			using namespace core::scoring;
			using namespace core::scoring::constraints;
			using namespace core::chemical;
			using namespace core::id;
			using namespace jobdist;

			core::pose::Pose start_pose;
			core::chemical::ResidueTypeSetCAP rsd_set;

			rsd_set	= core::chemical::ChemicalManager::get_instance()
                ->residue_type_set( "fa_standard" );
			core::io::pdb::pose_from_pdb( start_pose, *rsd_set,
				option[ OptionKeys::antibody::input_fv ]().name() + ".pdb" );
			core::pose::set_ss_from_phipsi( start_pose );

			core::pose::Pose native_pose;
			if ( option[ in::file::native ].user() ) {
				core::io::pdb::pose_from_pdb(	native_pose,
																			option[ in::file::native ]()	);
				core::pose::set_ss_from_phipsi( native_pose );
			} else
				native_pose = start_pose;

			// fragment initialization
			utility::vector1< core::fragment::FragSetOP > frag_libs;
			if ( option[ OptionKeys::antibody::model_h3 ]() )
				protocols::loops::read_loop_fragments( frag_libs );

			// job distributor initialization
			utility::vector1< BasicJobOP > input_jobs;
			core::Size const nstruct = static_cast< core::Size > (
				std::max( (int) 1, (int)option[ out::nstruct ] ) );
			BasicJobOP job = new BasicJob("S", "lr", nstruct);
			input_jobs.push_back( job );
			BaseJobDistributorOP jobdist;

			// output nonidealized silent file or PDBs?
			bool silent_output;
			if ( option[ OptionKeys::out::file::silent ].user() ) {
				TR.Debug << "Outputting non-idealized silent file\n";
				jobdist = new PlainSilentFileJobDistributor< BasicJobOP >(input_jobs );
				silent_output = true;
			} else {
				TR.Debug << "Outputting PDBs" << std::endl;
				jobdist = new PlainPdbJobDistributor< BasicJobOP >( input_jobs );
				silent_output = false;
			}

			BasicJobOP curr_job;
			int curr_nstruct;
			jobdist->startup();

			antibody::Antibody antibody_in( start_pose );

			// Read in CDR H3 C-terminal fragment file
			utility::vector1< core::fragment::FragData > H3_base_library;
			protocols::moves::read_H3_cter_fragment( antibody_in, H3_base_library );

			antibody::Antibody antibody_start = antibody_in;

			while ( jobdist->next_job(curr_job, curr_nstruct) ) { // loop over jobs
				using namespace protocols::moves;

				std::string curr_job_tag = curr_job->output_tag( curr_nstruct );
				antibody_in = antibody_start;

				SequenceMoverOP model_antibody ( new SequenceMover() );

				GraftMoverOP graft_move( new GraftMover() );
				graft_move->set_native_pose( new core::pose::Pose ( native_pose ) );
				model_antibody->add_mover( graft_move );

				if ( option[ OptionKeys::antibody::model_h3 ]() ) {
					CDRH3ModelerOP model_cdrh3( new CDRH3Modeler() );
					model_cdrh3->store_H3_cter_fragment( H3_base_library );
					model_cdrh3->store_frags( frag_libs );
					model_cdrh3->set_native_pose( new core::pose::Pose ( native_pose ) );
					model_antibody->add_mover( model_cdrh3 );
				}

				model_antibody->apply( antibody_in.Fv );

				core::pose::Pose pose_out;
				pose_out = antibody_in.Fv;

				///////////////////////////////////////////////////////////////////////
				////
				////   Filter
				////

				//float final_score( 0.0 );
				//using std::string;
				//using core::pose::getPoseExtraScores;
				//if ( getPoseExtraScores( pose, string("final_looprelax_score"),
				//												 final_score ) ) {
				//	if ( ( option[ OptionKeys::loops::final_score_filter ].user() ) &&
				//			 final_score>option[ OptionKeys::loops::final_score_filter ]() ){
				//		TR.Debug <<  "FailedFilter " << final_score << " > "
				//						 << option[  OptionKeys::loops::final_score_filter ]()
				//						 << std::endl;
				//		continue;
				//	}
				//}

				///////////////////////////////////////////////////////////////////////
				////
				////   Output
				////

				if ( silent_output ) {
					PlainSilentFileJobDistributor< BasicJobOP > *jd =
						dynamic_cast< PlainSilentFileJobDistributor< BasicJobOP > * >
						( jobdist() );

					std::string silent_struct_type( "binary" );  // default to binary
					if ( option[ out::file::silent_struct_type ].user() ) {
						silent_struct_type = option[ out::file::silent_struct_type ]();
					}

					using namespace core::io::silent;
					SilentStructOP ss = SilentStructFactory::get_silent_struct(
						silent_struct_type );

					ss->fill_struct( pose_out, curr_job_tag );

					// run PoseEvaluators
					//evaluator->apply( pose, curr_job_tag, *ss );

					jd->dump_silent( curr_nstruct, *ss );
				} else {
					using namespace core::pose;
					std::string outfile(
															option[ OptionKeys::out::path::path ]().name() +
															"/" +	option[ OptionKeys::out::prefix ] +
															"_" + right_string_of(curr_nstruct,4,'0') +
															".pdb" );

					std::ofstream out( outfile.c_str(), std::ios::out |
														 std::ios::binary );
					core::io::pdb::dump_pdb( pose_out, out );
					float final_looprms=0.0;
					float final_score=0.0;
					//float final_chainbreak=0.0;
					getPoseExtraScores( pose_out, "looprms", final_looprms );
					getPoseExtraScores( pose_out, "final_looprelax_score", final_score );
					//getPoseExtraScores( pose, "final_chainbreak", final_chainbreak );
					out << "loop_rms: " << final_looprms << std::endl;
					out << "total_energy: " << final_score << std::endl;
					//out << "chainbreak: " << final_chainbreak << std::endl;
					// mirror to tracer
					TR << "loop_rms: " << final_looprms << std::endl;
					TR << "total_energy: " << final_score << std::endl;
					//TR << "chainbreak: " << final_chainbreak << std::endl;

					// make sure buffer is flushed before attempting to gzip
					out.flush();

					if (option[ OptionKeys::out::pdb_gz ]()) {
						utility::file::gzip( outfile, true );
					}

				}
			} // loop over jobs
			jobdist->shutdown();
			return 0;
		} // antibodyjob_main

	} // end antibody
} // end protocols

