// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
///
/// @author Mike Tyka


#include <core/chemical/ChemicalManager.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/io/silent/ProteinSilentStruct.hh>
#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/silent.fwd.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/options/after_opts.hh>
#include <core/options/util.hh>
#include <core/pose/util.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/ScoreFunction.hh>
#include <ObjexxFCL/string.functions.hh>
#include <protocols/cluster/cluster.hh>
#include <protocols/evaluation/RmsdEvaluator.hh>
#include <protocols/jobdist/standard_mains.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/hotspot_hashing/HotspotStubSet.hh>
#include <utility/file/FileName.hh>
#include <core/scoring/constraints/ConstraintIO.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/util.hh>
#include <protocols/loops/loops_main.hh>

// C++ headers
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <string>
#include <deque>

// option key includes

#include <core/options/keys/cluster.OptionKeys.gen.hh>

namespace protocols {
namespace cluster {

using namespace core;
using namespace ObjexxFCL;
using namespace core::pose;
using namespace core::options;
using namespace evaluation;


Real CA_rmsd_with_exclude_res(
	const core::pose::Pose & pose1, const core::pose::Pose & pose2
) {
	using namespace core::options;
	using namespace core::options::OptionKeys;

	if ( option[ core::options::OptionKeys::cluster::exclude_res ].user() ) {
		ResidueSelection residues;
		invert_exclude_residues( pose1.total_residue(), option[ core::options::OptionKeys::cluster::exclude_res ](), residues );
		if ( pose1.residue(1).is_RNA() ) utility_exit_with_message( "Hey put in all atom rmsd code for residue subset!\n" ) ;
		return core::scoring::CA_rmsd( pose1, pose2, residues );
	} else {
		// no residues excluded from the native.
		if ( pose1.residue(1).is_RNA() ) return core::scoring::all_atom_rmsd( pose1, pose2 );
		return core::scoring::CA_rmsd( pose1, pose2 );
	}
} // native_CA_rmsd

bool compareIndexEnergyPair(
	const std::pair< int, float > & p1, const std::pair< int, float > & p2 )
{
	return p1.second  < p2.second;
}

GatherPosesMover::GatherPosesMover() : Mover()
{
	filter_ = 1000000.0;
	processed_ = 0;
}

void GatherPosesMover::set_score_function(  core::scoring::ScoreFunctionOP sfxn ) {
	sfxn_ = sfxn;
}

void GatherPosesMover::set_filter( float filter ) {
	filter_ = filter;
}

void GatherPosesMover::apply( core::pose::Pose & pose ) {

	float score;
	// does pose already have score set ?
	if ( !getPoseExtraScores( pose, "silent_score", score ) ) {
		// if not do we have scoring function ?

		if ( !sfxn_ ) {
			//oh well - can't do enything ... just set it to 0.
		} else {
			// if so - try and score the structure and set that energy column
			////////////////// add constraints if specified by user.
			core::scoring::constraints::add_constraints_from_cmdline( pose, *sfxn_ );
			core::Real score = (*sfxn_)(pose)	;
			std::cout << "RESCORING!" << std::endl;
			setPoseExtraScores( pose, "silent_score", score );

			if ( get_native_pose() ) {
				setPoseExtraScores(
					pose, "rms", CA_rmsd_with_exclude_res( pose, *get_native_pose() )
				);
			}
			pose.energies().clear();
		}

		// remember tag!
		tag_list.push_back( protocols::jobdist::extract_tag_from_pose( pose ) );
	}

	// filter structures if the filter is active
	if ( score > filter_ ) return; // ignore pose if filter value is too high

	// now save the structure.
	poselist.push_back( pose );
	processed_ ++;
	//std::cout << "Read " << poselist.size() << " structures" << std::endl;
}

bool GatherPosesMover::check_tag( const std::string &query_tag ) {
	using std::find;
	if ( find( tag_list.begin(), tag_list.end(), query_tag ) == tag_list.end() ) {
		return false;
	}
	return true;
}

ClusterBase::ClusterBase()
	: GatherPosesMover()
{}

void ClusterBase::set_cluster_radius( float cluster_radius ) {
	cluster_radius_ = cluster_radius;
}

float ClusterBase::get_cluster_radius() {
	return cluster_radius_;
}

float ClusterBase::get_distance_measure( int index1, int index2 ) {
	// core::Real dist = core::scoring::xyz_gdtmm( p1a, p2a );
	if ( option[ core::options::OptionKeys::cluster::hotspot_hash ]() ) {
		core::Size const resnum1 = poselist[index1].total_residue();
		core::Size const resnum2 = poselist[index2].total_residue();
		core::conformation::ResidueCOP res1( poselist[index1].residue(resnum1) );
		core::conformation::ResidueCOP res2( poselist[index2].residue(resnum2) );
		return protocols::hotspot_hashing::residue_sc_rmsd_no_super( res1, res2 );
	}
	else return CA_rmsd_with_exclude_res( poselist[index1], poselist[index2] );
}

void ClusterBase::calculate_distance_matrix() {
	using core::Real;
	FArray2D< Real > p1a, p2a;
	distance_matrix = FArray2D< Real > ( poselist.size(), poselist.size(), 0.0 );
	int count = 0;
	std::cout << "Calculating RMS matrix: " << std::endl;

	float hist_resolution = 0.25;
	int   hist_size = int(20.0 / hist_resolution);

	std::vector<int> histcount(hist_size,0);

	for ( Size i = 0; i < poselist.size(); i++ ) {
		for ( Size j = i; j < poselist.size(); j++ ) {
			// only do comparisons once
			if ( i < j ) {
				// get the similarity between structure with index i and with index j
				core::Real dist  = get_distance_measure(i,j);
				distance_matrix( i+1, j+1 ) = dist;
				distance_matrix( j+1, i+1 ) = dist;

				int histbin = int(dist/hist_resolution);
				if ( histbin < hist_size ) histcount[histbin]+=1;

				// print some stats of progress
				count ++;
				if ( count % 5000 == 0 ) {
					Real const percent_done ( static_cast< Real > (
						count / (poselist.size() - 1) * poselist.size()
					) );
					std::cout << count
						<< "/" << ( poselist.size() - 1 )*( poselist.size() )
						<< " ( " << percent_done << "% )"
						<< std::endl;
				}
			}
		} // for it2
	} // for it1


	std::cout << "Histogram of pairwise similarity values for the initial clustering set" << std::endl;
	for ( Size i = 0; i <(Size)hist_size; i++ ) {
		std::cout << "hist " <<  float(i)*hist_resolution << "   " << histcount[i] << std::endl;
	}
} // calculate_distance_matrix

// PostProcessing ---------------------------------------------------------
void ClusterBase::add_structure( core::pose::Pose & pose ) {

	poselist.push_back( pose );
	int nexindex = poselist.size() - 1;
	int lowrmsi=-1;
	float lowrms=1000000.0;
	for (int m=0;m<(int)clusterlist.size();m++ ) {
		float rms;
		rms =   get_distance_measure( nexindex,
				clusterlist[m][0] ); // clustercentre m
		if (rms < lowrms) {
			lowrms  = rms;
			lowrmsi = m;
		}
	}


	if (lowrmsi >= 0) {
		if ( lowrms < get_cluster_radius() ) {
			// if within our radius - then add to cluster
			clusterlist[lowrmsi].push_back(nexindex);
		}else{
			// else make a new cluster !!
			std::deque< int > newlist;
			newlist.push_back(  nexindex );
			clusterlist.push_back( newlist );
		}
	}
}

// PostProcessing ---------------------------------------------------------

void ClusterBase::sort_each_group_by_energy() {
	int i,j;
	for (i=0;i<(int)clusterlist.size();i++ ) {

		std::vector< std::pair< int, float > > cluster_energies;
		for (j=0;j<(int)clusterlist[i].size();j++ ) {
			float score=0.0;
			if ( !getPoseExtraScores( poselist[ clusterlist[i][j] ], "silent_score", score ) ) {
				std::cerr << "Warning: no score available for " << std::endl;
			}

			cluster_energies.push_back( std::pair< int, float > ( clusterlist[i][j], score ) );
		}
		// dont sort in the first member (the cluster center!!)
		std::sort( cluster_energies.begin()+1, cluster_energies.end(), compareIndexEnergyPair );

		clusterlist[i].clear();

		for (j=0;j<(int)cluster_energies.size();j++ ) clusterlist[i].push_back( cluster_energies[j].first );

	}

}

// PostProcessing ---------------------------------------------------------

void ClusterBase::remove_highest_energy_member_of_each_group() {
	int i,j;
	for (i=0;i<(int)clusterlist.size();i++ ) {
		if ( clusterlist[i].size() <= 1 ) continue;

		// sort group by energy
		std::vector< std::pair< int, float > > cluster_energies;
		for (j=0;j<(int)clusterlist[i].size();j++ ) {
			float score=0.0;
			if ( !getPoseExtraScores( poselist[ clusterlist[i][j] ], "silent_score", score ) ) {
				std::cerr << "Warning: no score available for " << std::endl;
			}

			cluster_energies.push_back( std::pair< int, float > ( clusterlist[i][j], score ) );
		}
		std::sort( cluster_energies.begin(), cluster_energies.end(), compareIndexEnergyPair );
		clusterlist[i].clear();
		for (j=0;j<(int)(cluster_energies.size()-1);j++ ) clusterlist[i].push_back( cluster_energies[j].first );
	}

}

void ClusterBase::sort_groups_by_energy() {
	int i;
	float populationfactor = 0.1; // conversion factor from score to population.
	std::vector < std::deque< int > >   temp = clusterlist;

	std::vector< std::pair< int, float > > cluster_energies;
	for (i=0;i<(int)clusterlist.size();i++ ) {
		float score = -populationfactor * clusterlist[i].size() ;
		// the cluster's energy is determined by the lowest energy member (usually the second member
		// after the groups were already clustered individually. FOr singleton groups just use the clutser
		// center (0)
		int energy_member = 0;
		if ( clusterlist[i].size() > 1 ) 	energy_member = 1;

		if ( !getPoseExtraScores( poselist[ clusterlist[i][energy_member] ], "silent_score", score ) ) {
			std::cerr << "Warning: no score available for " << std::endl;
		}

		cluster_energies.push_back( std::pair< int, float > ( i, score ) );
	}

	std::sort( cluster_energies.begin(), cluster_energies.end(), compareIndexEnergyPair );

	// clear the cluster list
	clusterlist.clear();

	// now put back every cluster in the order of energies
	for (i=0;i<(int)cluster_energies.size();i++ ) clusterlist.push_back( temp[cluster_energies[i].first ] );

}

void ClusterBase::remove_singletons() {
	int i;
	std::vector < std::deque< int > >   clusterlist_copy = clusterlist;
	clusterlist.clear();

	std::vector< std::pair< int, float > > cluster_energies;
	for (i=0;i<(int)clusterlist_copy.size();i++ ) {
		if ( clusterlist_copy[i].size() > 1 ) {
			clusterlist.push_back( clusterlist_copy[i] );
		}
	}

	//In case nothing is left, better at least return one cluster?
	if ( clusterlist.size() < 1 ) {
		clusterlist.push_back( clusterlist_copy[ 0 ] );
	}


}

// take first 'limit' from each cluster
void ClusterBase::limit_groupsize( int limit ) {
	int i,j;
	if ( limit < 0 ) return;
	for (i=0;i<(int)clusterlist.size();i++ ) {
		std::deque < int > temp = clusterlist[i];
		clusterlist[i].clear();
		for (j=0;(j<(int)temp.size()) && (j<limit);j++ ) {
			clusterlist[i].push_back( temp[j] );
		}
		for (j=limit;(j<(int)temp.size());j++ ) {
			float score=0.0;
			getPoseExtraScores(                        poselist[ temp[j] ] , "silent_score", score );
			std::cout << "limit_groupsize: removes: "
				<< protocols::jobdist::extract_tag_from_pose( poselist[ temp[j] ] )
				<< "  " << score
				<< std::endl;
		}
	}
}

// take first 'limit' clusters
void ClusterBase::limit_groups( int limit ) {
	return;  // TEMP - just a test !
	int i;
	if ( limit < 0 ) return;
	std::vector < std::deque< int > >   temp = clusterlist;
	clusterlist.clear();
	for (i=0;i<(int)limit;i++ ) {
		clusterlist.push_back( temp[i] );
	}
}

// take first 'limit' total_structures
void ClusterBase::limit_total_structures( int limit) {
	int i,j;
	if ( limit < 0 ) return;
	std::vector < std::deque< int > >   temp = clusterlist;
	clusterlist.clear();
	int count=0;
	for (i=0;i<(int)temp.size();i++ ) {
		std::deque< int > newcluster;
		for (j=0;j<(int)temp[i].size();j++ ) {
			if ( count < limit ) {
				newcluster.push_back( temp[i][j] );
			}else{
				float score=0.0;
				getPoseExtraScores(                        poselist[ temp[i][j] ] , "silent_score", score );
				std::cout << "limit_total_structures: removes: "
					<< protocols::jobdist::extract_tag_from_pose( poselist[ temp[i][j] ]  )
					<< "  " << score
					<< std::endl;
			}
			count ++;
		}
		if ( newcluster.size() > 0 )
			clusterlist.push_back( newcluster );

	}
}


// take first 'limit' total_structures
void ClusterBase::clean_pose_store() {
	// for each structure in the pose list
	for ( Size index=0; index < poselist.size(); index ++ ) {
		bool ispresent = false;
		// try and find it in the cluster assigments
		Size i,j;
		for (i=0;i<clusterlist.size();i++ ) {
			for (j=0;j<clusterlist[i].size();j++ ) {
				if ( (int)index == clusterlist[i][j] ) { ispresent = true; break; }
			}
			if ( ispresent) break;
		}
		if ( !ispresent) {
			// zap the pose (but retain a DUMMY POSE so that the indexing stays the smae !!! )
			poselist[index] = core::pose::Pose() ;
		}
	}
}


//  ------ OUTPUT -----------------------------------------------------------------------

void ClusterBase::print_summary() {
	std::cout << "---------- Summary ---------------------------------" << std::endl;
	int i,j;

	int count=0;

	for (i=0;i<(int)clusterlist.size();i++ ) {
		std::cout << "Cluster:  " << i << "  N: " << (int)clusterlist[i].size() << "   c."<<i<<".*.pdb " ;
		std::cout << std::endl;
		count += clusterlist[i].size();
		for (j=0;j<(int)clusterlist[i].size();j++ ) {
			std::cout << "    ";
			std::cout << protocols::jobdist::extract_tag_from_pose( poselist[ clusterlist[i][j] ]  ) << "  " ;
			float score = 0.0;
			if ( !getPoseExtraScores( poselist[ clusterlist[i][j] ], "silent_score", score ) ) {
				std::cout << "----" ;
			}else{
				std::cout << score ;
			}

			float rms;
			if ( getPoseExtraScores( poselist[ clusterlist[i][j] ], "rms", rms ) ) {
				std::cout << " " << rms << std::endl;
			}
			//if ( get_native_pose() )  std::cout << " " << CA_rmsd_with_exclude_res( *get_native_pose(), poselist[ clusterlist[i][j] ]  );

			std::cout << std::endl;
		}
	}
	std::cout << "----------------------------------------------------" << std::endl;
	std::cout << "  Clusters: " << clusterlist.size() << std::endl;
	std::cout << "  Structures: " << count << std::endl;
	std::cout << "----------------------------------------------------" << std::endl;
}

void ClusterBase::print_raw_numbers() {
	int i,j;
	for (i=0;i<(int)clusterlist.size();i++ ) {
		std::cout << i << " : ";
		for (j=0;j<(int)clusterlist[i].size();j++ ) {
			std::cout << clusterlist[i][j] << "  ";
		}
		std::cout << std::endl;
	}
}

void ClusterBase::print_cluster_assignement() {
	int i,j;
	for (i=0;i<(int)clusterlist.size();i++ ) {
		for (j=0;j<(int)clusterlist[i].size();j++ ) {
			std::cout << protocols::jobdist::extract_tag_from_pose( poselist[ clusterlist[i][j] ]  ) << "  " << i << "  " << j << std::endl;
		}
	}
}

void ClusterBase::print_cluster_PDBs( std::string prefix ) {
	int i,j;
	for ( i=0;i<(int)clusterlist.size();i++ ) {
		std::vector< int > newlist;
		for ( j=0;j<(int)clusterlist[i].size();j++ ) {
			//std::string output_name = "c." + string_of( i ) + "." + string_of( j ) + "." +
			//                         protocols::jobdist::extract_tag_from_pose( poselist[ clusterlist[i][j] ] ) + ".pdb";
			std::string output_name = "c." + string_of( i ) + "." + string_of( j ) + "." + "pdb";
			replace_in(output_name, '/', "_" );
			poselist[ clusterlist[i][j] ].dump_pdb( prefix + output_name );
		}
	}
}

void ClusterBase::create_constraints(
	std::string prefix,
	EnsembleConstraints &constraint_maker)
{
	int i,j;
	std::cout << "Making constraints .. " << std::endl;
	for (i=0;i<(int)clusterlist.size();i++ ) {
		constraint_maker.clear();
		for (j=0;j<(int)clusterlist[i].size();j++ ) {
			constraint_maker.push_back(  poselist[ clusterlist[i][j] ]  );
		}

		std::string output_name = "c." + string_of( i ) + "." + "cst";
		replace_in(output_name, '/', "_" );
		std::ofstream outf(  std::string(prefix + output_name).c_str() );
		constraint_maker.createConstraints( outf );
		outf.close();
	}

}




















ClusterPhilStyle::ClusterPhilStyle() : ClusterBase()
{

}

void ClusterPhilStyle::do_clustering() {

	using namespace core::options;
	using namespace core::options::OptionKeys;

	int listsize = poselist.size();

	std::cout << "Clustering " << listsize << " structures " << std::endl;
	calculate_distance_matrix();

	int i,j;
	std::vector < int > neighbors ( poselist.size(), 0 );
	std::vector < int > clusternr ( poselist.size(), -1 );
	std::vector < int > clustercenter;
	int    mostneighbors;
	int    nclusters=0;

	std::cout << "Clustering with radius " <<  get_cluster_radius() << std::endl;

	std::vector <int> clustercentre;

	std::cout << "Assigning initial cluster centres " << std::endl;
	// now assign groupings
	while(true) {
		// count each's neighbors
		for (i=0;i<listsize;i++ ) {
			neighbors[i] = 0;
			if (clusternr[i]>=0) continue; // ignore ones already taken
			for (j=0;j<listsize;j++ ) {
				if (clusternr[j]>=0) continue; // ignore ones already taken
				if ( distance_matrix( i+1, j+1 ) < get_cluster_radius()) neighbors[i]++;
			}
		}

		mostneighbors = 0;
		for (i=0;i<listsize;i++ ) {
			if (neighbors[i]>neighbors[mostneighbors]) mostneighbors=i;
		}
		if (neighbors[mostneighbors] <= 0) break;  // finished!


		for (i=0;i<listsize;i++ ) {
			if (clusternr[i]>=0) continue; // ignore ones already taken
			if ( distance_matrix( i+1, mostneighbors+1 ) < get_cluster_radius()) {
				clusternr[i] = mostneighbors;
			}
		}

		clustercentre.push_back(mostneighbors);
		nclusters++;

		if (nclusters > 100) break;  // currently fixed but ought to be a paraemter
		if ((nclusters%10)==0) std::cout << ".";
		std::cout.flush();
	}
	std::cout << std::endl;


	for (i=0;i<(int)clustercentre.size();i++ ) {
		std::deque< int > newlist;
		for (j=0;j<listsize;j++ ) {
			// if that struture belongs to a given cluster
			if (clusternr[j] == clustercentre[i]) {
				//add it
				//newlist.push_back(j);
				if (j!= clustercentre[i]) newlist.push_back(j);         // add structure
				else                     newlist.push_front(j);        // add cluster centre at the front
			}
		}

		// add the new list to the clusterlist
		clusterlist.push_back(newlist);
	}


}

// this ensures every structure is in the cluster to who's cluster center it is most similar too
void ClusterPhilStyle::do_redistribution() {
	using namespace core::options;
	using namespace core::options::OptionKeys;

	int i,j;

	std::cout << "Redistributing groups ...\n";

	// redistribute groups - i.e. take each structure, calculate the rms to each cluster centre.
	for (i=0;i<(int)clusterlist.size();i++ ) {
		for (j=1;j<(int)clusterlist[i].size();j++ ) {
			int lowrmsi=i;
			float lowrms=10000.0;
			for (int m=0;m<(int)clusterlist.size();m++ ) {
				float rms;
				rms = distance_matrix( clusterlist[i][j]+1,  // current structure
						clusterlist[m][0]+1); // clustercentre m
				//std::cout << "C: " << i << " S: " << j << " rms " << m << " --> " << rms << " " << std::endl;

				if (rms < lowrms) {
					lowrms  = rms;
					lowrmsi = m;
				}
			}
			if (lowrmsi != i) { // is a different cluster centre closer to us than our current centre ?
				std::cout << "Switched " << lowrmsi << "<--" << i << std::endl;
				// switch over;
				clusterlist[lowrmsi].push_back(clusterlist[i][j]);
				std::deque< int >::iterator it = clusterlist[i].begin();
				it += j;
				clusterlist[i].erase(it);
			}
		}

	}



}





float ClusterPhilStyle_Loop::get_distance_measure(int index1, int index2) {
	float rms = protocols::loops::loop_rmsd_with_superimpose(poselist[index1], poselist[index2] , loop_def_, false );
	return rms;
}








































AssignToClustersMover::AssignToClustersMover( ClusterBaseOP cluster_base): GatherPosesMover()
{
	cluster_base_ = cluster_base;
	processed_ = 0;
}

void AssignToClustersMover::apply( core::pose::Pose & pose ) {
	using namespace core::options;
	using namespace core::options::OptionKeys;
	poselist.clear();
	GatherPosesMover::apply( pose );

	// check we hvnt added this structure before !
	if ( cluster_base_->check_tag( protocols::jobdist::extract_tag_from_pose( pose ) ) ) {
		std::cout << "Already added: " << protocols::jobdist::extract_tag_from_pose( pose )  << std::endl;
		return;
	}

	// find the best group by looping round all the existing groups and comparing
	// protected:
	cluster_base_->add_structure( pose );
	processed_ ++;


	// to prevent memory crazyness, periodically limit back to the number of
	// desired structures!  giving preference to low energy clusters
	static int count=0;

	float score;
	getPoseExtraScores(                        pose , "silent_score", score );
	std::cout << "Adding a "
		<< protocols::jobdist::extract_tag_from_pose( pose )
		<< "  " << score
		<< std::endl;

	if (( count % 50 ) == 0 ) {
		cluster_base_->print_summary();
	}

	if ( option[ core::options::OptionKeys::cluster::limit_total_structures] >= 0 ) {
		if (( (count+1) % ( std::max(50, int( option[ core::options::OptionKeys::cluster::limit_total_structures] / 10.0 )) ) )==0) {
			cluster_base_->sort_each_group_by_energy();
			cluster_base_->sort_groups_by_energy();
			cluster_base_->limit_groupsize( option[ core::options::OptionKeys::cluster::limit_cluster_size] );
			cluster_base_->limit_groups( option[core::options::OptionKeys::cluster::limit_clusters] );
			cluster_base_->limit_total_structures( option[ core::options::OptionKeys::cluster::limit_total_structures] );
			cluster_base_->clean_pose_store();

			cluster_base_->print_summary();
		}

	}
	count++;
}

void EnsembleConstraints_Simple::createConstraints( std::ostream &out) {
	int nres = poselist[0].total_residue();
	int residuesep = 5;
	float strength = 1.0;

	out << "[ atompairs ]" << std::endl;

	for (int ir = 1; ir <= nres; ir ++ ) {
		for (int jr = 1; jr <= nres; jr ++ ) {
			if ( ir >= (jr - residuesep) ) continue;

			float lowdist=1000000.0;
			float highdist=-100000.0;
			for ( Size i=0; i<poselist.size(); i++ ) {
				Vector ir_CA = poselist[i].residue(ir).xyz( "CA" );
				Vector jr_CA = poselist[i].residue(jr).xyz( "CA" );
				float dist = ir_CA.distance( jr_CA );
				if ( dist < lowdist ) lowdist = dist;
				if ( dist > highdist)highdist = dist;
			}

			if ( lowdist > 11.0 ) continue;
			if ( highdist > 11.0 ) continue;

			if ( ( highdist -  lowdist ) < minimum_width_ ) {
				highdist += 0.5 * minimum_width_;
				lowdist  -= 0.5 * minimum_width_;
			}

			out << "     CA" << right_string_of(ir,7,' ')
					<< "     CA" << right_string_of(jr,7,' ')
					<< " BOUNDED"
				 << F( 12, 3, lowdist)
				 << F( 12, 3, highdist)
				 << F(4,1,strength )
				 << "  na"
				 << "; " <<  F( 12, 3, highdist - lowdist)
				 << std::endl;
		}
	}
}

} // cluster
} // protocols
