// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
///
/// @author Mike Tyka

#ifndef INCLUDED_protocols_cluster_cluster_HH
#define INCLUDED_protocols_cluster_cluster_HH

#include <core/pose/Pose.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/loops/LoopClass.hh>

// ObjexxFCL headers
#include <ObjexxFCL/FArray2D.hh>

// C++ headers
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <string>
#include <deque>

namespace protocols {
namespace cluster {

class GatherPosesMover;
typedef utility::pointer::owning_ptr< GatherPosesMover > GatherPosesMoverOP;
typedef utility::pointer::owning_ptr< GatherPosesMover const > GatherPosesMoverCOP;

class GatherPosesMover: public moves::Mover {
	public:
		GatherPosesMover();

		void set_score_function( core::scoring::ScoreFunctionOP sfxn );
		void set_filter( float filter );

		virtual void apply( core::pose::Pose & pose );

		bool check_tag( const std::string &query_tag );

		void push_back( const core::pose::Pose & pose ) {
			poselist.push_back( pose );
		}

		void clear() {
			poselist.clear();
		}

		int processed() const { return processed_; }

	protected:
		std::vector< core::pose::Pose > poselist;

	private:
		core::scoring::ScoreFunctionOP sfxn_;
		core::Real filter_;
		std::vector< std::string > tag_list;

		int processed_;
};



// Main base class for making constraints out of groups of structures

class EnsembleConstraints;
typedef utility::pointer::owning_ptr< EnsembleConstraints > EnsembleConstraintsOP;
typedef utility::pointer::owning_ptr< EnsembleConstraints const > EnsembleConstraintsCOP;

class EnsembleConstraints: public protocols::cluster::GatherPosesMover {
public:
	EnsembleConstraints(): GatherPosesMover() {};
#ifndef BOINC // gives windows build error
	//EnsembleConstraints* clone() const = 0;
#endif
	virtual void createConstraints( std::ostream &out ) = 0;
};



// A super simple implementation of the above - jsut create bounds for close CA atoms.
class EnsembleConstraints_Simple;
typedef utility::pointer::owning_ptr< EnsembleConstraints_Simple > EnsembleConstraints_SimpleOP;
typedef utility::pointer::owning_ptr< EnsembleConstraints_Simple const > EnsembleConstraints_SimpleCOP;

class EnsembleConstraints_Simple: public protocols::cluster::EnsembleConstraints {
public:
	EnsembleConstraints_Simple( float minimum_width = 0): EnsembleConstraints() {
		minimum_width_ = minimum_width;
	};
#ifndef BOINC // gives windows build error
	protocols::moves::MoverOP clone() const { return new EnsembleConstraints_Simple( *this ) ; }
#endif
	virtual void createConstraints( std::ostream &out);

protected:
	float minimum_width_;
};




class ClusterBase;
typedef utility::pointer::owning_ptr< ClusterBase > ClusterBaseOP;
typedef utility::pointer::owning_ptr< ClusterBase const > ClusterBaseCOP;

class ClusterBase: public GatherPosesMover {
	public:
		ClusterBase();
		void set_cluster_radius( float cluster_radius );

		float get_cluster_radius();

		virtual float get_distance_measure(int index1, int index2);

		void calculate_distance_matrix();

		void add_structure( core::pose::Pose & pose );


		void sort_each_group_by_energy();

		// PostProcessing ---------------------------------------------------------

		void remove_highest_energy_member_of_each_group();
		void sort_groups_by_energy();
		void remove_singletons();
		void limit_groupsize( int limit = -1);
		void limit_groups( int limit = -1);
		void limit_total_structures( int limit = -1);
		void clean_pose_store();

		// Printing --------------------------------------------------------------

		void print_summary();
		void print_raw_numbers();
		void print_cluster_assignement();
		void print_cluster_PDBs( std::string prefix );

		void create_constraints( std::string prefix, EnsembleConstraints &constraint_maker);

	protected:
		ObjexxFCL::FArray2D< core::Real >    distance_matrix;
		std::vector < std::deque< int > >    clusterlist;

		float cluster_radius_;
};


class ClusterPhilStyle;
typedef utility::pointer::owning_ptr< ClusterPhilStyle > ClusterPhilStyleOP;
typedef utility::pointer::owning_ptr< ClusterPhilStyle const > ClusterPhilStyleCOP;

class ClusterPhilStyle: public ClusterBase {
public:
	ClusterPhilStyle();
	virtual ~ClusterPhilStyle() {};
	protocols::moves::MoverOP clone() const { return new ClusterPhilStyle( *this ) ; }
	virtual void do_clustering();

	// this ensures every structure is in the cluster to who's cluster center it is most similar too
	void do_redistribution();
};

class ClusterPhilStyle_Loop: public ClusterPhilStyle {
public:
	ClusterPhilStyle_Loop( protocols::loops::Loops loop_def )
		: loop_def_(loop_def)
	{}

	virtual ~ClusterPhilStyle_Loop() {}
	protocols::moves::MoverOP clone() const {
		return new ClusterPhilStyle_Loop( *this );
	}

	virtual float get_distance_measure( int index1, int index2 );

private:
	protocols::loops::Loops loop_def_;
};

class AssignToClustersMover;
typedef utility::pointer::owning_ptr< AssignToClustersMover > AssignToClustersMoverOP;
typedef utility::pointer::owning_ptr< AssignToClustersMover const > AssignToClustersMoverCOP;

class AssignToClustersMover: public GatherPosesMover {
public:
	AssignToClustersMover( ClusterBaseOP cluster_base );
#ifndef BOINC // gives windows build error
	protocols::moves::MoverOP clone() const {
		return new AssignToClustersMover( *this );
	}
#endif
	virtual void apply( core::pose::Pose & pose );

	int processed() const { return processed_; }

protected:
	ClusterBaseOP cluster_base_;
	int processed_;
};

} //namespace cluster
} //namespace protocols

#endif
