// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/protocols/comparative_modeling/ConstraintRemodelMover.hh
/// @brief
/// @author Liz Kellogg ekellogg@u.washington.edu
/// @author James Thompson

// libRosetta headers


#include <core/types.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/constraints/ConstraintIO.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/ConstraintSet.fwd.hh>

#include <core/pose/Pose.fwd.hh>

#include <core/options/option.hh>
#include <core/options/after_opts.hh>

#include <core/util/Tracer.hh>

#include <core/pack/pack_rotamers.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>

#include <core/kinematics/MoveMap.hh>
#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <core/io/silent/silent.fwd.hh>
#include <core/io/silent/ProteinSilentStruct.hh>
#include <core/io/silent/SilentFileData.hh>

#include <protocols/moves/Mover.hh>
#include <protocols/loops/LoopBuild.hh>
#include <protocols/comparative_modeling/ConstraintRemodelMover.hh>

#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/SilentStruct.hh>
#include <core/io/silent/SilentStructFactory.hh>
#include <core/io/silent/silent.fwd.hh>
#include <core/io/silent/SilentStructFactory.hh>

#include <utility/vector1.hh>
#include <utility/io/ozstream.hh>

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Warning;
using core::util::Error;

// C++ headers
#include <fstream>
#include <iostream>
#include <string>


// option key includes

#include <core/options/keys/loops.OptionKeys.gen.hh>
#include <core/options/keys/james.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>
#include <core/options/keys/constraints.OptionKeys.gen.hh>
#include <core/options/keys/casp.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>


static core::util::Tracer tr("protocols.comparative_modeling.ConstraintRemodelMover");

namespace protocols {
namespace comparative_modeling {

void ConstraintRemodelMover::apply( core::pose::Pose & pose ) {
	using namespace core::scoring;
	using namespace core::options;
	using namespace core::options::OptionKeys;

	ScoreFunctionOP scorefxn( getScoreFunction() );

	// setup ScoreFunction and constraints for minimization.
	// core::scoring::constraints::ConstraintSetOP cstset
	// 	= core::scoring::constraints::ConstraintIO::read_constraints(
	// 	  option[ core::options::OptionKeys::constraints::cst_file ],
	// 	  new core::scoring::constraints::ConstraintSet,pose
	// );
	// pose.constraint_set( cstset );

	scorefxn->set_weight(
		atom_pair_constraint, option[ core::options::OptionKeys::constraints::cst_weight ]()
	);

	scorefxn->set_weight(
		dihedral_constraint, option[ core::options::OptionKeys::constraints::cst_weight ]()
	);

	// minimization
	core::kinematics::MoveMap mm;
	if ( option[ in::file::movemap ].user() ) {
		std::string fn = option[ in::file::movemap ]();
		mm.init_from_file( fn );
	} else {
		mm.set_bb(true);
		mm.set_chi(true);
	} // if ( option[ in::file::movemap ].user() )

	//	core::Real score_before_min( (*scorefxn)( pose ) );
	//tr.Debug << " score before minimization " << score_before_min << " " <<
	//	pose.energies().total_energies().weighted_string_of( scorefxn->weights() ) << std::endl;

	std::string min_type ("dfpmin_armijo_nonmonotone");
	if ( option[ run::min_type ].user() ) {
		min_type = option[ run::min_type ]();
	}
	core::Size iter_count   = 0;
	core::Real before_score = 0;
	//	core::Real before_rms   = 0;
	core::Real after_score  = (*scorefxn)(pose);

	core::Size max_iter = 10000;

	while ( std::abs(before_score - after_score) > 3.0 && iter_count < max_iter ) {
		// go to next iteration
		before_score = after_score;
		//before_rms   = after_rms;
		iter_count++;

		if ( option[ casp::repack ]() ) {
		  core::pack::task::PackerTaskOP task(
				core::pack::task::TaskFactory::create_packer_task( pose )
			);
			task->initialize_from_command_line();
			task->restrict_to_repacking();
			core::pack::pack_rotamers(pose, (*scorefxn), task);
		}

		// looprelax
		// tex - fix this up!
		//if ( option[ core::options::OptionKeys::loops::loop_file ].user() ) {
		//	protocols::LoopRemodelMover loopremodel_mover;
		//	loopremodel_mover.apply( pose );
		//}

		// if ( option[ james::debug ]() ) {
		// 	// pose.dump_pdb( "test" + string_of(iter_count) + ".pdb" );
		// 	core::io::silent::SilentFileData sfd;
		// 	core::io::silent::SilentStructOP ss
		// 		= core::io::silent::SilentStructFactory::get_silent_struct();
		// 	ss->fill_struct( pose );
		//
		// 	sfd.write_silent_struct( *ss, "james.debug" );
		// }

		core::optimization::AtomTreeMinimizer().run(
			pose,
			mm,
			(*scorefxn),
			core::optimization::MinimizerOptions(min_type,0.001,true)
		);

		after_score = (*scorefxn)(pose);
		tr.Debug << iter_count << ": (" << before_score << ")"
			<< "(" << after_score << ")" << std::endl;
		scorefxn->show( tr.Debug, pose );
		//	<< pose.energies().total_energies().weighted_string_of( scorefxn->weights() ) << std::endl;
		//pose.dump_pdb( "test" + string_of(iter_count) + ".pdb" );
		//		core::Real score_after_min( (*scorefxn)( pose ) );
		//tr.Debug << " score after minimization " << score_after_min << " " <<
		//	pose.energies().total_energies().weighted_string_of( scorefxn->weights() ) << std::endl;

	} // while ( std::abs(before_score - after_score) > 0.1 )
} // apply

} // comparative_modeling
} // protocols
