// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file IgnoreSubsetConstraintSet.cc
/// @brief
/// @detailed
/// @author James Thompson
///

// Package Headers

// Project Headers
#include <core/pose/Pose.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/types.hh>
#include <protocols/comparative_modeling/IgnoreSubsetConstraintSet.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>

// Utility headers
#include <utility/vector1.fwd.hh>
#include <utility/pointer/ReferenceCount.hh>

#include <core/util/Tracer.hh>

//// C++ headers
#include <cstdlib>
#include <string>
#include <fstream>
#include <set>

static core::util::Tracer tr("protocols.comparative_modeling");

using core::scoring::constraints::ConstraintSet;
using core::scoring::constraints::ConstraintSetOP;

namespace protocols {
namespace comparative_modeling {

using namespace core;

IgnoreSubsetConstraintSet::IgnoreSubsetConstraintSet(
	std::set< int > residues_to_ignore,
	ConstraintSet const & other
) :
  ConstraintSet( other ),
	ignore_list_( residues_to_ignore )
{}

/// @copy constructor. Does nothing.
IgnoreSubsetConstraintSet::IgnoreSubsetConstraintSet(
	IgnoreSubsetConstraintSet const & other
) :
	ConstraintSet( other ),
	ignore_list_( other.ignore_list() )
{}

void
IgnoreSubsetConstraintSet::eval_atom_derivative_for_residue_pairs (
  id::AtomID const & atom_id,
  pose::Pose const & pose,
  scoring::ScoreFunction const &,
  scoring::EnergyMap const & weights,
  Vector & F1,
  Vector & F2
) const
{
  using scoring::constraints::ResidueConstraints;

  Size const resi( atom_id.rsd() );

  if ( residue_pair_constraints_.size() >= resi && residue_pair_constraints_[ resi ] ) {
    ResidueConstraints const & resi_constraints( *residue_pair_constraints_[ resi ] );
    for ( ResidueConstraints::const_iterator it = resi_constraints.begin(),
					ite = resi_constraints.end(); it != ite; ++it
		) {
      Size const resj( it->first );
      if ( !ignore(resi) && !ignore(resj) ) {
			it->second->eval_atom_derivative(
				atom_id, pose.conformation(), weights, F1, F2
			);
      } // if ( !ignore)
    } // for ( ResidueConstraints )
  } // if ( we have ResidueConstraints )
} // eval_atom_derivative_for_residue_pairs


///
void
IgnoreSubsetConstraintSet::residue_pair_energy(
   Residue const & rsd1,
   Residue const & rsd2,
   Pose const & pose,
   core::scoring::ScoreFunction const & scorefxn,
   core::scoring::EnergyMap & emap
) const
{
  int const pos1( rsd1.seqpos() ), pos2( rsd2.seqpos() );
  if ( ignore(pos1) || ignore(pos2) ) {
    return; //cast avoids warning
  }
  ConstraintSet::residue_pair_energy( rsd1, rsd2, pose, scorefxn, emap );
}

bool
IgnoreSubsetConstraintSet::ignore( int const pos ) const {
	if ( ignore_list_.find(pos) == ignore_list_.end() ) {
		return false;
	}

	return true;
}

void
IgnoreSubsetConstraintSet::ignore_residue( int const pos ) {
	ignore_list_.insert( pos );
}

std::set< int >
IgnoreSubsetConstraintSet::ignore_list() const {
	return ignore_list_;
}

} // comparative_modeling
} // protocols

