// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/comparative_modeling/ThreadingMover.hh
/// @brief
/// @author James Thompson

// libRosetta headers

#include <protocols/comparative_modeling/ThreadingMover.hh>
#include <protocols/comparative_modeling/StealSideChainsMover.hh>
#include <protocols/comparative_modeling/util.hh>

#include <protocols/loops/looprelax_protocols.hh>
#include <protocols/loops/LoopClass.hh>
#include <protocols/loops/loops_main.hh>
#include <protocols/loops/LoopMover.hh>
#include <protocols/loops/LoopMoverFactory.hh>

#include <core/types.hh>

#include <core/options/option.hh>

#include <core/chemical/util.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/ChemicalManager.hh>

#include <core/pose/util.hh>
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>

#include <core/conformation/Residue.functions.hh>
#include <core/conformation/Residue.fwd.hh>

#include <core/fragment/FragSet.fwd.hh>
#include <core/fragment/FragmentIO.hh>

#include <core/id/AtomID.hh>
#include <core/id/AtomID_Map.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/id/AtomID_Mask.fwd.hh>

#include <core/sequence/Sequence.hh>
#include <core/sequence/SWAligner.hh>
#include <core/sequence/ScoringScheme.fwd.hh>
#include <core/sequence/ScoringSchemeFactory.hh>
#include <core/sequence/SimpleScoringScheme.hh>
#include <core/sequence/SequenceMapping.hh>
#include <core/sequence/SequenceAlignment.hh>
#include <core/sequence/util.hh>

#include <core/pack/pack_rotamers.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/optimizeH.hh>
#include <core/pack/pack_missing_sidechains.hh>

#include <core/scoring/rms_util.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunction.fwd.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/scoring/constraints/Func.hh>
#include <core/scoring/constraints/Func.fwd.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/scoring/constraints/ConstraintIO.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/ConstraintSet.fwd.hh>
#include <core/scoring/constraints/AtomPairConstraint.hh>
#include <core/scoring/constraints/util.hh>

#include <core/kinematics/MoveMap.hh>
#include <core/kinematics/MoveMap.fwd.hh>
#include <core/optimization/AtomTreeMinimizer.hh>

#include <numeric/random/random.hh>
#include <core/optimization/MinimizerOptions.hh>

#include <utility/vector1.hh>
#include <utility/file/FileName.hh>
#include <utility/io/izstream.hh>

#include <core/util/basic.hh>
#include <core/util/Tracer.hh>

// C++ headers
#include <string>

// option key includes

#include <core/options/keys/cm.OptionKeys.gen.hh>

/// random number generator for randomizing missing density coordinates
static numeric::random::RandomGenerator RG(298211);  // <- Magic number, do not change it!

namespace protocols {
namespace comparative_modeling {

utility::vector1< core::fragment::FragSetOP > ThreadingMover::frag_libs() const {
    return frag_libs_;
}

void ThreadingMover::frag_libs(
   utility::vector1< core::fragment::FragSetOP > new_libs
) {
   frag_libs_ = new_libs;
}

void ThreadingMover::apply(
	core::pose::Pose & query_pose
) {
	using core::Real;
	using core::Size;
	using namespace core::options;
	using namespace core::options::OptionKeys;
	using utility::vector1;
	static core::util::Tracer tr("protocols.comparative_modeling.threading");

	// put checks here to make sure that the template pose and the template
	// pose in the alignment file match up!
	using namespace core::sequence;

	SequenceOP query_sequence(
		new Sequence(
			align_.sequence( 1 )->ungapped_sequence(),
			align_.sequence( 1 )->id(),
			align_.sequence( 1 )->start()
		)
	);

	SequenceOP aligned_template(
		align_.sequence(template_index_)->clone()
	);

	SequenceOP t_align_seq(
		new Sequence(
			aligned_template->ungapped_sequence(),
			aligned_template->id() + "_align_seq",
			aligned_template->start()
		)
	);

	SequenceOP t_pdb_seq(
		new Sequence (
			template_pose_.sequence(),
			aligned_template->id() + "_pdb_seq",
			1
		)
	);

	// construct an intermediate alignment of the sequence from the alignment
	// to the sequence in the PDB file.
	SWAligner sw_align;
  ScoringSchemeOP ss( new SimpleScoringScheme( 120, 0, -100,   0 ) );

	tr.Debug << "query sequence         : " << query_pose.sequence() << std::endl;
	tr.Debug << "query sequence         : " << (*query_sequence) << std::endl;
	tr.Debug << "aligned_template       : " << (*aligned_template) << std::endl;
	tr.Debug << "template_sequence (aln): " << (*t_align_seq) << std::endl;
	tr.Debug << "template_sequence (pdb): " << (*t_pdb_seq) << std::endl;

	SequenceAlignment intermediate = sw_align.align( t_align_seq, t_pdb_seq, ss );

	if ( intermediate.identities() != intermediate.length() ) {
		tr.Warning << "Error: potential mismatch between sequence from alignment ";
		tr.Warning << " and sequence from PDB!" << std::endl;
		tr.Warning << "alignment: " << std::endl << intermediate
			<< std::endl;
	}

	SequenceMapping query_to_fullseq = align_.sequence_mapping( query_index_, template_index_ );
	tr.Debug << "Query:    " << *align_.sequence( query_index_ ) << std::endl;
	tr.Debug << "Template: " << *align_.sequence( template_index_ ) << std::endl;
	tr.Debug << "Original Mapping:" <<  query_index_ << "-->" << template_index_ <<  std::endl;
	query_to_fullseq.show( tr.Debug );

	SequenceMapping intermed_map = intermediate.sequence_mapping( 1, 2 );

	// final mapping is the mapping from query to the template PDB sequence,
	// rather then the direct template sequence.
	SequenceMapping query_to_pdbseq = core::sequence::transitive_map(
		query_to_fullseq, intermed_map
	);
	tr.Debug << "Transitive Map" << std::endl;
	query_to_pdbseq.show( tr.Debug );

	//SequenceAlignment final_aln = mapping_to_alignment(
	//			query_to_pdbseq, query_sequence ,t_pdb_seq);

	//tr.Debug << "Original Aln: " << original_aln << std::endl;

	//tr.Debug  << "Remapped Aln: " << final_aln << std::endl;
	tr.Debug  << "mapping from " << query_index_ << " to " << template_index_
		<< std::endl;
	query_to_pdbseq.show( tr.Debug );

	std::string const template_id( template_pose_.pdb_info()->name() );
	core::pose::add_score_line_string( query_pose, "template", template_id );

	core::Size n_copied( 0 );

	core::id::AtomID_Mask missing( true );
	core::id::initialize( missing, query_pose ); // used for repacking atoms
	for ( Size resi = 1; resi <= query_pose.total_residue(); resi++ ) {
		Size t_resi = query_to_pdbseq[ resi ];

		// skip this residue if we're not aligned
		if ( t_resi == 0 ) {
			continue;
		}

		if ( t_resi > template_pose_.total_residue() ) {
			tr.Error << "Error: don't have residue " << t_resi
				<< " in template_pose!" << std::endl;

			tr.Error << "template_pose.total_residue() == "
				<< template_pose_.total_residue() << std::endl;
			continue;
		}

		using std::string;
		using utility::vector1;
		for ( Size atomj = 1; atomj <= query_pose.residue(resi).natoms(); ++atomj ) {
			std::string const atom_name( query_pose.residue(resi).atom_name( atomj ));

			//determine if we copy this atom: backbone and CB always, sidechain if identical aa
			core::id::AtomID cb_id(
				core::id::NamedAtomID( "CB", resi ),
				query_pose,
				false /*don't raise exception we check explicitly*/
			);
			if ( !query_pose.residue(resi).atom_is_backbone(atomj)
				&& query_pose.residue(resi).name1() != template_pose_.residue(t_resi).name1()
				&& cb_id.atomno() != atomj
			) continue;

			core::id::NamedAtomID query_id   ( atom_name, resi   );
			core::id::NamedAtomID template_id( atom_name, t_resi );

			// check to make sure that both poses have this atom_name
			if ( !query_pose.residue_type(resi).has( atom_name ) ) {
				tr.Warning 	<< "skipping atom,position " << atom_name << "," << resi
					<< " because query doesn't have atom " << atom_name << "." << std::endl;
				continue;
			}
			if ( !template_pose_.residue_type(t_resi).has( atom_name ) ) {
				tr.Warning 	<< "skipping atom,position " << atom_name << "," << resi
					<< " because template doesn't have atom " << atom_name << "."
					<< std::endl;
				continue;
			}

			missing[ core::id::AtomID( atomj, resi ) ] = false;
			query_pose.set_xyz( query_id, template_pose_.xyz( template_id ) );

			// put the OXT at the same place as the O! brilliant hack!
			if ( query_pose.residue_type(resi).has_atom_name("OXT") &&
					atom_name == "O"
			) {
				core::id::NamedAtomID oxt_id( "OXT", resi );
				query_pose.set_xyz( oxt_id, template_pose_.xyz( template_id ) );
			}
		} // for atom_i

		++n_copied;
	} // for resi

	tr.Debug << "Built threading model for sequence "
		<< query_pose.sequence() << std::endl;
	tr.Debug	<< "Copied " << n_copied << " / "
		<< query_pose.total_residue() << " from "
		<< align_.sequence(template_index_)->id() << std::endl;

	using namespace core::options;
	using namespace core::options::OptionKeys;;

	if ( randomize_loop_coords() ) {
		typedef core::conformation::ResidueOPs::iterator ResIterator;
		for ( ResIterator it = query_pose.res_begin(); it != query_pose.res_end();
				++it
		) {
			core::conformation::idealize_hydrogens(
				*(*it), query_pose.conformation()
			);
		}

		//randomize also all missing atoms
		for ( Size pos = 1; pos <= query_pose.total_residue(); pos++ ) {
			Size atomj( 1 );
			for ( core::id::AtomID_Mask::AtomMap::const_iterator
					it = missing[ pos ].begin(), eit = missing[ pos ].end(); it != eit;
					++it, ++atomj
			) {
				if ( query_pose.residue( pos ).atom_is_hydrogen( atomj ) ) continue;
				if ( *it ) { //entry is missng == true
					core::Vector ai(
						900.000 + RG.uniform()*100.000,
						900.000 + RG.uniform()*100.000,
						900.000 + RG.uniform()*100.000
					);
					query_pose.set_xyz( core::id::AtomID( atomj, pos ), ai );
					//now randomize also attached hydrogens
					for ( Size atom_nr = query_pose.residue( pos ).attached_H_begin( atomj );
								atom_nr <= query_pose.residue( pos ).attached_H_end( atomj ); ++atom_nr ) {
						core::Vector ai(
								900.000 + RG.uniform()*100.000,
								900.000 + RG.uniform()*100.000,
								900.000 + RG.uniform()*100.000
						);
						query_pose.set_xyz( core::id::AtomID( atom_nr, pos ), ai );
					}
				}
			}
		} //for missing atoms
	} // if randomize_loop_coords()

	if ( build_loops() ) {
		using protocols::loops::Loops;
		tr.Debug << "building query loops." << std::endl;
		Loops query_loops = loops_from_alignment(
			 query_pose.total_residue(), align_, min_loop_size()
		);
		query_loops.choose_cutpoints( query_pose );
		tr.Debug << query_loops << std::endl;

		if ( query_loops.size() > 0 ) {
			// switch to centroid ResidueTypeSet for loop remodeling
			std::string const orig_rsd_set_name(
				query_pose.residue_type(1).residue_type_set().name()
			);

			core::chemical::switch_to_residue_type_set(
				query_pose, core::chemical::CENTROID
			);

			loops::LoopMoverOP loop_mover = protocols::loops::get_loop_mover(
				option[ cm::loop_mover ](), query_loops
			);
			for ( Size ii = 1; ii <= frag_libs().size(); ++ii ) {
				loop_mover->add_fragments( frag_libs()[ii] );
			}
			loop_mover->apply( query_pose );

			// switch back to original ResidueTypeSet after loop modeling
			if ( orig_rsd_set_name != core::chemical::CENTROID ) {
				core::chemical::switch_to_residue_type_set(
					query_pose,
					orig_rsd_set_name
				);
			}
		} else {
			 tr.Warning << "No loops found!" << std::endl;
		}
	} // build_query_loops

	// steal side chains
	//std::cout << "copying sidechains" << std::endl;
	//StealSideChainsMover sc_mover( template_pose_, query_to_pdbseq );
	//sc_mover.apply( query_pose );
	//std::cout << "finished copying sidechains" << std::endl;

	// repack the structure if specified by the user
	if ( repack_query() ) {
		using namespace core::scoring;
		ScoreFunctionOP scorefxn( getScoreFunction() );

		// repack missing sidechains
		core::id::AtomID_Mask missing( true );
		core::id::initialize( missing, query_pose );
		tr.Debug << "repacking residues on pose with ScoreFunction: " << std::endl;
		scorefxn->show( tr.Debug );
		tr.Debug << std::endl << std::endl;
		core::pack::pack_missing_sidechains( query_pose, missing );

		tr.Debug << "setting up ideal hydrogen geometry on all residues."
			<< std::endl;
		typedef core::conformation::ResidueOPs::iterator iter;
		for ( iter it = query_pose.res_begin(), end = query_pose.res_end();
					it != end; ++it
		) {
			core::conformation::idealize_hydrogens( *(*it), query_pose.conformation() );
		}

		tr.Debug << "optimizing hydrogen placement with the packer."
			<< std::endl;
		core::pack::optimize_H_and_notify( query_pose, missing );

		scorefxn->set_weight( core::scoring::peptide_bond, 1.0 );
		(*scorefxn)(query_pose);
		scorefxn->show( tr.Debug, query_pose );
	} // repack_query

	tr.flush();
} // apply

} // comparative_modeling
} // protocols
