// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/comparative_modeling/ThreadingMover.hh
/// @brief
/// @author James Thompson

// libRosetta headers

#ifndef protocols_comparative_modeling_ThreadingMover_HH
#define protocols_comparative_modeling_ThreadingMover_HH

#include <protocols/moves/Mover.hh>

#include <core/types.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/pose/Pose.hh>
#include <core/chemical/util.hh>

#include <core/sequence/Sequence.hh>
#include <core/sequence/SequenceAlignment.hh>
#include <core/fragment/FragSet.hh> // ???
#include <core/fragment/FragSet.fwd.hh>

#include <utility/vector1.hh>
#include <utility/file/FileName.hh>
#include <utility/io/izstream.hh>

#include <core/util/basic.hh>
#include <core/util/Tracer.hh>

#include <protocols/comparative_modeling/AlignmentID.hh>

// C++ headers
#include <set>
#include <string>


namespace protocols {
namespace comparative_modeling {

class ThreadingMover : public protocols::moves::Mover {
	typedef utility::file::FileName FileName;

public:

	/// @brief align describes the association between the query and template
	/// sequences, template_pose is the conformation from which to build a
	/// threading model.
	ThreadingMover(
		core::sequence::SequenceAlignment align,
		core::pose::Pose template_pose
	) :
		protocols::moves::Mover("ThreadingMover"),
		query_index_( 1 ),
		template_index_( 2 ),
		template_pose_( template_pose ),
		align_( align ),
		build_query_loops_( true ),
		repack_query_( true ),
		randomize_loop_coords_( false ),
		min_loop_size_( 3 )
	{}

	virtual ~ThreadingMover() {}

	/// @brief Returns the index of the query sequence in SequenceAlignment
	/// object.
	core::Size query_index() const {
		return query_index_;
	}

	/// @brief Returns the index of the template sequence in SequenceAlignment
	/// object.
	core::Size template_index() const {
		return template_index_;
	}

	/// @brief Returns the SequenceAlignment object used in threading.
	core::sequence::SequenceAlignment alignment() {
		return align_;
	}

	/// @brief Sets the index of the query sequence in SequenceAlignment object.
	void query_index( core::Size new_index ) {
		query_index_ = new_index;
	}

	/// @brief Sets the index of the template sequence in SequenceAlignment
	/// object.
	void template_index( core::Size new_index ) {
		template_index_ = new_index;
	}

	/// @brief Sets the SequenceAlignment associated with this object.
	void alignment( core::sequence::SequenceAlignment new_align ) {
		align_ = new_align;
	}

	void template_pose( core::pose::Pose template_pose ) {
		template_pose_ = template_pose;
	}

	//boolean setters
	void build_loops( bool setting ) {
		build_query_loops_ = setting;
	}

	void randomize_loop_coords( bool setting ) {
		randomize_loop_coords_ = setting;
	}

	void repack_query( bool setting ) {
		repack_query_ = setting;
	}

	//boolean getters
	bool build_loops() const {
		return build_query_loops_;
	}

	bool repack_query() const {
		return repack_query_;
	}

	bool randomize_loop_coords() {
		return randomize_loop_coords_;
	}

	void min_loop_size( core::Size const new_size ) {
		min_loop_size_ = new_size;
	}

	Size min_loop_size() const {
		return min_loop_size_;
	}

	utility::vector1< core::fragment::FragSetOP > frag_libs() const;

	void frag_libs(
		utility::vector1< core::fragment::FragSetOP > new_libs
	);

	/// @brief Threads the given Pose onto the template_pose with the
	/// SequenceAlignment provided.
	virtual void apply( core::pose::Pose & query_pose );

private: // methods

	///
	bool atoms_are_equiv( std::string const & a1, std::string const & a2 );

	/// @brief add pair of equivalent atoms to atom_equiv_ table.
	void add_equiv_atoms( std::string const & a1, std::string const & a2 );

	/// @brief initialize table of equivalent atoms between all residues.
	void init_atom_equiv();

private: // data members

	core::Size query_index_, template_index_;
	core::pose::Pose template_pose_;
	core::sequence::SequenceAlignment align_;
	bool build_query_loops_;
	bool repack_query_;
	bool randomize_loop_coords_;
	core::Size min_loop_size_;

	//std::map< std::string, utility::vector1< std::string > > atom_equiv_;
	std::map< std::string, std::set< std::string > > atom_equiv_;
	utility::vector1< core::fragment::FragSetOP > frag_libs_;
}; // ThreadingMover

} // comparative_modeling
} // protocols

#endif
