// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file DockingInitialPerturbation.cc
/// @brief initial position functions
/// @detailed
///		This contains the functions that create initial positions for docking
///		You can either randomize partner 1 or partner 2, spin partner 2, or
///		perform a simple perturbation.
/// 	Also contains docking mcm protocol
/// @author Monica Berrondo

#include <protocols/docking/DockingInitialPerturbation.hh>

// Rosetta Headers
#include <protocols/moves/Mover.hh>
#include <protocols/moves/RigidBodyMover.hh>

#include <core/options/option.hh>
#include <core/scoring/Energies.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <string>

//Utility Headers
#include <numeric/conversions.hh>

#include <numeric/trig.functions.hh>
#include <numeric/xyzMatrix.fwd.hh>

#include <core/util/Tracer.hh>
using core::util::T;

// option key includes

#include <core/options/keys/docking.OptionKeys.gen.hh>


using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.docking.DockingInitialPerturbation");
static core::Size trans ( 1 ), rot ( 2 );

using namespace core;

namespace protocols {
namespace docking {


// initial perturbation on one of the partners
// the type of perturbation is defined in the options
// some of the options are randomize1 (randomizes the first partner)
// randomize2 (randomizes the second partner), dock_pert, and spin
//------------------------------------------------------------------------------
//
//     there are several ways to perturb the structure before beginning
//     the search; they are controlled through command-line flags
//
//     at the end, partners are slid into contact and scored
//


//constructors
DockingInitialPerturbation::DockingInitialPerturbation(): Mover(), rb_jump_( 1 ), slide_( true )
	{
		Mover::type( "DockingInitialPerturbation" );
	}

DockingInitialPerturbation::DockingInitialPerturbation(
		int const rb_jump_in,
		bool const slide_in
	) : Mover(), rb_jump_(rb_jump_in), slide_(slide_in)
	{
		Mover::type( "DockingInitialPerturbation" );
	}

//destructor
DockingInitialPerturbation::~DockingInitialPerturbation() {}

////////////////////////////////////////////////////////////////////////////////
/// @begin initial_perturbation
///
/// @brief   Make starting perturbations for rigid body moves
///
/// @detailed    There are several ways to perturb the structure before beginning
///     the search; they are controlled through command-line flags
///     At the end, partners are slid into contact and scored (including
///     mc_reset).
///     Also, they are tested for passing the FAB filter.
///
///
/// @references see dock_structure or pose_docking from rosetta++
///
/// @authors Monica Berrondo June 14 2007
///
/// @last_modified October 17 2007
/////////////////////////////////////////////////////////////////////////////////
void DockingInitialPerturbation::apply( core::pose::Pose & pose )
{
	using namespace moves;
	using namespace options;
	//////////////////////////////////
	//The options are -randomize1
	//			      -randomize2
	//								-spin
	//								-dock_pert
	//						for dock_pert, also need to get the normal and parallel perturbation
	//////////////////////////////////
	TR << "Reading options..." << std::endl;
	if( option[ OptionKeys::docking::randomize1 ]() ) {
		TR << "randomize1: true" << std::endl;
		RigidBodyRandomizeMover mover( pose, rb_jump_, partner_upstream );
		mover.apply( pose );
	}
	if( option[ OptionKeys::docking::randomize2 ]() ) {
		TR << "randomize2: true" << std::endl;
		RigidBodyRandomizeMover mover( pose, rb_jump_, partner_downstream );
		mover.apply( pose );
	}
	if( option[ OptionKeys::docking::dock_pert ].user() ) {
		// DO NOT supply default values for this option -- reasonable values differ for protein and ligand protocols.
		// Also, default values will cause a perturbation to *always* occur, even with no command line flags -- very surprising.
		// Adding defaults WILL BREAK existing protocols in unexpected ways.
		// Decided by Jeff, Monica, Ian, and Sid in March 2008.
		//
		// Jeff notes that eventually there should be 3 parameters, like Rosetta++:
		// rotation, normal translation, and perpendicular translation.
		TR << "dock_pert: true" << std::endl;
		/// read in dock_pert options from commandline.  the first value is the
		/// rotation magnitude and the second value is the translational value
		utility::vector1< Real > pert_mags = option[ OptionKeys::docking::dock_pert ]();
		//TR << "option[ docking::rotational ]()" << rot << "\n";
		//TR << "option[ docking::parallel ]()" << trans << "\n";
		TR << "option[ docking::dock_pert ]()" << pert_mags[rot] << ' ' << pert_mags[trans] << std::endl;
		RigidBodyPerturbMover mover( rb_jump_, pert_mags[rot], pert_mags[trans] );
		mover.apply( pose );
	}

	if( option[ OptionKeys::docking::uniform_trans ].user() ) {
		core::Real mag( option[ OptionKeys::docking::uniform_trans ]() );
		TR << "uniform_trans: " << mag << std::endl;
		UniformSphereTransMover mover( rb_jump_, mag );
		mover.apply( pose );
	}
	if( option[ OptionKeys::docking::spin ]() ) {
		TR << "axis_spin: true" << std::endl;
		RigidBodySpinMover mover( rb_jump_ );
		mover.apply( pose );
	}
	// DO NOT do this for e.g. ligand docking
	if ( slide_ ) {
		DockingSlideIntoContact slide( rb_jump_ );
		slide.apply( pose );
	}
}


//constructor
DockingSlideIntoContact::DockingSlideIntoContact(
	int const rb_jump_in
) : Mover(), rb_jump_(rb_jump_in)
{
	using namespace core::scoring;
	Mover::type( "DockingSlideIntoContact" );
	scorefxn_ = ScoreFunctionFactory::create_score_function( CENTROID_WTS, DOCK_LOW_PATCH );
		scorefxn_ = ScoreFunctionFactory::create_score_function( "interchain_cen" );
}

//destructor
DockingSlideIntoContact::~DockingSlideIntoContact() {}


void DockingSlideIntoContact::apply( core::pose::Pose & pose )
{
	using namespace moves;

	RigidBodyTransMover mover( pose, rb_jump_ );
	( *scorefxn_ )( pose );

	TR << "sliding into contact" << std::endl;
	TR << "Moving away" << std::endl;
	core::Size const counter_breakpoint( 500 );
	core::Size counter( 0 );
	// first try moving away from each other
	while ( pose.energies().total_energies()[ scoring::interchain_vdw ] > 0.1 && counter <= counter_breakpoint ) {
		mover.apply( pose );
		( *scorefxn_ )( pose );
		++counter;
	}
	if( counter > counter_breakpoint ){
		TR<<"failed moving away with original vector. Aborting DockingSlideIntoContact."<<std::endl;
		set_current_tag( "fail" );
		return;
	}
	counter = 0;
	// then try moving towards each other
	TR << "Moving together" << std::endl;
	mover.trans_axis().negate();
	while ( counter <= counter_breakpoint && pose.energies().total_energies()[ scoring::interchain_vdw ] < 0.1 ) {
		mover.apply( pose );
		( *scorefxn_ )( pose );
		++counter;
	}
	if( counter > counter_breakpoint ){
		TR<<"moving together failed. Aborting DockingSlideIntoContact."<<std::endl;
		set_current_tag( "fail" );
		return;
	}
	// move away again until just touching
	mover.trans_axis().negate();
	mover.apply( pose );
}

//constructor
FaDockingSlideTogether::FaDockingSlideTogether(
		int const rb_jump_in
	) : Mover(), rb_jump_(rb_jump_in), tolerance_(0.2)
	{
		Mover::type( "FaDockingSlideTogether" );
		scorefxn_ = new core::scoring::ScoreFunction();
		scorefxn_->set_weight( core::scoring::fa_rep, 1.0 );
	}

//destructor
FaDockingSlideTogether::~FaDockingSlideTogether() {}

void FaDockingSlideTogether::apply( core::pose::Pose & pose )
{
	using namespace core::scoring;

	// A very hacky way of guessing whether the components are touching:
	// if pushed together by 1A, does fa_rep change at all?
	// (The docking rb_* score terms aren't implemented as of this writing.)
	(*scorefxn_)( pose );
	core::Real const initial_fa_rep = pose.energies().total_energies()[ fa_rep ];
	bool are_touching = false;
	moves::RigidBodyTransMover trans_mover( pose, rb_jump_ );

	//int i=1;
	// Take 2A steps till clash, then back apart one step.  Now you're within 2A of touching.
	// Repeat with 1A steps, 0.5A steps, 0.25A steps, etc until you're as close are you want.
	for( core::Real stepsize = 2.0; stepsize > tolerance_; stepsize /= 2.0 ) {
		trans_mover.trans_axis( trans_mover.trans_axis().negate() ); // now move together
		trans_mover.step_size(stepsize);
		core::Size const counter_breakpoint( 500 );
		core::Size counter( 0 );
		do
		{
			trans_mover.apply( pose );
			(*scorefxn_)( pose );
			core::Real const push_together_fa_rep = pose.energies().total_energies()[ fa_rep ];
			//std::cout << "fa_rep = " << push_together_fa_rep << std::endl;
			are_touching = (std::abs(initial_fa_rep - push_together_fa_rep) > 1e-4);
			//std::ostringstream s;
			//s << "snapshot" << i << ".pdb";
			//pose.dump_pdb(s.str());
			//i += 1;
			++counter;
		} while( counter <= counter_breakpoint && !are_touching );
		if( counter > counter_breakpoint ){
			TR<<"Failed Fadocking Slide Together. Aborting."<<std::endl;
			set_current_tag( "fail" );
		}
		trans_mover.trans_axis( trans_mover.trans_axis().negate() ); // now move apart
		trans_mover.apply( pose );
	}
}

} // namespace docking
} // namespace protocols
