// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file docking_initialization_protocols
/// @brief initialization protocols for docking
/// @detailed
///		This contains the functions that create initial positions for docking
///		You can either randomize partner 1 or partner 2, spin partner 2, or
///		perform a simple perturbation.
/// @author Monica Berrondo


#ifndef INCLUDED_protocols_docking_DockingInitialPerturbation_HH
#define INCLUDED_protocols_docking_DockingInitialPerturbation_HH

// Package headers
#include <core/pose/Pose.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <protocols/moves/Mover.hh>
#include <protocols/moves/RigidBodyMover.hh>

#include <utility/pointer/owning_ptr.hh>
#include <string>

namespace protocols {
namespace docking {

/// @brief this mover carries out the initial perturbation phase of the RosettaDock algorithm
/// based on user-inputted command line options
class DockingInitialPerturbation; // fwd declaration
typedef utility::pointer::owning_ptr< DockingInitialPerturbation > DockingInitialPerturbationOP;
typedef utility::pointer::owning_ptr< DockingInitialPerturbation const > DockingInitialPerturbationCOP;

class DockingInitialPerturbation : public moves::Mover
{
public:

	// default constructor
	DockingInitialPerturbation();

	// constructor with arguments
	DockingInitialPerturbation(
		int const rb_jump_in,
		bool const slide_in=true
	);

	//destructor
	~DockingInitialPerturbation();

	// protocol functions
	virtual void apply( core::pose::Pose & pose );

private:
	// docking
	int rb_jump_;

	/// do slide into context?
	bool slide_;
};

/// @brief Contrary to the name, slides things apart first, then together.
/// OK for proteins, bad for ligands (because they may escape the pocket permanently).
class DockingSlideIntoContact : public moves::Mover
{
public:

	// constructor with arguments
	DockingSlideIntoContact( int const rb_jump_in=1 );

	//destructor
	~DockingSlideIntoContact();

	// protocol functions
	virtual void apply( core::pose::Pose & pose );

private:
	core::scoring::ScoreFunctionOP scorefxn_;

	/// which jump to use for docking
	int rb_jump_;
};

class FaDockingSlideTogether; // fwd declaration
typedef utility::pointer::owning_ptr< FaDockingSlideTogether > FaDockingSlideTogetherOP;
typedef utility::pointer::owning_ptr< FaDockingSlideTogether const > FaDockingSlideTogetherCOP;

/// @brief Slides docking partners together by monitoring fa_rep.
/// @details
///		If partners are already touching, no change is made.
///		Separation will be 1A or less after calling this function.
class FaDockingSlideTogether : public moves::Mover
{
public:
  FaDockingSlideTogether( int const rb_jump_in=1 );

	//destructor
	~FaDockingSlideTogether();

	virtual void apply( core::pose::Pose & pose );

private:
	int rb_jump_;
	core::scoring::ScoreFunctionOP scorefxn_;
	core::Real tolerance_; ///< how accurate do you want to be?

};

} // docking
} // protocols

#endif
