// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file DockingLowRes
/// @brief protocols that are specific to docking low resolution
/// @detailed
/// @author Monica Berrondo
/// @author Modified by Sergey Lyskov

#include <protocols/docking/DockingLowRes.hh>

// Rosetta Headers
#include <core/kinematics/MoveMap.hh>

#include <core/options/option.hh>
#include <core/options/keys/OptionKeys.hh>

#include <core/kinematics/FoldTree.hh>

#include <core/pose/Pose.hh>

#include <core/scoring/ScoreFunction.hh>

#include <protocols/moves/ConformerSwitchMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/moves/OutputMovers.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <protocols/moves/TrialMover.hh>
#include <protocols/moves/RepeatMover.hh>
#include <protocols/moves/MoverContainer.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <string>

//Utility Headers
#include <numeric/conversions.hh>
//	#include <numeric/random.functions.hh>

#include <numeric/trig.functions.hh>
#include <numeric/xyzMatrix.fwd.hh>

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.docking.DockingLowRes");

//     originally from dock_structure.cc Jeff Gray April 2001
//
//

using namespace core;

namespace protocols {
namespace docking {

// default constructor
DockingLowRes::DockingLowRes() :
	Mover(),
	rb_jump_( 1 )
{
	scorefxn_ = core::scoring::ScoreFunctionFactory::create_score_function( "interchain_cen" );
	moves::Mover::type( "DockingLowRes" );
}


// constructor with arguments
DockingLowRes::DockingLowRes(
	core::scoring::ScoreFunctionCOP scorefxn_in,
	int const rb_jump_in
) : Mover(), scorefxn_(scorefxn_in), rb_jump_(rb_jump_in)
{
	moves::Mover::type( "DockingLowRes" );
}

//destructor
DockingLowRes::~DockingLowRes() {}

protocols::moves::MoverOP
DockingLowRes::clone() const {
	return new DockingLowRes(*this);
}

void DockingLowRes::set_default( core::pose::Pose & pose ) {
	using namespace core::options;

	// sets up the stuff in pose
	(*scorefxn_)( pose );
	/// Now handled automatically.  scorefxn_->accumulate_residue_total_energies( pose );

	// cycles
	inner_cycles_ = option[ OptionKeys::docking::docking_centroid_inner_cycles ]();
	outer_cycles_ = option[ OptionKeys::docking::docking_centroid_outer_cycles ]();

	trans_magnitude_ = 0.7;
	rot_magnitude_ = 5.0;
	chi_ = false;
	bb_ = false;

	temperature_ = 0.8;
	nb_list_ = true; /// not sure if this should be true or not
	accept_rate_ = 0.0;

	set_default_mc( pose );
	set_default_move_map();
  set_default_protocol( pose );
}


void DockingLowRes::set_default_mc( pose::Pose & pose ) {
	// create the monte carlo object and movemap
	mc_ = new moves::MonteCarlo( pose, *scorefxn_, temperature_ );
}

void DockingLowRes::set_default_move_map() {
	// the movable dof's
	movemap_ = new kinematics::MoveMap();
	movemap_->set_chi( chi_ ); // is this right?
	movemap_->set_bb( bb_ ); // is this right?
	movemap_->set_jump( rb_jump_, true );
}

void DockingLowRes::set_default_protocol( core::pose::Pose & pose){
	using namespace moves;

	rb_mover_ = new RigidBodyPerturbNoCenterMover( rb_jump_, rot_magnitude_, trans_magnitude_ );

	docking_lowres_protocol_ = new SequenceMover;
	docking_lowres_protocol_->add_mover( rb_mover_ );

	set_flexible_docking_protocol( pose );

}

void DockingLowRes::set_flexible_docking_protocol( core::pose::Pose & pose ){
	using namespace moves;
	using namespace options;


//for ensemble mode
	if ((option[ OptionKeys::docking::ensemble1 ]() != "") ||  (option[ OptionKeys::docking::ensemble2 ]() != "")) {
		Size start_res(1), end_res(1), cutpoint(pose.fold_tree().cutpoint_by_jump( rb_jump_ ));
		std::string ensemble_file;

		inner_cycles_ = 25;

	if ( option[ OptionKeys::docking::ensemble1 ]() != ""){
			start_res = 1;
			end_res = cutpoint;
			ensemble_file = option[ OptionKeys::docking::ensemble1 ]();
			}

	if ( option[ OptionKeys::docking::ensemble2 ]() != ""){
			start_res = cutpoint+1;
			end_res = pose.total_residue();
			ensemble_file = option[ OptionKeys::docking::ensemble2 ]();
			}

    ConformerSwitchMoverOP ensemble_mover = new ConformerSwitchMover( start_res, end_res, rb_jump_, scorefxn_, ensemble_file);
		docking_lowres_protocol_->add_mover( ensemble_mover );
    }

}


////////////////////////////////////////////////////////////////////////////////
/// @begin DockingLowRes.apply
///
/// @brief Perform several cycles of rigid-body Monte Carlo moves
///       and adapt the step size.
/// @detailed
///
/// @remarks
///       currently used only in the low-resolution step (centroid mode)
///
/// @references pose_docking_centroid_rigid_body_adaptive from pose_docking.cc and
///				rigid_body_MC_cycle_adaptive from dock_structure.cc
///
/// @authors Monica Berrondo October 22 2007
///
/// @last_modified October 22 2007
/////////////////////////////////////////////////////////////////////////////////
void DockingLowRes::apply( core::pose::Pose & pose )
{
	using namespace scoring;

	TR << "in DockingLowRes.apply\n";

	set_default( pose );


	TR << "::::::::::::::::::Centroid Rigid Body Adaptive:::::::::::::::::::\n";

	for ( int i=1; i<=outer_cycles_; ++i) {
		rigid_body_trial( pose );
		if ( accept_rate_ < 0.5 ) {
			trans_magnitude_ *= 0.9;
			rot_magnitude_ *= 0.9;
		} else {
			trans_magnitude_ *= 1.1;
			rot_magnitude_ *= 1.1;
		}
		// if ( jump_out_check() ) return;
	}
	mc_->recover_low( pose );
	TR.flush();
//	pose.energies().show( std::cout );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rigid_body_trial
///
/// @brief Perform a cycle of rigid-body Monte Carlo moves
///
/// @detailed  Performs a number (nattempts) of MC rigid-body moves
///       (of size trans_magnitude, rot_magnitude). The number of successful
///       attempts is stored in accept_rate_ and used in adaptive trials.
///
/// @remarks the success_rate defines
///       whether the translation/rotation size is increased or decreased for
///       the next cycle.
///       currently used only in the low-resolution step (centroid mode)
///
/// @references pose_docking_rigid_body_trial from pose_docking.cc and
///				rigid_body_MC_cycle from dock_structure.cc
///
/// @authors Monica Berrondo October 22 2007
///
/// @last_modified October 22 2007
/////////////////////////////////////////////////////////////////////////////////
void DockingLowRes::rigid_body_trial( core::pose::Pose & pose )
{
	using namespace moves;

	PDBDumpMoverOP dump = new PDBDumpMover("lowres_cycle_");
//	dump->apply( pose );
	MCShowMoverOP mc_show = new MCShowMover( mc_ );
//	mc_show->apply( pose );

	rb_mover_->rot_magnitude( rot_magnitude_ );
	rb_mover_->trans_magnitude( trans_magnitude_ );

	TrialMoverOP rb_trial = new TrialMover( docking_lowres_protocol_, mc_ );
	//rb_trial->set_keep_stats( true );
	rb_trial->keep_stats_type( accept_reject );

	RepeatMoverOP rb_cycle = new RepeatMover( rb_trial, inner_cycles_ );

	rb_cycle->apply( pose );

	pose = mc_->lowest_score_pose();
	//pose.energies().show( std::cout );
	mc_->reset( pose );

	accept_rate_ = rb_trial->acceptance_rate();
}

moves::MonteCarloOP DockingLowRes::get_mc() { return mc_; }


} // namespace docking
} // namespace protocols
