// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet;
//
// ThiS file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   DockingProtocol.cc
///
/// @brief
/// @author Monica Berrondo
/// @author Sid Chaudhury

#include <protocols/docking/DockingProtocol.hh>
#include <protocols/docking/RestrictTaskForDocking.hh>
#include <protocols/docking/SidechainMinMover.hh>
#include <protocols/docking/SidechainMinMover.fwd.hh>

#include <protocols/ScoreMap.hh>

#include <core/chemical/ChemicalManager.hh>

#include <core/options/option.hh>

#include <core/kinematics/FoldTree.hh>

#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/TaskFactory.fwd.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/task/operation/OperateOnCertainResidues.hh>
#include <core/pack/task/operation/ResLvlTaskOperations.hh> // PreventRepackingRLT
#include <core/pack/task/operation/ResFilters.hh> // ResidueLacksProperty
#include <core/pack/task/operation/NoRepackDisulfides.hh>
#include <core/pack/task/operation/ResFilters.hh> // ResidueLacksProperty

#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>
#include <core/pose/datacache/CacheableDataType.hh>

#include <core/scoring/rms_util.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/util/datacache/BasicDataCache.hh>
#include <core/util/datacache/DiagnosticData.hh>

#include <core/types.hh>

#include <protocols/docking/DockingInitialPerturbation.hh>
#include <protocols/docking/DockingLowRes.hh>
#include <protocols/docking/DockingHighRes.hh>
//#include <protocols/docking/DockingPackRotamers.fwd.hh>
//#include <protocols/docking/DockingPackRotamers.hh>

#include <protocols/geometry/RB_geometry.hh>

#include <protocols/moves/ConstraintSetMover.fwd.hh>
#include <protocols/moves/ConstraintSetMover.hh>
#include <protocols/moves/Mover.fwd.hh>
#include <protocols/moves/OutputMovers.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/PackRotamersMover.fwd.hh>
#include <protocols/moves/SwitchResidueTypeSetMover.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <protocols/moves/ScoreMover.hh>
#include <protocols/moves/ReturnSidechainMover.hh>
#include <protocols/moves/RotamerTrialsMinMover.hh>
#include <protocols/moves/RotamerTrialsMinMover.fwd.hh>

#include <protocols/viewer/viewers.hh>
//for resfile reading
#include <core/options/keys/packing.OptionKeys.gen.hh>
#include <core/options/option.hh>

#include <utility/file/FileName.hh>
#include <utility/file/file_sys_util.hh>

#include <ObjexxFCL/FArray1D.hh>

#include <core/util/Tracer.hh>
using core::util::T;

// option key includes

#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/cluster.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>
#include <core/options/keys/docking.OptionKeys.gen.hh>
#include <core/options/keys/constraints.OptionKeys.gen.hh>

using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.docking.DockingProtocol");
using namespace core;

namespace protocols {
namespace docking {

DockingProtocol::DockingProtocol() :
		Mover(), rb_jump_(1)
{
		Mover::type( "DockingProtocol" );
		set_default();
}

DockingProtocol::DockingProtocol(Size const rb_jump_in) :
		Mover(), rb_jump_(rb_jump_in)
{
		Mover::type( "DockingProtocol" );
		set_default();
}



DockingProtocol::DockingProtocol(
		Size const rb_jump_in,
		bool const fullatom,
		bool const local_refine,
		bool const view
	  ) : Mover()
{
		Mover::type( "DockingProtocol" );
		set_default();
		rb_jump_ = rb_jump_in;
		set_fullatom(fullatom);
		set_local_refine(local_refine);
		set_view(view);
		set_autofoldtree(true);
}

DockingProtocol::DockingProtocol(
		Size const rb_jump_in,
		bool const fullatom,
		bool const local_refine,
		bool const view,
		core::scoring::ScoreFunctionCOP docking_score_low,
		core::scoring::ScoreFunctionCOP docking_score_high,
		bool const autofoldtree
	  ) : Mover()
{
		Mover::type( "DockingProtocol" );
		set_default();
		rb_jump_ = rb_jump_in;
		set_fullatom(fullatom);
		set_local_refine(local_refine);
		set_view(view);
		set_autofoldtree(autofoldtree);

		docking_score_low_ = docking_score_low;
		docking_score_high_ = docking_score_high;
		docking_score_high_min_ = docking_score_high;
		docking_score_pack_ = docking_score_high;

}

//destructor
DockingProtocol::~DockingProtocol() {}

//clone
protocols::moves::MoverOP
DockingProtocol::clone() const {
		return( new DockingProtocol( rb_jump_, fullatom_, local_refine_, view_, docking_score_low_, docking_score_high_, autofoldtree_ ) );
	}

//set functions
void DockingProtocol::set_view( bool view_in ) { view_=view_in; }
void DockingProtocol::set_rtmin( bool rtmin_in ) { rtmin_=rtmin_in; }

void DockingProtocol::set_fullatom( bool const fullatom_in )
{
		fullatom_ = fullatom_in;
		if (!fullatom_) passed_highres_filter_ = true;
}

void DockingProtocol::set_local_refine( bool const local_refine_in )
{
		local_refine_ = local_refine_in;
		if (local_refine_){
			set_fullatom(true);
			passed_lowres_filter_ = true;
			}
}

void DockingProtocol::set_lowres_scorefxn( core::scoring::ScoreFunctionCOP docking_score_low_in )
{
		docking_score_low_ = docking_score_low_in;
		docking_low_ = 0;
}

void DockingProtocol::set_highres_scorefxn( core::scoring::ScoreFunctionCOP docking_score_high_in )
{
		docking_score_high_ = docking_score_high_in;
		docking_score_high_min_ = docking_score_high_in;
		docking_high_ = 0;
}

void DockingProtocol::set_highres_scorefxn(
	core::scoring::ScoreFunctionCOP docking_score_high_in,
	core::scoring::ScoreFunctionCOP docking_score_pack_in )
{
		docking_score_high_ = docking_score_high_in;
		docking_score_high_min_ = docking_score_high_in;
		docking_score_pack_ = docking_score_pack_in;
		docking_high_ = 0;
}

void DockingProtocol::set_autofoldtree( bool const autofoldtree ) { autofoldtree_ = autofoldtree; }

void
DockingProtocol::set_default()
{
	using namespace core::options;
	using namespace core::scoring;

	passed_lowres_filter_ = false;
	passed_highres_filter_ = false;

	// score function setup
	docking_score_low_ = ScoreFunctionFactory::create_score_function( "interchain_cen" ) ;
	docking_score_high_ = ScoreFunctionFactory::create_score_function( "docking" );
	docking_score_high_min_ = ScoreFunctionFactory::create_score_function( "docking", "docking_min" ) ;
	docking_score_pack_ = core::scoring::ScoreFunctionFactory::create_score_function( "standard") ;

	set_fullatom(option[ OptionKeys::out::file::fullatom ]());
	set_local_refine(option[ OptionKeys::docking::docking_local_refine ]());

	set_rtmin(option[ OptionKeys::docking::dock_rtmin ]());

	if (option[ OptionKeys::docking::dock_ppk ]()) set_local_refine(true);
	set_view(option[ OptionKeys::docking::view ]());
	set_autofoldtree(true);

}

///@brief setup the docking fold tree
///@details ensure that the fold tree is set up such that the jump points
///		are at the center of masses of the two partners
//
void
DockingProtocol::setup_foldtree( pose::Pose & pose, std::string const partner_chainID)
{

	using namespace core::options;

	// identify the chainIDs for partner1 and partner2
	TR.Debug << "Get the jump point for jump " << rb_jump_ << "............." << std::endl;

	core::pose::PDBInfoCOP pdb_info = pose.pdb_info();
	char second_chain = '_';
	Size cutpoint = 0;

	using namespace kinematics;
	FoldTree f( pose.fold_tree() );

	// determine cutpoint for rb_jump_
	if ( partner_chainID == "_"){//chains not specified
		if (f.num_jump()){	//if a jump exists, use rb_jump_ for cutpoint
			cutpoint = f.cutpoint_by_jump(rb_jump_);
			} else {	//otherwise use second chain
			second_chain = pdb_info->chain( pose.total_residue() );
			}
		} else {//chains specified
			for (Size i=1; i<=partner_chainID.length()-1; i++){ //identify second chain from input partner_chainID
				if (partner_chainID[i-1] == '_') second_chain = partner_chainID[i];
				}

			for ( Size i=2; i<= pose.total_residue(); ++i ) {
				if(pdb_info->chain( i ) == second_chain){ //identify cutpoint corresponding to second chain in partner_chainID
					cutpoint = i-1;
					break;
					}
				}
			}

	// calculate jump points from partners
	runtime_assert( cutpoint );
	TR.Debug << "cutpoint: " << cutpoint << std::endl;
	Size jump_pos1 ( geometry::residue_center_of_mass( pose, 1, cutpoint ) );
	TR.Debug << "jump1: " << jump_pos1 << std::endl;
	Size jump_pos2 ( geometry::residue_center_of_mass( pose, cutpoint+1, pose.total_residue() ) );
	TR.Debug << "jump2: " << jump_pos2 << std::endl;

	// build docking fold tree
	// SJF The default foldtree (when the pose is read from disk) sets all the jumps from the first residue to the first residue of each chain.
	// We want a rigid body jump to be between centres of mass of the two partners (jump_pos1, 2) and all of the rest of the jumps to be
	// sequential so that all of the chains are traversed from the jump position onwards
	// default tree. pas simple...

	//SC reset the jump that is based on the cutpoint, use all other jumps to connect other chains sequentially
	for (Size i=1; i<=f.num_jump(); i++){
		Size current_cutpoint = f.cutpoint_by_jump(i);
		if( current_cutpoint  == cutpoint){
			rb_jump_ = i;
			f.slide_jump( rb_jump_, jump_pos1, jump_pos2 );
			} else {
				f.slide_jump( i, current_cutpoint, current_cutpoint+1);
				}
		}


	//	// This will only work on foldtrees where there are multiple protein chains. It will connect all non-rb_jump_ and non-ligand chains
//	using namespace kinematics;
//	//FoldTree f( pose.fold_tree() );
//	f.slide_jump( rb_jump_, jump_pos1, jump_pos2 );
//// connect all other chains sequentially. First going downstream from jump position
//	for( core::Size jump=rb_jump_-1; jump>=1; --jump ){
//		core::Size const downstream( f.downstream_jump_residue( jump+1 ) );
//		f.slide_jump( jump, downstream - 1, downstream );
//	}
//// and then upstream from jump position
//	for( core::Size jump=rb_jump_+1; jump<=f.num_jump(); ++jump ){
//		core::Size const downstream( f.downstream_jump_residue( jump ) );
//		if( !pose.residue( downstream ).is_polymer() ) continue; // if not polymer -> ligand
//		f.slide_jump( jump, downstream - 1, downstream );
//	}
//
	f.reorder( 1 );
	f.check_fold_tree();
	runtime_assert( f.check_fold_tree() );
	pose.fold_tree( f );
}

void
DockingProtocol::setup_foldtree( pose::Pose & pose )
{
	using namespace core::options;
	std::string const partner_chainID = option[ OptionKeys::docking::partners ]();
	setup_foldtree(pose, partner_chainID);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Docking apply
///
/// @brief main docking protocol
///
/// @detailed
/// Main function for docking. Includes the following steps:
///      0) prepack mode: prepare a starting structure for later runs
///   OR:
///      1) perturbation of decoy (see docking_perturb_decoy): changes
///         orientation of docking partners
///      2) low-resolution search:
///         refine structure with rigid-body, centroid-mode MC cycles
///      3) high-resolution search:
///         further refinement of structure in fullatom mode
void
DockingProtocol::apply( pose::Pose & pose )
{
	using namespace scoring;
	using namespace options;
	using namespace moves;
	using core::pose::datacache::CacheableDataType::SCORE_MAP;
	using utility::file::FileName;

	runtime_assert( (autofoldtree_ == true) || (autofoldtree_ == false) );
	if( autofoldtree_ ) {
		TR << "Setting docking foldtree" << std::endl;
		TR << "old fold tree: " << pose.fold_tree() << std::endl;
		setup_foldtree( pose );
		TR << "new fold tree: " << pose.fold_tree() << std::endl;
	}

	//initialize docking protocol movers
	if (!docking_low_) docking_low_ = new DockingLowRes( docking_score_low_, rb_jump_ );
	if (!docking_high_) docking_high_ = new DockingHighRes( docking_score_high_min_, docking_score_pack_, rb_jump_ );

	// Residue movers
	SwitchResidueTypeSetMover to_centroid( core::chemical::CENTROID );
	SwitchResidueTypeSetMover to_all_atom( core::chemical::FA_STANDARD );
	ReturnSidechainMover recover_sidechains( *get_input_pose());

	core::scoring::ScoreFunctionOP docking_scorefxn;

	util::prof_reset();

	TR.Debug << "fold tree: " << pose.fold_tree() << std::endl;

	core::pose::Pose starting_pose = pose;

	if ( option[ OptionKeys::constraints::cst_file ].user() ){
		ConstraintSetMoverOP docking_constraint = new ConstraintSetMover();
		docking_constraint->apply( starting_pose );

		core::scoring::ScoreFunctionOP docking_scorefxn_cst;
		docking_scorefxn_cst = new core::scoring::ScoreFunction( *docking_score_low_ ) ;
		docking_scorefxn_cst->set_weight(core::scoring::atom_pair_constraint, 1.0);
		set_lowres_scorefxn(docking_scorefxn_cst);
		}

	if ( option[ OptionKeys::run::score_only ]() ) {
		score_only( pose );
		return;
		}

	core::Size const max_repeats( option[ OptionKeys::docking::max_repeats ]() );

	//start loop of decoy creation until filters are all passed
	for (Size r = 1; r <= max_repeats; r++){
		pose = starting_pose;

		//MonteCarloOP mc;
		if ( !local_refine_ ) {
			docking_scorefxn = new core::scoring::ScoreFunction( *docking_score_low_ ) ;

			// first convert to centroid mode
			to_centroid.apply( pose );

			// make starting perturbations based on command-line flags
			DockingInitialPerturbation initial( rb_jump_, true /*slide into contact*/ );
			initial.apply( pose );
			TR << "finished initial perturbation" << std::endl;

			score_map_["st_rmsd"] = calc_Lrmsd( pose, *get_native_pose() );

			// low resolution docking

			TR.Debug << "DockingLowRes object created" << std::endl;
			docking_low_->apply( pose );

			//check low res filter to see if it should repeat low res or not
			passed_lowres_filter_ = docking_lowres_filter( pose );

			// add scores to map for output
			ScoreMap::nonzero_energies( score_map_, docking_scorefxn, pose );
			score_map_["cen_rms"] = calc_Lrmsd( pose, *get_native_pose() );
			}

		// only do this is full atom is true
		if ( fullatom_ && passed_lowres_filter_ ) {

			docking_scorefxn =  new core::scoring::ScoreFunction( *docking_score_high_ ) ;

			if (!local_refine_ || !pose.is_fullatom()){
				to_all_atom.apply( pose );
				recover_sidechains.apply( pose );
			//	recover_sidechains( pose, *get_input_pose());
				}

			// run high resolution docking
			TR.Debug << "DockingHighRes object created" << std::endl;
			docking_high_->apply( pose );

			// add scores to map for output
			ScoreMap::nonzero_energies( score_map_, docking_scorefxn, pose );
			score_map_["I_sc"] = calc_interaction_energy( pose );


			// check highres docking filter
			passed_highres_filter_ = docking_highres_filter( pose );
			}

		if ( passed_lowres_filter_ && passed_highres_filter_ ) break;
		  else  TR<<"REPEAT STRUCTURE "<< r <<std::endl;

		}//end of repeat decoy creation

	// calculate and store the rms no matter which mode was used
	score_map_["rms"] = calc_Lrmsd( pose, *get_native_pose() );
	score_map_["I_sc"] = calc_interaction_energy( pose );

	if (pose.is_fullatom()){
		score_map_["Irms"] = calc_Irmsd(pose, *get_native_pose());
		score_map_["Fnat"] = calc_Fnat(pose, *get_native_pose());
		}

	if ( option[ OptionKeys::run::debug ]() ) util::prof_show();

	// cache the score map to the pose
	pose.data().set(SCORE_MAP, new core::util::datacache::DiagnosticData(score_map_));
}

bool
DockingProtocol::docking_lowres_filter( core::pose::Pose & pose){

	using namespace scoring;
	using namespace options;

	bool passed_filter = true;
	if ( !option[ OptionKeys::docking::fake_native ]() ){
		//criterion for failure
		Real interchain_contact_cutoff	= 10.0;
		Real interchain_vdw_cutoff = 1.0;

		if( option[ OptionKeys::docking::dock_lowres_filter ].user() ) {
			utility::vector1< Real > dock_filters = option[ OptionKeys::docking::dock_lowres_filter ]();
			interchain_contact_cutoff = dock_filters[1];
			interchain_vdw_cutoff = dock_filters[2];
			}

		if (pose.energies().total_energies()[ interchain_contact ] >= interchain_contact_cutoff ) passed_filter = false;
		if (pose.energies().total_energies()[ interchain_vdw ] >= interchain_vdw_cutoff ) passed_filter = false;

		if ( option[ OptionKeys::constraints::cst_file ].user() ){
			Real distance_constraint_cutoff = 1.0; //distance constraint is soft for now
			if (pose.energies().total_energies()[ atom_pair_constraint ] >= distance_constraint_cutoff ) passed_filter = false;
			}
		}

	if (!passed_filter) TR << "STRUCTURE FAILED LOW-RES FILTER" << std::endl;

	return passed_filter;
}

bool
DockingProtocol::docking_highres_filter( core::pose::Pose & pose){

	using namespace scoring;
	using namespace options;

	bool passed_filter = true;
	if ( option[ OptionKeys::docking::dock_ppk ]() ) return passed_filter;

	Real score_cutoff = option[ OptionKeys::cluster::output_score_filter ]();
	//criterion for failure
	if (pose.energies().total_energy() >= score_cutoff) passed_filter = false;
	if (score_map_["I_sc"] >= 0.0) passed_filter = false;

	if (!passed_filter) TR << "STRUCTURE FAILED HIGH-RES FILTER" << std::endl;

	return passed_filter;
}

core::Real
DockingProtocol::calc_interaction_energy( const core::pose::Pose & pose ){
        using namespace scoring;

        core::scoring::ScoreFunctionOP docking_scorefxn;
        core::pose::Pose complex_pose = pose;

        Real trans_magnitude = 1000;
        moves::RigidBodyTransMoverOP translate_away ( new moves::RigidBodyTransMover( complex_pose, rb_jump_ ) );
        translate_away->step_size( trans_magnitude );

        if ( pose.is_fullatom() ){
                docking_scorefxn = new core::scoring::ScoreFunction( *docking_score_high_ ) ;
        } else {
                docking_scorefxn = new core::scoring::ScoreFunction( *docking_score_low_ ) ;
        }

        Real bound_energy = (*docking_scorefxn)( complex_pose );
        translate_away->apply( complex_pose );
        Real unbound_energy = (*docking_scorefxn)( complex_pose );

        return (bound_energy - unbound_energy);

}

core::Real
DockingProtocol::calc_Lrmsd( const core::pose::Pose & pose, const core::pose::Pose & native_pose ){
        using namespace scoring;

        FArray1D_bool temp_part ( pose.total_residue(), false );
        FArray1D_bool superpos_partner ( pose.total_residue(), false );
        /// this gets the wrong partner, therefore it is stored in a temporary
        /// array and then the opposite is put in the actualy array that is used
        /// for superpositioning.  there is probably a better way to do this
        /// need to check TODO
        pose.fold_tree().partition_by_jump( rb_jump_, temp_part );
        for ( Size i=1; i<= pose.total_residue(); ++i ) {
                if (temp_part(i)) superpos_partner(i)=false;
                else superpos_partner(i)=true;
        }

        core::Real Lrmsd = rmsd_no_super_subset( native_pose, pose, superpos_partner, is_protein_CA );
        return Lrmsd;
}

core::Real
DockingProtocol::calc_Irmsd( const core::pose::Pose & pose, const core::pose::Pose & native_pose ){
        using namespace scoring;

	if (!pose.is_fullatom() || !native_pose.is_fullatom()){
		TR << "Irmsd calc called with non-fullatom pose!!!"<<std::endl;
		return 0.0;
	}

        core::pose::Pose native_docking_pose = native_pose;
	using namespace kinematics;
	FoldTree ft( pose.fold_tree() );
	native_docking_pose.fold_tree(ft);

	//score to set up interface object
	core::scoring::ScoreFunctionOP scorefxn = new core::scoring::ScoreFunction( *docking_score_high_ ) ;
        (*scorefxn)( native_docking_pose );

	core::conformation::Interface interface( rb_jump_ );
	interface.distance( 8.0 );
	interface.calculate( native_docking_pose );
        FArray1D_bool is_interface ( pose.total_residue(), false );

        for ( Size i=1; i<= pose.total_residue(); ++i ) {
		if (interface.is_interface(i)) is_interface(i) = true;
		}

	core::Real Irmsd = rmsd_with_super_subset(native_docking_pose, pose, is_interface, is_heavyatom);
	return Irmsd;
	}

core::Real
DockingProtocol::calc_Fnat( const core::pose::Pose & pose, const core::pose::Pose & native_pose ){
        using namespace scoring;
        using namespace conformation;

	if (!pose.is_fullatom() || !native_pose.is_fullatom()){
		TR << "Fnat calc called with non-fullatom pose!!!"<<std::endl;
		return 0.0;
		}

        core::pose::Pose native_docking_pose = native_pose;
	using namespace kinematics;
	FoldTree ft( pose.fold_tree() );
	native_docking_pose.fold_tree(ft);
	Real cutoff = 5.0;

	//score to set up interface object
	core::scoring::ScoreFunctionOP scorefxn = new core::scoring::ScoreFunction( *docking_score_high_ ) ;
        (*scorefxn)( native_docking_pose );

	FArray1D_bool temp_part ( pose.total_residue(), false );
        pose.fold_tree().partition_by_jump( rb_jump_, temp_part );

	utility::vector1< Size > partner1;
	utility::vector1< Size > partner2;

	Interface interface( rb_jump_ );
	interface.distance( 8.0 );
	interface.calculate( native_docking_pose );

	//generate list of interface residues for partner 1 and partner 2
	for ( Size i=1; i <= pose.total_residue(); i++){
		if (interface.is_interface(i)){
			if (!temp_part(i)) partner1.push_back(i);
			if (temp_part(i)) partner2.push_back(i);
			}
		}

	//create contact pair list
	FArray2D_bool contact_list(partner1.size(), partner2.size(), false);

	//identify native contacts across the interface
	//this will probably be changed to use PoseMetrics once I figure out how to use them - Sid
	for ( Size i=1; i<= partner1.size(); i++){
		ResidueOP rsd1 = new Residue(native_docking_pose.residue(partner1[i]));
		for ( Size j=1; j<=partner2.size();j++){
			ResidueOP rsd2 = new Residue(native_docking_pose.residue(partner2[j]));
			contact_list(i,j)=calc_res_contact(rsd1, rsd2, cutoff);
			}
		}

	Real native_ncontact = 0;
	Real decoy_ncontact = 0;

	//identify which native contacts are recovered in the decoy
	for ( Size i=1; i<=partner1.size(); i++){
		for ( Size j=1; j<=partner2.size(); j++){
			if (contact_list(i,j)){
				native_ncontact++;
				ResidueOP rsd1 = new Residue(pose.residue(partner1[i]));
				ResidueOP rsd2 = new Residue(pose.residue(partner2[j]));
				if (calc_res_contact(rsd1, rsd2, cutoff)) decoy_ncontact++;
				}
			}
		}

	Real fnat = decoy_ncontact/native_ncontact;
	return fnat;
}

bool DockingProtocol::calc_res_contact(
	conformation::ResidueOP rsd1,
	conformation::ResidueOP rsd2,
	Real dist_cutoff
)
{
	Real dist_cutoff2 = dist_cutoff*dist_cutoff;

	for (Size m=1; m<=rsd1->nheavyatoms(); m++){
		for (Size n=1; n<=rsd2->nheavyatoms(); n++){
			double dist2 = rsd1->xyz(m).distance_squared( rsd2->xyz(n) );
			if (dist2 <= dist_cutoff2) return true;
			}
		}

	return false;
}


void
DockingProtocol::recover_sidechains( core::pose::Pose & pose, const core::pose::Pose & native_pose )
{
	protocols::moves::ReturnSidechainMover recover_sidechains( native_pose );
	recover_sidechains.apply( pose );

	TR << "Doing initial repack of sidechains" << std::endl;

	//  Do initial pack over all residues within 1000A of the interface.

	using namespace moves;
	using namespace core::pack::task;
	using namespace core::pack::task::operation;
	TaskFactoryOP tf = new TaskFactory;

	tf->push_back( new OperateOnCertainResidues( new PreventRepackingRLT, new ResidueLacksProperty("PROTEIN") ) );
	tf->push_back( new InitializeFromCommandline );
	tf->push_back( new IncludeCurrent );
	tf->push_back( new RestrictToRepacking );
	tf->push_back( new NoRepackDisulfides );
	if( core::options::option[core::options::OptionKeys::packing::resfile].user() ) tf->push_back( new ReadResfile );
	tf->push_back( new protocols::docking::RestrictTaskForDocking( docking_score_pack_, rb_jump_, true ) );

	PackRotamersMoverOP dock_pack = new PackRotamersMover(docking_score_pack_);
	dock_pack->task_factory( tf );
	dock_pack->apply( pose );

	if (rtmin_){
		RotamerTrialsMinMoverOP rtmin_trial = new RotamerTrialsMinMover( docking_score_pack_, tf);
		rtmin_trial->apply( pose );
		}
	if (core::options::option[ core::options::OptionKeys::docking::sc_min ]()){
		InterfaceSidechainMinMoverOP scmin_mover = new InterfaceSidechainMinMover(rb_jump_, docking_score_pack_, 1000);
		scmin_mover->apply(pose);
		}
}

void
DockingProtocol::score_only( core::pose::Pose & pose )
{
	using namespace scoring;
	using namespace moves;
	if ( fullatom_ ) {
		core::scoring::ScoreFunctionOP high_score = new core::scoring::ScoreFunction( *docking_score_high_ );
		ScoreMover score_and_exit( high_score ) ;
		score_and_exit.insert_rms( calc_Lrmsd( pose, *get_native_pose()) );
		score_and_exit.apply( pose );
	} else {
		SwitchResidueTypeSetMover to_centroid( core::chemical::CENTROID );
		to_centroid.apply( pose );
		core::scoring::ScoreFunctionOP low_score = new core::scoring::ScoreFunction( *docking_score_low_ );
		ScoreMover score_and_exit( low_score );
		score_and_exit.insert_rms( calc_Lrmsd( pose, *get_native_pose() ) );
		score_and_exit.apply( pose );
		}
}

}//docking
}//protocols
