// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet;
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   DockingProtocol.hh
///
/// @brief
/// @author Monica Berrondo

#ifndef INCLUDED_protocols_docking_DockingProtocol_HH
#define INCLUDED_protocols_docking_DockingProtocol_HH


#include <core/types.hh>

#include <core/conformation/Residue.fwd.hh>
#include <core/options/option.hh>
#include <core/options/keys/OptionKeys.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/scoring/ScoreFunction.fwd.hh>

#include <protocols/moves/Mover.hh>
#include <protocols/moves/MonteCarlo.fwd.hh>
#include <protocols/moves/MonteCarlo.hh>

#include <protocols/docking/DockingLowRes.hh>
#include <protocols/docking/DockingHighRes.hh>

namespace protocols {
namespace docking {

class DockingProtocol;
typedef utility::pointer::owning_ptr< DockingProtocol > DockingProtocolOP;
typedef utility::pointer::owning_ptr< DockingProtocol const > DockingProtocolCOP;

/// @brief This is a protocol that does the standard RosettaDock protocol on apply.
/// It also contains useful functions for docking such as those that set up a
/// docking fold tree, calculate docking metrics, and docking energies.
class DockingProtocol : public moves::Mover
{
public:
	/// @brief
	/// 	empty constructor fills values with the expected defaults
	///		rb_jump will be assigned as 1 (meaning the first jump will
	///		be used as the jump across which all rigid-body perturbations
	///		will occur

	/// constructor
	DockingProtocol();

	DockingProtocol(Size const rb_jump_in);

	DockingProtocol(Size const rb_jump_in,
		bool const fullatom,
		bool const local_refine,
		bool const view=false
	  );

	DockingProtocol(Size const rb_jump_in,
		bool const fullatom,
		bool const local_refine,
		bool const view,
		core::scoring::ScoreFunctionCOP docking_score_low,
		core::scoring::ScoreFunctionCOP docking_score_high,
		bool const autofoldtree=true
	  );

	//destructor
	~DockingProtocol();

	/// @brief setup that is called from constructor
	void set_default();

	/// @brief setter
	void set_view( bool view_in );

	void set_rtmin( bool rtmin_in );

	void set_fullatom( bool const fullatom_in );

	void set_local_refine( bool const local_refine_in );

	void set_lowres_scorefxn( core::scoring::ScoreFunctionCOP docking_score_low_in );

	void set_highres_scorefxn( core::scoring::ScoreFunctionCOP docking_score_high_in );

	void set_highres_scorefxn(
		core::scoring::ScoreFunctionCOP docking_score_high_in,
		core::scoring::ScoreFunctionCOP docking_score_pack_in );

/// @brief sets up a default docking fold tree for the inputted pose
	void setup_foldtree( core::pose::Pose & pose );
/// @brief sets up a docking fold tree based on user-specified chains for the first and second partner
	void setup_foldtree( core::pose::Pose & pose, std::string const partner_chainID );

	/// @brief should docking try to make an ideal foldtree for docking?

/// @brief triggers a setting up of a docking fold tree
	void set_autofoldtree( bool const autofoldtree );

	bool docking_lowres_filter( core::pose::Pose & pose );
	bool docking_highres_filter( core::pose::Pose & pose );

/// @brief calculates the difference in energy between the inputted complex, and the complex with the two partners at 500A from each other
	core::Real calc_interaction_energy( const core::pose::Pose & pose );
/// @brief calculates C-alpha RMSD of the smaller partner after superposition of the larger partner
	core::Real calc_Lrmsd( const core::pose::Pose & pose, const core::pose::Pose & native_pose );
/// @brief calculates the all-atom RMSD of all residues within 5A of the interface at superposition along those same atoms
	core::Real calc_Irmsd( const core::pose::Pose & pose, const core::pose::Pose & native_pose );
/// @brief calculates the fraction of native contacts recovered between the input and native pose.  A native-contact is defined
/// as defined by a residue from one partner being within 5A of a residue from the other partner in the native structure
	core::Real calc_Fnat( const core::pose::Pose & pose, const core::pose::Pose & native_pose );
	bool calc_res_contact( core::conformation::ResidueOP rsd1, core::conformation::ResidueOP rsd2, core::Real dist_cutoff);

/// @brief recovers the side-chains from the native-pose
	void recover_sidechains( core::pose::Pose & pose, const core::pose::Pose & native_pose );

	void score_only( core::pose::Pose & pose );

	virtual void apply( core::pose::Pose & pose );

	virtual protocols::moves::MoverOP clone() const;
private:
	/// information about the mode
	bool fullatom_;
	bool local_refine_;
	bool rtmin_;
	/// the jump number across which to do rigid_body transformations
	core::Size rb_jump_;
	/// should docking change the foldtree?
	bool autofoldtree_;
	/// whether or not to initialize the viewer (for opengl)
	bool view_;
	bool passed_lowres_filter_;
	bool passed_highres_filter_;
	// for outputting scorefiles
	std::map < std::string, core::Real > score_map_;

	// for scoring
	core::scoring::ScoreFunctionCOP docking_score_low_;
	core::scoring::ScoreFunctionCOP docking_score_high_;
	core::scoring::ScoreFunctionCOP docking_score_high_min_;
	core::scoring::ScoreFunctionCOP docking_score_pack_;

	moves::MonteCarloOP mc_;

	//protocols
	protocols::docking::DockingLowResOP docking_low_;
	protocols::docking::DockingHighResOP docking_high_;
};
} // docking
} // protocols

#endif

