// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file RestrictTaskForDocking.cc
/// @brief When passed to a PackerTask, pack/design is limited to the protein interface
/// @author ashworth

#include <protocols/docking/RestrictTaskForDocking.hh>

#include <core/conformation/Residue.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/conformation/Interface.hh>

namespace protocols {
namespace docking {

using namespace core;
using namespace scoring;
using namespace pack;

RestrictTaskForDocking::RestrictTaskForDocking()
	: parent(),
		scorefxn_( 0 ),
		rb_jump_( 0 ),
		include_current_( true ),
		distance_( 0 )
{}

RestrictTaskForDocking::RestrictTaskForDocking(
	ScoreFunctionCOP scorefxn,
	core::Size rb_jump,
	bool include_current,
	core::Real distance
) : parent(),
		scorefxn_( scorefxn ),
		rb_jump_( rb_jump ),
		include_current_( include_current ),
		distance_( distance )
{}

RestrictTaskForDocking::~RestrictTaskForDocking(){}


task::operation::TaskOperationOP RestrictTaskForDocking::clone() const
{
	return new RestrictTaskForDocking( *this );
}

void
RestrictTaskForDocking::apply(
	pose::Pose const & pose,
	task::PackerTask & task
) const
{
	task.initialize_from_command_line().restrict_to_repacking().or_include_current( include_current_ );

	runtime_assert( scorefxn_ != 0 );
	runtime_assert( rb_jump_ );
	runtime_assert( distance_ );
	// (existing comment) /// why is this still necessary???
//	(*scorefxn_)(pose);
//	scorefxn_->accumulate_residue_total_energies( pose );

	core::conformation::Interface interface( rb_jump_ );
	interface.distance( distance_ );
	interface.calculate( pose );
	interface.set_pack( pose, &task );
}

DockingNoRepack1::DockingNoRepack1()
	: parent(), rb_jump_(1)
{}

DockingNoRepack1::DockingNoRepack1( core::Size rb_jump_in )
	: parent(), rb_jump_(rb_jump_in)
{}

DockingNoRepack1::~DockingNoRepack1(){}

task::operation::TaskOperationOP DockingNoRepack1::clone() const
{
	return new DockingNoRepack1( *this );
}

void
DockingNoRepack1::apply(
	pose::Pose const & pose,
	task::PackerTask & task
) const
{
Size cutpoint ( pose.fold_tree().cutpoint_by_jump( rb_jump_ ) );
		for ( Size ii = 1 ; ii <= cutpoint; ++ii ) {
			 task.nonconst_residue_task( ii ).prevent_repacking();
		}
}

DockingNoRepack2::DockingNoRepack2()
	: parent(), rb_jump_(1)
{}

DockingNoRepack2::DockingNoRepack2( core::Size rb_jump_in )
	: parent(), rb_jump_(rb_jump_in)
{}

DockingNoRepack2::~DockingNoRepack2(){}

task::operation::TaskOperationOP DockingNoRepack2::clone() const
{
	return new DockingNoRepack2( *this );
}

void
DockingNoRepack2::apply(
	pose::Pose const & pose,
	task::PackerTask & task
) const
{
Size cutpoint ( pose.fold_tree().cutpoint_by_jump( rb_jump_ ) );
		for ( Size ii = cutpoint+1 ; ii <= pose.total_residue(); ++ii ) {
			 task.nonconst_residue_task( ii ).prevent_repacking();
		}
}

RestrictToInterface::RestrictToInterface()
	: parent(),
		rb_jump_( 1 ),
		distance_( 8 )
{}

RestrictToInterface::RestrictToInterface(
	core::Size rb_jump_in,
	core::Real distance_in
) : parent(),
		rb_jump_( rb_jump_in ),
		distance_( distance_in )
{}

RestrictToInterface::~RestrictToInterface(){}

task::operation::TaskOperationOP RestrictToInterface::clone() const
{
	return new RestrictToInterface( *this );
}

void
RestrictToInterface::apply(
	pose::Pose const & pose,
	task::PackerTask & task
) const
{

	core::conformation::Interface interface( rb_jump_ );
	interface.distance( distance_ );
	interface.calculate( pose );

	for ( Size ii=1; ii<=pose.total_residue(); ++ii ) {
    if ( !interface.is_interface(ii) || pose.residue(ii).is_ligand() ) {
      task.nonconst_residue_task( ii ).prevent_repacking();
    }
  }

}

} // namespace docking
} // namespace protocols

