// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Sid Chaudhury

// Unit headers
#include <protocols/docking/SidechainMinMover.fwd.hh>
#include <protocols/docking/SidechainMinMover.hh>

// Project headers
#include <core/conformation/Interface.hh>

#include <core/types.hh>

#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/PackerTask.fwd.hh>
#include <core/pack/task/TaskFactory.fwd.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/task/operation/OperateOnCertainResidues.hh>
#include <core/pack/task/operation/ResLvlTaskOperations.hh> // PreventRepackingRLT
#include <core/pack/task/operation/ResFilters.hh> // ResidueLacksProperty

#include <core/pose/Pose.hh>
#include <core/kinematics/MoveMap.fwd.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/options/option.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>
#include <core/options/keys/docking.OptionKeys.gen.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/moves/MinMover.fwd.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/PackRotamersMover.fwd.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/docking/RestrictTaskForDocking.hh>

// Utility Headers
#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.docking.SidechainMinMover");

namespace protocols {
namespace docking {

//default constructor
SidechainMinMover::SidechainMinMover() : moves::Mover()
{
		Mover::type( "SidechainMin" );
		update_movemap_ = true;
		set_default_options();
}

//construtor with arguments
SidechainMinMover::SidechainMinMover( core::scoring::ScoreFunctionCOP scorefxn_in) : moves::Mover(), scorefxn_(scorefxn_in)
{
	Mover::type( "SidechainMin" );
	set_default_options();
}

//construtor with arguments
SidechainMinMover::SidechainMinMover(
	core::scoring::ScoreFunctionCOP scorefxn_in,
	core::kinematics::MoveMapOP movemap_in
) : moves::Mover(), scorefxn_(scorefxn_in), movemap_(movemap_in), update_movemap_(false)
{
	Mover::type( "SidechainMin" );
	set_default_options();
}

//constructor with arguments
SidechainMinMover::SidechainMinMover(
	core::scoring::ScoreFunctionCOP scorefxn_in,
	core::pack::task::PackerTaskOP task_in) : moves::Mover(), scorefxn_(scorefxn_in), task_(task_in), update_movemap_(true)
{
	Mover::type( "SidechainMin" );
	set_default_options();
}

//constructor with arguments
SidechainMinMover::SidechainMinMover(
	core::scoring::ScoreFunctionCOP scorefxn_in,
	core::pack::task::TaskFactoryOP tf_in) : moves::Mover(), scorefxn_(scorefxn_in), tf_(tf_in), update_movemap_(true)
{
	Mover::type( "SidechainMin" );
	set_default_options();
}

//destructor
SidechainMinMover::~SidechainMinMover() {}

void SidechainMinMover::set_minmover( moves::MinMoverOP minmover_in ){ minmover_ = minmover_in; }

//default options setup for SidechainMinMover
void SidechainMinMover::set_default_options()
{
	if(update_movemap_){
		movemap_ = new core::kinematics::MoveMap();
		movemap_->set_chi( true );
		}

	scorefxn_ = core::scoring::ScoreFunctionFactory::create_score_function("standard");
	minmover_ = new moves::MinMover(movemap_, scorefxn_, "dfpmin_armijo_nonmonotone", 0.01, true/*nblist*/, false/*deriv_check*/  );
}

//SidechainMinMover updates the movemap based on a packer task
void SidechainMinMover::update_movemap( core::pose::Pose & pose)
{
	movemap_->set_chi( true );
	if (tf_) task_ = tf_->create_task_and_apply_taskoperations( pose );
	if (task_){
		for(Size i = 1; i <= pose.total_residue(); i++){
			if (!task_->nonconst_residue_task(i).being_packed()) movemap_->set_chi(i, false);
			}
		}
}

//SidechainMinMover apply function
void SidechainMinMover::apply( core::pose::Pose & pose )
{
	//runtime_assert(pose.is_fullatom());
	if(update_movemap_) update_movemap( pose );
	minmover_->apply( pose );
}

//constructor
InterfaceSidechainMinMover::InterfaceSidechainMinMover() : SidechainMinMover(), rb_jump_( 1), interface_dist_(8.0)
{
			Mover::type( "InterfaceSidechainMinMover" );
			set_default_options();
}

InterfaceSidechainMinMover::InterfaceSidechainMinMover(
		core::Size rb_jump_in,
		core::scoring::ScoreFunctionCOP scorefxn_in,
		core::Real interface_dist_in
	) : SidechainMinMover(scorefxn_in), rb_jump_( rb_jump_in ),interface_dist_(interface_dist_in)
{
		Mover::type( "InterfaceSidechainMin" );
		set_default_options();
}

//destructor
InterfaceSidechainMinMover::~InterfaceSidechainMinMover() {}

void InterfaceSidechainMinMover::set_interface_dist( core::Real interface_dist_in)
{
		interface_dist_ = interface_dist_in;
		interface_->distance(interface_dist_);
}

//default options setup for InterfaceSidechainMinMover
void
InterfaceSidechainMinMover::set_default_options()
{
	interface_ = new core::conformation::Interface( rb_jump_ );

	using namespace core::options;

	set_norepack1(option[ OptionKeys::docking::norepack1 ]());
	set_norepack2(option[ OptionKeys::docking::norepack2 ]());
}

//apply function for InterfaceSidechainMinMover
void
InterfaceSidechainMinMover::apply( core::pose::Pose & pose )
{
	//reset movemap
	movemap_->set_bb(false);
	movemap_->set_jump(false);
	movemap_->set_chi(false);

	//calculate interface
	interface_->calculate( pose );

  Size cutpoint ( pose.fold_tree().cutpoint_by_jump( rb_jump_ ) );

	//set move map for interface residues only
	if ( !no_repack1_ ){
		for(core::Size i = 1; i <= cutpoint; i++){
			if (interface_->is_interface(i)) movemap_->set_chi(i, true);
			}
		}

	if ( !no_repack2_ ){
		for(core::Size i = cutpoint+1; i <= pose.total_residue(); i++){
			if (interface_->is_interface(i)) movemap_->set_chi(i, true);
			}
		}

	minmover_->apply(pose);

}

} // docking
} // protocols
