// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief protocols for folding into density
/// @detailed
/// @author Frank DiMaio

#include <protocols/electron_density/util.hh>
#include <core/scoring/electron_density/util.hh>

#include <core/id/NamedAtomID.hh>
#include <core/id/AtomID.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/scoring/electron_density/ElectronDensity.hh>

#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>

// Symmetry
#include <core/scoring/symmetry/SymmetricScoreFunction.hh>
#include <core/conformation/symmetry/util.hh>
#include <protocols/moves/symmetry/SymMinMover.hh>
#include <protocols/moves/symmetry/SymPackRotamersMover.hh>
#include <protocols/moves/symmetry/SymRotamerTrialsMover.hh>

#include <core/pose/Pose.hh>
#include <core/pose/util.hh>

#include <core/conformation/ResidueFactory.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoringManager.fwd.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/kinematics/MoveMap.hh>

#include <core/options/option.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// option key includes
#include <core/options/keys/edensity.OptionKeys.gen.hh>
#include <core/options/keys/symmetry.OptionKeys.gen.hh>

//
#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;


namespace protocols {
namespace electron_density {

static core::util::Tracer TR("protocols.electron_density.util");

using namespace protocols;
using namespace core;


void set_pose_and_scorefxn_for_edens_scoring( core::pose::Pose & pose, core::scoring::ScoreFunction &scorefxn ) {
	core::pose::addVirtualResAsRoot( pose );
	core::scoring::electron_density::add_dens_scores_from_cmdline_to_scorefxn( scorefxn );
}


///////////////////////////////////////
///////////////////////////////////////


void SetupForDensityScoringMover::apply( core::pose::Pose & pose ) {
	core::pose::addVirtualResAsRoot( pose );
	core::scoring::electron_density::getDensityMap().maskResidues( mask_reses_ );
	last_score = dockPoseIntoMap( pose );
	core::scoring::electron_density::getDensityMap().clearMask(  );
}

void SetupForDensityScoringMover::mask( protocols::loops::Loops const & loops ) {
	mask_reses_.clear();
	for( protocols::loops::Loops::LoopList::const_iterator it=loops.loops().begin(), it_end=loops.loops().end(); it != it_end; ++it )
		for (core::Size i=it->start(), i_end=it->stop(); i<i_end; ++i )
			mask_reses_.push_back(i);
}


///////////////////////////////////////
///////////////////////////////////////


//
// dock pose into a density map
//    respect recentering flags
//    should we ensure we are set up for density scoring?????
core::Real dockPoseIntoMap( core::pose::Pose & pose, std::string align_in /* ="" */ ) {
	using namespace core::options;

	std::string align = align_in;
	if (align.length() == 0)
		align = option[ OptionKeys::edensity::realign ]();

	/////
	// minimization
	// for speed reasons, use elec_dens_whole_structure_allatom to get initial alignment
	core::scoring::ScoreFunctionOP scorefxn_dens = new core::scoring::ScoreFunction();
	if ( core::conformation::symmetry::is_symmetric(pose) )
		scorefxn_dens = new core::scoring::symmetry::SymmetricScoreFunction( scorefxn_dens );
	//scorefxn_dens->set_weight( core::scoring::elec_dens_whole_structure_ca, 1.0 );
	core::scoring::electron_density::add_dens_scores_from_cmdline_to_scorefxn( *scorefxn_dens );

	core::Real dens_score = 0.0;

	if (align == "no") {
		dens_score = (*scorefxn_dens)( pose );
		return dens_score; // do nothing
	}

	/////
	// initial alignment
	if ( align.substr(0,7) == "fastrot") {
		// fast rotational alignment of pose to map
		TR.Error << "[ ERROR ]  edensity::realign = 'fastrot' unimplemented!" << std::endl;
		exit(1);
	} else if ( align.substr(0,9) == "fasttrans" ) {
		// align CoMs of pose and map
		fastTransAlignPose( pose );
	} else if ( align.substr(0,5) == "trans" ) {
		// translational alignment of pose to map
		TR.Error << "[ ERROR ]  edensity::realign = 'trans' unimplemented!" << std::endl;
		exit(1);
	} else if ( align.substr(0,2) == "rt" ) {
		// complete 6D alignment of pose to map
		TR.Error << "[ ERROR ]  edensity::realign = 'rt' unimplemented!" << std::endl;
		exit(1);
	}

	/////
	// minimization
	// special case for symmetric poses
	if ( align.length() >= 3 && align.substr( align.length()-3 ) == "min" ) {
		bool isSymm = core::conformation::symmetry::is_symmetric(pose);

		// get jump index of root jump
		int root = pose.fold_tree().root();
		utility::vector1< core::kinematics::Edge > root_edges = pose.fold_tree().get_outgoing_edges (root);

		core::kinematics::MoveMapOP rbmm = new core::kinematics::MoveMap;
		rbmm->set_bb( false ); rbmm->set_chi( false );
		// TODO? a flag which toggles minimization of:
		//  a) all rigid-body DOFs
		//  b) all symmetric DOFs
		// HOWEVER, if this is done then fa_rep / vdw should be turned on
 		TR << "RBminimizing pose into density alongs jump(s)";
 		for (core::Size i=1; i<=root_edges.size(); ++i) {
 			TR << "  " << root_edges[i].label();
 			rbmm->set_jump ( root_edges[i].label() , true );
 		}
 		TR << std::endl;

		if (isSymm) {
			core::scoring::ScoreFunctionOP symmscorefxn_dens = new core::scoring::symmetry::SymmetricScoreFunction( scorefxn_dens );

			core::conformation::symmetry::make_symmetric_movemap( pose, *rbmm );
			//moves::MoverOP min_mover = new moves::symmetry::SymMinMover( rbmm, symmscorefxn_dens,  "dfpmin_armijo_nonmonotone", 1e-5, true, true, true );
			moves::MoverOP min_mover = new moves::symmetry::SymMinMover( rbmm, symmscorefxn_dens,  "dfpmin_armijo_nonmonotone", 1e-5, true );

			bool densInMinimizer = core::scoring::electron_density::getDensityMap().getUseDensityInMinimizer();
			core::scoring::electron_density::getDensityMap().setUseDensityInMinimizer( true );
			//core::scoring::electron_density::getDensityMap().setUseExactDerivatives( true );
			min_mover->apply( pose );
			core::scoring::electron_density::getDensityMap().setUseDensityInMinimizer( densInMinimizer );

			symmscorefxn_dens->show( TR, pose ); TR<<std::endl;
			dens_score = (*symmscorefxn_dens)( pose );
		} else {
			//moves::MoverOP min_mover = new moves::MinMover( rbmm, scorefxn_dens, "dfpmin_armijo_nonmonotone", 1e-5, true, true, true );
			moves::MoverOP min_mover = new moves::MinMover( rbmm, scorefxn_dens, "dfpmin_armijo_nonmonotone", 1e-5, true );

			bool densInMinimizer = core::scoring::electron_density::getDensityMap().getUseDensityInMinimizer();
			//core::scoring::electron_density::getDensityMap().setUseExactDerivatives( true );
			core::scoring::electron_density::getDensityMap().setUseDensityInMinimizer( true );
			min_mover->apply( pose );
			core::scoring::electron_density::getDensityMap().setUseDensityInMinimizer( densInMinimizer );
			dens_score = (*scorefxn_dens)( pose );
		}
	}
	return dens_score;
}


//
// align pose CoM to density CoM
core::Real fastTransAlignPose(core::pose::Pose & pose) {
	// align CoM of map and fragment
	int  nres( pose.total_residue() ), nAtms = 0;
  numeric::xyzVector<core::Real> massSum(0,0,0), poseCoM, mapCoM, mapOri;
	for ( int i=1; i<= nres; ++i ) {
		conformation::Residue const & rsd( pose.residue(i) );
		if (rsd.aa() == core::chemical::aa_vrt) continue;

		// use every atom
		for ( Size j=1; j<= rsd.nheavyatoms(); ++j ) {
			conformation::Atom const & atom( rsd.atom(j) );
			massSum += atom.xyz();
			nAtms++;
		}
	}
	poseCoM = massSum / (core::Real)nAtms;
	mapCoM = core::scoring::electron_density::getDensityMap().getCoM();
	mapOri = core::scoring::electron_density::getDensityMap().getOrigin();

  numeric::xyzVector<core::Real> translation = -poseCoM+mapCoM+mapOri-1.0;
	TR << "Aligning inital pose to density ... " << std::endl;
	TR << "   translation (structure -> map) = " << translation  << std::endl;

	// apply translation then rotation to each atom in the pose
	for ( int i=1; i<= nres; ++i ) {
		conformation::Residue const & rsd( pose.residue(i) );
		if (rsd.type().aa() == core::chemical::aa_vrt)
			continue;
		for ( Size j=1; j<= rsd.natoms(); ++j ) {
			numeric::xyzVector<core::Real> atom_ij = pose.xyz( id::AtomID(j,i) );
			pose.set_xyz( id::AtomID(j,i) ,  (atom_ij+translation) );
		}
	}
	return 0.0;
}


}
}

