// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file IO-functionality for enzyme design constraints
/// @brief
/// @author Florian Richter, floric@u.washington.edu

// Unit headers
#include <protocols/enzdes/EnzConstraintIO.hh>
#include <protocols/enzdes/EnzConstraintParameters.hh>
//#include <protocols/enzdes/MatchConstraintFileInfo.hh>

// Package headers
#include <protocols/enzdes/EnzCstTemplateRes.hh>
#include <protocols/enzdes/MatchConstraintFileInfo.hh>

// Project headers
#include <core/chemical/ChemicalManager.hh>
#include <core/conformation/Residue.fwd.hh>
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh> //reading remarks
#include <core/pose/PDBPoseMap.hh> //for PDB-info-to-resid functionality
#include <core/io/pdb/file_data.hh> //reading remarks

#include <core/scoring/ScoreFunction.hh> //scoring ambiguous constraints
#include <core/scoring/constraints/Constraint.fwd.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/Constraints.hh>
#include <core/scoring/constraints/ResidueTypeConstraint.hh>

#include <core/pack/task/PackerTask.hh> //favor_native_res
#include <core/options/option.hh> //options

// Utility Headers
//#include <utility/pointer/ReferenceCount.hh>
#include <utility/io/izstream.hh>
#include <utility/string_util.hh>
#include <iostream>
#include <fstream>
#include <string>
#include <sstream>

#include <core/util/Tracer.hh>

// option key includes

#include <core/options/keys/enzdes.OptionKeys.gen.hh>



static core::util::Tracer tr("protocols.enzdes.EnzConstraintIO");

namespace protocols {
namespace enzdes {

EnzConstraintIOOP EnzConstraintIO::generic_instance_ = NULL;

///@ brief constructor for EnzConstraintIO class, builds up function types
EnzConstraintIO::EnzConstraintIO (core::chemical::ResidueTypeSetCAP src_restype_set) {
		restype_set_ = src_restype_set;
		//cstset_ = new core::scoring::constraints::ConstraintSet();
		cst_pair_data_consistent_ = false;
		favor_native_constraints_.clear();

		mcfi_lists_.clear();
}

EnzConstraintIO::~EnzConstraintIO() {}

EnzConstraintIO*
EnzConstraintIO::get_instance(){

	using namespace core::chemical;

	if( !generic_instance_ ){

		generic_instance_ = new EnzConstraintIO( ChemicalManager::get_instance()->residue_type_set( FA_STANDARD ) );
		generic_instance_->read_enzyme_cstfile( core::options::option[core::options::OptionKeys::enzdes::cstfile]);

	}

	return &(*generic_instance_);

} //get_instance


/// @brief reads the enzyme cstfile and for each block of residue residue constraints, creates a new
/// @brief instance of EnzConstraintParameters
void
EnzConstraintIO::read_enzyme_cstfile( std::string fname ) {

	utility::io::izstream data( fname.c_str() );
	std::istringstream line_stream;

	if ( !data ) {
		std::cerr << "ERROR:: Unable to open constraints file: "
				<< fname << std::endl;
		std::exit( 1 );
	}
	tr.Info << "read enzyme constraints from " << fname << " ...";

	EnzConstraintParametersOP parameters;
	MatchConstraintFileInfoListOP mcfil;

	std::string line, key("");
	core::Size counted_blocks(0);

	bool in_variable_block( false );

	while( !data.eof() ) {
		key = "";
		getline(data,line);
		line_stream.clear();
		line_stream.str(line);
		line_stream >> key;

		if( key == "VARIABLE_CST::BEGIN" ){
			in_variable_block = true;
			mcfil = new MatchConstraintFileInfoList( restype_set_ );
		}

		if( key == "VARIABLE_CST::END" ){

			if( !in_variable_block ){
				utility_exit_with_message("Error when reading cstfile. Stray VARIABLE_CST::END tag in file.");
			}
			in_variable_block = false;
			parameters = new EnzConstraintParameters(counted_blocks, restype_set_, this);
			parameters->set_mcfi_list( mcfil );
			cst_pairs_.push_back( parameters );
			mcfi_lists_.push_back( mcfil );
		}

		if( key == "CST::BEGIN" ) {

			counted_blocks++;

			if( !in_variable_block) mcfil = new MatchConstraintFileInfoList( restype_set_ );


			if( mcfil->read_data( data ) ){

				if( !in_variable_block ){
					parameters = new EnzConstraintParameters(counted_blocks, restype_set_, this);
					parameters->set_mcfi_list( mcfil );
					cst_pairs_.push_back( parameters );
					mcfi_lists_.push_back( mcfil );
				}
			}

			else{
				utility_exit_with_message("Undefined error when reading cstfile. Something is wrong with the format (no CST::END tag maybe? ).\n");
			}
		}

	} // file reading

	if( in_variable_block ) utility_exit_with_message("Error when reading cstfile. VARIABLE_CST::BEGIN tag without corresponding VARIABLE_CST::END tag found.");

	tr.Info << " done, " << cst_pairs_.size() << " cst blocks were read." << std::endl;
} //funtion read enzyme cst



MatchConstraintFileInfoListCOP
EnzConstraintIO::mcfi_list( core::Size block ) const
{

	runtime_assert( block <= mcfi_lists_.size() );

	return mcfi_lists_[ block ];
}



/// @brief reads the residue numbers that the constraints will be applied to.
void
EnzConstraintIO::process_pdb_header(
	core::pose::Pose const & pose,
	bool accept_missing_blocks)
{


	core::pose::PDBPoseMap PDB_map( pose.pdb_info()->pdb2pose() );

	std::set< Size > found_cst_blocks;

	std::istringstream line_stream;
	std::map< Size, std::pair< utility::vector1<Size >, utility::vector1< Size > > > cst_block_to_residues;

	//core::util::datacache::BasicDataCache const & pose_cache = pose.data();
	core::pose::PDBInfoCOP pose_pdbinfo = pose.pdb_info();
	core::io::pdb::Remarks const & pose_remarks = pose_pdbinfo->remarks();


	std::string line, buffer(""), tag("");

	Size cst_block(0), counted_blocks(0);

	//std::cerr << "There are " << pose_remarks->size() << " remark lines." << std::endl;

	for( std::vector< core::io::pdb::RemarkInfo >::const_iterator remark_it = pose_remarks.begin(); remark_it != pose_remarks.end(); remark_it++) {

		line_stream.clear();
		line_stream.str( remark_it->value );

		//std::cerr << "this remark string is: " << remark_it->value << std::endl;
		//std::cerr << "and the number is: " << remark_it->num << std::endl;

		line_stream >> buffer >> tag;
		if( tag == "TEMPLATE"){

			std::string resA_type(""), resB_type("");
			Size resA_num(0), resB_num(0), pose_resnumA(0), pose_resnumB(0);
			std::string resA_chain(""),resB_chain("");
			counted_blocks++;

			//Size resA_pose_num(0), resB_pose_num(0);

			line_stream >> resA_chain >> resA_type >> resA_num;
			line_stream >> buffer >> buffer >> resB_chain >> resB_type >> resB_num >> cst_block;

			found_cst_blocks.insert( cst_block );
			//debug output
			//tr.Info << "remark debug " << buffer << tag << resA_chain << resA_type << resA_num;
			//tr.Info << resB_chain << resB_type << resB_num << cst_block << std::endl;
			if(resA_num != 0) {
				pose_resnumA = PDB_map.find(resA_chain[0], resA_num);

				if( pose_resnumA == 0 ) utility_exit_with_message( "residue at chain "+resA_chain+" position "+utility::to_string( resA_num )+" not found in pose.");

			}

			if(resB_num != 0) {
				pose_resnumB = PDB_map.find(resB_chain[0], resB_num);

				if( pose_resnumB == 0 ) utility_exit_with_message( "residue at chain "+resB_chain+" position "+utility::to_string( resB_num )+" not found in pose.");

			}

			// first do sanity checks for format and consistency of REMARKs with actual atom data in the pdb
			if(cst_block == 0 || cst_block > cst_pairs_.size() ) {
				std::cerr << "Error: catalytic map in pdb file and information in cst file don't match. Either there is no correctly formatted info given in the REMARK block, or there are more constraint REMARKS than blocks in the .cst file." << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			if( (resA_chain.size() != 1) || (resB_chain.size() !=1) ) {
				std::cerr << "Error: format in pdb file header is wrong, missing chains of catalytic residues." << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			if( ( (resA_num != 0 ) && (pose.residue(pose_resnumA).name3() != resA_type ) ) ||
				( (resB_num != 0 ) && (pose.residue(pose_resnumB).name3() != resB_type) ) )
			{
				std::cerr << "Error: residue names/positions in catalytic header map in pdb file don't match actual protein residues" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			cst_block--; //starts at 1, while vector starts at 0

			//then get the correct residues in the pose
			//special feature: if res(A/B)_num is 0, all residues with 3 letter name res(A/B)_type will be added


			if(resA_num != 0){
				cst_pairs_[cst_block]->nonconst_resA()->set_position_in_pose( pose_resnumA );
			}
			if(resB_num != 0){
				cst_pairs_[cst_block]->nonconst_resB()->set_position_in_pose( pose_resnumB );
			}
			if( resA_num == 0 || resB_num == 0){

				bool resA_missing( true ), resB_missing( true );
				if( resA_num != 0 ) resA_missing = false;
				if( resB_num != 0 ) resB_missing = false;

				for(Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i){
					if(resA_num == 0 && pose.residue(i).name3() == resA_type){
						cst_pairs_[cst_block]->nonconst_resA()->add_position_in_pose(i);
						resA_missing = false;
					}
					if(resB_num == 0 && pose.residue(i).name3() == resB_type){
						cst_pairs_[cst_block]->nonconst_resB()->add_position_in_pose(i);
						resB_missing = false;
					}
				}
				//make sure that even though the respos was declared 0 in the header,
				//at least one of the demanded type was found
				if( resA_missing ) utility_exit_with_message("Residue with name "+resA_type+" declared in header not found in pose.");
				if( resB_missing ) utility_exit_with_message("Residue with name "+resB_type+" declared in header not found in pose.");

			} //if one of the residue positions is given as 0 in the header

			//resA_pose_num = pose.pdb_info()->pdb2pose(resA_num, resA_chain);
			//resB_pose_num = pose.pdb_info()->pdb2pose(resB_num, resB_chain);

		}//remark line processing done


	}// end file reading

	//start to process missing blocks: try to find position of residues based on information in cst file
	if(counted_blocks != cst_pairs_.size() ){

		for( Size i = 1; i <= cst_pairs_.size(); ++i){

			if( found_cst_blocks.find( i ) == found_cst_blocks.end() ){
				bool resA_missing = !cst_pairs_[i-1]->nonconst_resA()->find_in_pose_if_missing_from_header( pose );
				bool resB_missing = !cst_pairs_[i-1]->nonconst_resB()->find_in_pose_if_missing_from_header( pose );

				if( resA_missing || resB_missing ) {

					//make sure at least one of the residues is in the pose
					if( accept_missing_blocks && ( resA_missing != resB_missing) ){} //cst_pairs_[i]->missing_in_pose_ = true;
					else{
						std::cerr << "Error: catalytic map in pdb file and information in cst file don't match, unequal number of constraints. should be " << cst_pairs_.size() << ", is " << counted_blocks << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}
				}
			}
		}
	} //if counted blocks != cst pairs size

}// end of process pdb header info function


/// @brief prepares the class for reading in data from a pose/pdb
void
EnzConstraintIO::clear_pose_specific_data()
{
	for ( core::Size block = 1; block <= cst_pairs_.size(); ++block ) {
		clear_pose_specific_data_for_block( block );
	}

	favor_native_constraints_.clear();

}//clear pdb specific data function

void
EnzConstraintIO::clear_pose_specific_data_for_block( core::Size cst_block )
{
	cst_pairs_[ cst_block - 1]->clear_pose_specific_data();
}


void
EnzConstraintIO::generate_pose_specific_data(
	core::pose::Pose & pose,
	core::scoring::ScoreFunctionCOP scofx)
{

	for ( core::Size block = 1; block <= cst_pairs_.size(); ++block) {
		generate_pose_specific_data_for_block( pose, scofx, block );
	}
	cst_pair_data_consistent_ = true;
}// end check consistency function



void
EnzConstraintIO::generate_pose_specific_data_for_block(
	core::pose::Pose & pose,
	core::scoring::ScoreFunctionCOP scofx,
	core::Size cst_block)
{

	if( cst_pairs_[ cst_block - 1]->missing_in_pose() ) {
		tr.Info << "Block " << cst_block << " wasn't found in the pose, so no constraints will be generated." << std::endl;
	} else {
		tr.Info << "checking cst data consistency for block " << cst_block << "... ";
		cst_pairs_[ cst_block - 1 ]->generate_pose_specific_data( pose, scofx );

		tr.Info << " done" << std::endl;
	}
}

void
EnzConstraintIO::add_constraints_to_pose(
	core::pose::Pose & pose,
	core::scoring::ScoreFunctionCOP scofx,
	bool accept_blocks_missing_header
)
{

	//in case this function gets called twice, we remove constraints from the pose
	remove_constraints_from_pose( pose, false, false );

	clear_pose_specific_data();

	process_pdb_header( pose, accept_blocks_missing_header );

	if(core::options::option[core::options::OptionKeys::enzdes::enz_debug] ) show_cst_definitions();

	tr.Info << "Generating constraints for pose... " << std::endl;
	//generate_pose_specific_data( pose, scofx );

	for( core::Size block = 1; block <= cst_pairs_.size(); ++block) {

		add_constraints_to_pose_for_block_without_clearing_and_header_processing( pose, scofx, block);

		tr.Info << "Cst Block " << block << "done... " << std::endl;
	}

	//tr.Info << std::endl << "All constraints generated and added. " << std::endl;
} //add_constraints_to_pose functino


void
EnzConstraintIO::add_constraints_to_pose_for_block_without_clearing_and_header_processing(
	core::pose::Pose & pose,
	core::scoring::ScoreFunctionCOP scofx,
	core::Size cst_block
)
{

	generate_pose_specific_data_for_block( pose, scofx, cst_block);

	//debug shit
	//utility::vector1< core::scoring::constraints::ConstraintCOP > pcst = cst_pairs_[ cst_block - 1]->active_pose_constraints();
	//core::Size num_to_add = pcst.size();
	//std::cerr << "about to add " << num_to_add << " constraints for block " << cst_block << " to pose." << std::endl;
	//debug shit over
	using namespace core::scoring::constraints;
	ConstraintCOPs tmp = pose.add_constraints( cst_pairs_[cst_block - 1]->active_pose_constraints() );
	cst_pairs_[ cst_block - 1 ]->active_pose_constraints( tmp );

} // add_constraints_to_pose_for_block_without_clearing_and_header_processing


void
EnzConstraintIO::remove_constraints_from_pose(
	core::pose::Pose & pose,
		bool const keep_covalent,
		bool const fail_on_constraints_missing
)
{
	//std::cerr << "removing constraints from pose" << std::endl;
	for( core::Size block = 1; block <= cst_pairs_.size(); ++block){
		if( !cst_pairs_[ block - 1 ]->missing_in_pose() && (! (keep_covalent && cst_pairs_[ block -1 ]->is_covalent() )) ){
			remove_constraints_from_pose_for_block( pose, block, fail_on_constraints_missing);
		}
	}
} //remove constraints from pose function


void
EnzConstraintIO::remove_constraints_from_pose_for_block(
	core::pose::Pose & pose,
	core::Size cst_block,
	bool const fail_on_constraints_missing
)
{

	bool constraints_found(false);

	if( pose.remove_constraints( cst_pairs_[ cst_block - 1]->active_pose_constraints() )  ){
		constraints_found = true;
	}

	else if( fail_on_constraints_missing ){

		utility::vector1< core::scoring::constraints::ConstraintCOP > pcst = cst_pairs_[ cst_block - 1]->active_pose_constraints();

		std::cerr << "trying to remove the following constraints... " << std::endl;

		for( utility::vector1< core::scoring::constraints::ConstraintCOP >::const_iterator cst_it = pcst.begin();
				 cst_it != pcst.end(); ++cst_it){
			(*cst_it)->show( std::cerr );
		}
		utility_exit_with_message("Error: an enzdes constraint that should be in the pose got lost along the way.\n");

	}

	if( constraints_found && cst_pairs_[cst_block - 1 ]->is_covalent() ){
		cst_pairs_[cst_block - 1 ]->remove_covalent_connections_from_pose( pose );
	}

} //remove_constraints_from_pose_for_block

void
EnzConstraintIO::remove_position_from_template_res_for_block( core::Size pos, core::Size cst_block )
{
	cst_pairs_[ cst_block - 1 ]->remove_position_from_template_res( pos );
}

void
EnzConstraintIO::add_pregenerated_constraints_to_pose(
	core::pose::Pose & pose,
	core::scoring::ScoreFunctionCOP scofx )
{

	using namespace core::scoring::constraints;
	//std::cerr << "adding pregenerated constraints" << std::endl;
	utility::vector1< ConstraintCOP > all_pose_constraints = pose.constraint_set()->get_all_constraints();

	for (std::vector< EnzConstraintParametersOP >::iterator it = cst_pairs_.begin(); it != cst_pairs_.end(); ++it){

		if( (*it)->missing_in_pose() ) continue;

		if( ( ! (*it)->is_empty() ) && ( (*it)->active_pose_constraints().size() == 0 )  ){
			utility_exit_with_message("trying to add pregenerated constraints to the pose even though they haven't been generated yet.");
		}
		bool covalent_kept(false);

		utility::vector1< ConstraintCOP > & cur_active_constraints = (*it)->active_pose_constraints();


		for( utility::vector1< ConstraintCOP >::iterator cst_it = cur_active_constraints.begin();
				 cst_it != cur_active_constraints.end(); ++cst_it ){

			utility::vector1< ConstraintCOP >::iterator cst_find = find(all_pose_constraints.begin(), all_pose_constraints.end(), *cst_it);

			if( cst_find != all_pose_constraints.end() ){
				if( ! (*it)->is_covalent() ){
					tr << "WARNING: tried to add an enzdes constraint that's already in the pose. Something's a bit unclean somewhere." << std::endl;
				}
				else covalent_kept = true;
			}
			else{
				//std::cerr << "There are a total of " << cur_active_constraints.size() << " constraints in this parameter.\n" << std::endl;
				//std::cerr << "showing definition for a constraint with " << (*cst_it)->natoms() << " atoms... ";
				//(*cst_it)->show( std::cerr );
				if( ! (*it)->is_covalent() ) *cst_it = pose.add_constraint( *cst_it );
			}

		}
		if( !covalent_kept && (*it)->is_covalent() ) {
			add_constraints_to_pose_for_block_without_clearing_and_header_processing( pose, scofx, (*it)->cst_block() );
		}

	} //loop over cst_pairs

} //add pregenerated constraints function


bool
EnzConstraintIO::contains_position( core::Size const seqpos ) const
{

	for (std::vector< EnzConstraintParametersOP >::const_iterator it = cst_pairs_.begin(); it != cst_pairs_.end(); ++it) {
		if ( (*it)->contains_position( seqpos ) ) return true;

	}
	return false;

}


bool
EnzConstraintIO::is_backbone_only_cst( core::Size const seqpos ) const
{

	bool to_return(false);

	for (std::vector< EnzConstraintParametersOP >::const_iterator it = cst_pairs_.begin(); it != cst_pairs_.end(); ++it) {

		if ( (*it)->resA()->contains_position( seqpos ) ){
			if( (*it)->resA()->is_backbone() ) to_return = true;
			else return false;
		}

		if ( (*it)->resB()->contains_position( seqpos ) ){
			if( (*it)->resB()->is_backbone() ) to_return = true;
			else return false;
		}
	}
	return to_return;
}

void
EnzConstraintIO::update_pdb_remarks_for_backbone_params(
	core::pose::Pose & pose )
{

	for (std::vector< EnzConstraintParametersOP >::const_iterator it = cst_pairs_.begin(); it != cst_pairs_.end(); ++it) {

		if( !(*it)->resA()->is_backbone() && !(*it)->resB()->is_backbone() ) continue;
		if( (*it)->missing_in_pose() ) continue;

		if( !(*it)->update_pdb_remarks( pose ) ) utility_exit_with_message("Error when trying to update pdb remarks.");
	}
}



utility::vector1< std::string >
EnzConstraintIO::allowed_res_name3_at_position( core::Size const seqpos ) const
{

	std::set< std::string > found;

	for (std::vector< EnzConstraintParametersOP >::const_iterator it = cst_pairs_.begin(); it != cst_pairs_.end(); ++it) {

		std::set< std::string > res_this_param = (*it)->allowed_res_name3_at_position( seqpos );

		if( res_this_param.size() == 0 ) continue;

		//we need to make two checks:
		//1. are there already residues in the found set? if so, dont include anything,
		//but remove all that are not part of the second set
		//2. if not, the found set will become res_this_param

		if( found.size() == 0 ) found = res_this_param;

		else{

			for( std::set< std::string >::iterator set_it = found.begin(); set_it != found.end();  ){

				if( res_this_param.find( *set_it ) == res_this_param.end() ){

					std::set< std::string >::iterator to_erase = set_it;
					++set_it;
					found.erase( to_erase );

				}
				else ++set_it;

			}
		}
	} //iterator over all params

	utility::vector1< std::string > to_return;

	for( std::set< std::string >::iterator set_it = found.begin(); set_it != found.end();  ++set_it ){
		to_return.push_back( *set_it );
	}

	return to_return;
}


void
EnzConstraintIO::show_cst_definitions() const
{
	if (cst_pairs_.size() == 0) {
		tr.Info << "No constraints have been read in." << std::endl;
	}
	else{
		tr.Info << cst_pairs_.size() << " constraint blocks have been read in: " << std::endl;
		for (std::vector< EnzConstraintParametersOP >::const_iterator it = cst_pairs_.begin(); it != cst_pairs_.end(); ++it) { (*(*it)).show_definitions(); }
	}
}

void
EnzConstraintIO::remap_resid( core::sequence::SequenceMapping const & smap )
{

	for (std::vector< EnzConstraintParametersOP >::iterator it = cst_pairs_.begin(); it != cst_pairs_.end(); ++it){
		(*it)->remap_resid( smap );
	}
}


void
EnzConstraintIO::set_position_for_missing_res_in_parameter_block(
	core::Size cst_block,
	core::Size respos
)
{

	runtime_assert( cst_block <= cst_pairs_.size() );

	cst_pairs_[ cst_block -1  ]->set_position_for_missing_res( respos );
}


void
EnzConstraintIO::clear_active_pose_constraints_for_block(
		core::Size cst_block
){

	runtime_assert( cst_block <= cst_pairs_.size() );

	cst_pairs_[ cst_block -1  ]->clear_active_pose_constraints();
}


void
EnzConstraintIO::set_external_position_for_resA_in_parameter_block(
	core::Size cst_block,
	core::Size respos
)
{

	runtime_assert( cst_block <= cst_pairs_.size() );

	cst_pairs_[ cst_block -1  ]->set_external_position_for_resA( respos );
}

void
EnzConstraintIO::set_external_position_for_resB_in_parameter_block(
	core::Size cst_block,
	core::Size respos
)
{

	runtime_assert( cst_block <= cst_pairs_.size() );

	cst_pairs_[ cst_block -1  ]->set_external_position_for_resB( respos );
}


void
EnzConstraintIO::setup_favor_native_constraints(
	core::pose::Pose & pose,
	core::pack::task::PackerTaskCOP task,
	core::pose::Pose const & native_pose
)
{
	using namespace core::scoring::constraints;

	core::Real bonus = core::options::option[core::options::OptionKeys::enzdes::favor_native_res].value();

	tr.Info << "favor_native_res: adding a bonus of " << bonus << " for native residues to pose." << std::endl;

	//safety check first
	if( favor_native_constraints_.size() != 0 ){
		tr.Info << "Warning: when setting up favor native constraints, there might already be some previously generated favor_native constraints in the pose, trying to remove these first." << std::endl;
		remove_favor_native_constraints( pose );

	}

	favor_native_constraints_.clear();
	for( core::Size i = 1; i <= pose.total_residue(); ++i){

		if( task->design_residue(i) ){

			ConstraintOP resconstraint = new ResidueTypeConstraint( native_pose, i, bonus );
			favor_native_constraints_.push_back( resconstraint );

		}
	}
	favor_native_constraints_ = pose.add_constraints( favor_native_constraints_ );


}//setup_favor_native_constraints


void
EnzConstraintIO::remove_favor_native_constraints(
	core::pose::Pose & pose
)
{
	if( !( pose.remove_constraints( favor_native_constraints_ ) ) ){
		tr.Info << "Warning: some of the favor native constraints that were previously added to the pose are not there anymore, something's a little unclean somewhere." << std::endl;
	}
	favor_native_constraints_.clear();
}


EnzConstraintParametersCOP
EnzConstraintIO::enz_cst_params( core::Size block ) const
{
	return cst_pairs_[ block - 1];
}

utility::vector1< EnzConstraintParametersCOP >
EnzConstraintIO::enz_cst_params_missing_in_pose() const
{
	utility::vector1< EnzConstraintParametersCOP > to_return;

	for( Size i = 0; i < cst_pairs_.size(); ++i){
		if( cst_pairs_[i]->missing_in_pose() ){
			to_return.push_back( cst_pairs_[i] );

			runtime_assert( cst_pairs_[i]->active_pose_constraints().size() == 0 );
		}
	}

	return to_return;

}

utility::vector1< core::Size >
EnzConstraintIO::ordered_constrained_positions( core::pose::Pose const & pose ) const
{
	using namespace core;
	utility::vector1< Size > found_protein_positions;
	utility::vector1< Size > found_lig_positions;

	for (std::vector< EnzConstraintParametersOP >::const_iterator cst_it = cst_pairs_.begin(); cst_it != cst_pairs_.end(); ++cst_it){

		for(std::map< Size, EnzCstTemplateResAtomsOP >::const_iterator resA_it = (*cst_it)->resA()->respos_map_begin();
				resA_it != (*cst_it)->resA()->respos_map_end(); ++resA_it ){
			if( pose.residue_type( resA_it->first ).is_ligand() ) {
				if( find( found_lig_positions.begin(), found_lig_positions.end(), resA_it->first ) == found_lig_positions.end() ) {
					found_lig_positions.push_back( resA_it->first );
				}
			}
			else found_protein_positions.push_back( resA_it->first );
		}

		for(std::map< Size, EnzCstTemplateResAtomsOP >::const_iterator resB_it = (*cst_it)->resB()->respos_map_begin();
				resB_it != (*cst_it)->resB()->respos_map_end(); ++resB_it ){
			if( pose.residue_type( resB_it->first ).is_ligand() ) {
				if( find( found_lig_positions.begin(), found_lig_positions.end(), resB_it->first ) == found_lig_positions.end() ) {
					found_lig_positions.push_back( resB_it->first );
				}
			}
			else found_protein_positions.push_back( resB_it->first );
		}

	}// loop over EnzConstraintParams

	for( 	utility::vector1< Size >::const_iterator lig_it = found_lig_positions.begin(); lig_it != found_lig_positions.end(); ++lig_it ){
		found_protein_positions.push_back( *lig_it );
	}

	return found_protein_positions;

}//ordered catalytic positions

core::Size
EnzConstraintIO::mcfi_lists_size() const
{
	return mcfi_lists_.size();
}

/*MatchConstraintFileInfoListCOP
EnzConstraintIO::mcfi_list( Size index ) const
{
	return mcfi_lists_[ index ];
}*/

} //enzdes
} //protocols


