// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file IO-functionality for enzyme Constraints
/// @brief
/// @author Florian Richter, floric@u.washington.edu


#ifndef INCLUDED_protocols_enzdes_EnzCstTemplateRes_HH
#define INCLUDED_protocols_enzdes_EnzCstTemplateRes_HH



// Unit headers
#include <protocols/enzdes/EnzConstraintIO.fwd.hh>
// Package headers

// Project headers
//#include <core/types.hh>
#include <core/conformation/Residue.fwd.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/id/AtomID.fwd.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/sequence/SequenceMapping.fwd.hh>


// Utility Headers
#include <utility/vector1.hh>
#include <utility/pointer/ReferenceCount.hh>

//Utility Headers

// C++ Headers

namespace protocols {
namespace enzdes {


/// @brief helper class for EnzCstTemplateRes, holds atom ids corresponding
/// @brief to the residue types in a certain pose
class EnzCstTemplateResAtoms : public utility::pointer::ReferenceCount {

public:

	friend class EnzCstTemplateRes; //::check_data_consistency(pose::Pose const & pose);
	friend class EnzConstraintParameters; //::add_constraints_to_cst_set(pose::Pose & pose);

	void
	remap_resid( core::sequence::SequenceMapping const & smap );

private:

	void
	remap_atomid_vector(
		core::sequence::SequenceMapping const & smap,
		std::vector< core::id::AtomID > & atomid_vec
	);

	std::vector< core::id::AtomID > atom1_;
	std::vector< core::id::AtomID > atom2_;
	std::vector< core::id::AtomID > atom3_;
};


/// @brief helper class for class EnzConstraintParameters, gathers information
/// @brief from cst input and pdb input
class EnzCstTemplateRes : public utility::pointer::ReferenceCount {

	//friend class EnzConstraintParameters;
public:

	typedef std::map< core::chemical::ResidueTypeCAP, utility::vector1< utility::vector1< core::Size > > > RestypeToTemplateAtomsMap;

public:
	typedef std::map< core::chemical::ResidueTypeCAP, utility::vector1< utility::vector1< core::Size > > > AtomIndsForRestypeMap;

public:

	EnzCstTemplateRes(
		core::chemical::ResidueTypeSetCAP src_restype_set
	);

	EnzCstTemplateRes(
		core::chemical::ResidueTypeSetCAP src_restype_set,
		EnzConstraintParametersCAP src_enzio_param );

	EnzCstTemplateRes(
		EnzCstTemplateResCOP other,
		EnzConstraintParametersCAP new_ref_param
	);

	void
	read_params(std::istringstream & line_stream);

	void
	show_params() const;

	void
	get_pose_data(core::pose::Pose const & pose);

	void
	set_enzio_param( EnzConstraintParametersCAP src_enz_io_param ) {
		enz_io_param_ = src_enz_io_param;
	}

	bool
	contains_position( core::Size const seqpos ) const {
		return respos_map_.find( seqpos ) != respos_map_.end();
	}

	EnzCstTemplateResAtomsCOP
	get_template_atoms_at_pos( core::Size seqpos ) const;

	bool
	rb_minimizable() const {
		return rb_minimizable_; }

	bool
	is_backbone() const {
		return is_backbone_; }

	bool
	find_in_pose_if_missing_from_header( core::pose::Pose const & pose);

	void
	set_position_in_pose( core::Size resnum );

	void
	set_external_position( core::Size resnum );


	/// @brief WARNING if you use this function to sequentially put positions into the object,
	/// @brief you MUST make sure it was empty before (i.e. clear it), otherwise you'll
	/// @brief get replications in the internal data structures.
	void
	add_position_in_pose( core::Size resnum );

	utility::vector1< std::string > const &
	allowed_res_types() const {
		return allowed_res_types_;
	}

	bool
	not_in_pose() const {
		return not_in_pose_;
	}

	std::map< Size, EnzCstTemplateResAtomsOP >::const_iterator
	respos_map_begin() const{
		return respos_map_.begin();
	}

	std::map< Size, EnzCstTemplateResAtomsOP >::const_iterator
	respos_map_end() const{
		return respos_map_.end();
	}

	RestypeToTemplateAtomsMap::const_iterator
	atom_inds_for_restype_begin() const {
		return atom_inds_for_restype_.begin(); }

	RestypeToTemplateAtomsMap::const_iterator
	atom_inds_for_restype_end() const {
		return atom_inds_for_restype_.end(); }


	core::Size
	respos_map_size() const{
		return respos_map_.size(); }

	bool
	remove_position( core::Size resnum );

	inline
	void
	clear_pose_info(){
		respos_map_.clear();
		not_in_pose_ = true; pose_data_uptodate_ = false;
	}

	void
	clear_all();

	void
	remap_resid( core::sequence::SequenceMapping const & smap );

	void
	determine_atom_inds_for_restype( core::chemical::ResidueTypeCAP restype );

	utility::vector1< core::Size > const &
	atom_inds_for_restype(
		core::Size template_atom,
		core::chemical::ResidueTypeCAP restype
  ) const;

	void
	identical_info_consistency_check() const;

private:

	inline
	void
	insert_pdb_info(Size resnum)
	{
		respos_map_.insert( std::pair<Size,EnzCstTemplateResAtomsOP>(resnum, new EnzCstTemplateResAtoms() ) );
	}

	//information from pdb file
	//Size resnum_;
	std::map< Size, EnzCstTemplateResAtomsOP > respos_map_;

	//information from cst file
	utility::vector1< std::string > atom1_, atom2_, atom3_;   //names
	std::string at1_type_, at2_type_, at3_type_;    //types

	//contains the names of allowed res types as specified by cst file
	utility::vector1< std::string > allowed_res_types_;

	//contains all allowed atoms for a certain residue type
	//represents the translation of the above cst file info (allowed_res_types_, atom1_, at1_type_, etc )
	//into workable data
	RestypeToTemplateAtomsMap atom_inds_for_restype_;

	bool not_in_pose_;
	bool pose_data_uptodate_;

	//if this is a ligand, do we want to allow rb minimization?
	//used i.e. in cases when there are two ligands, one of them should stay fixed
	bool rb_minimizable_;

	//if this residue interaction is mediated through backbone only
	//right now can only be set through file, but should be changed
	//to automatic detection in the future.
	bool is_backbone_;

	utility::vector1< core::Size > respos_from_external_;

	//if there is another residue that's identical to this
	bool identical_tag_found_;
	core::Size corresponding_res_block_, corresponding_res_num_in_block_;

	//residue set that we are working with, needed to look up allowed res types
	core::chemical::ResidueTypeSetCAP restype_set_;

	EnzConstraintParametersCAP enz_io_param_; // the params object that this residue belongs to


}; //class EnzCstTemplateRes



} //protocols
} //enzdes


#endif
