// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/enzdes/EnzdesBaseProtocol.cc
///
/// @brief
/// @author Florian Richter, floric@u.washington.edu



#include <protocols/enzdes/EnzdesBaseProtocol.hh>
#include <protocols/enzdes/EnzConstraintIO.hh>
#include <protocols/enzdes/enzdes_util.hh>

#include <protocols/ligand_docking/LigandBaseProtocol.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/io/pdb/file_data.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>
#include <core/options/option.hh>
#include <core/io/database/open.hh> //database reading
#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/constraints/BoundConstraint.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/dunbrack/SingleResidueDunbrackLibrary.hh>
#include <core/scoring/methods/EnergyMethodOptions.hh>


#include <core/chemical/util.hh> //needed for adding variant types in cst opt
#include <protocols/toolbox/IGEdgeReweighters.hh>
#include <core/pack/task/IGEdgeReweightContainer.hh>
#include <core/pack/rotamer_set/UnboundRotamersOperation.hh>
#include <core/pose/metrics/CalculatorFactory.hh>
#include <core/pose/metrics/PoseMetricCalculatorBase.fwd.hh>
#include <core/util/MetricValue.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/toolbox/PoseMetricCalculators/InterfaceSasaDefinitionCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/InterfaceNeighborDefinitionCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/InterfaceDeltaEnergeticsCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/NonlocalContactsCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/NumberHBondsCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/BuriedUnsatisfiedPolarsCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/PackstatCalculator.hh>
#include <protocols/toolbox/pose_manipulation.hh>

#include <protocols/moves/MinMover.hh>
#include <utility/string_util.hh>

#include <utility/io/izstream.hh>


// option key includes

#include <core/options/keys/score.OptionKeys.gen.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>
#include <core/options/keys/enzdes.OptionKeys.gen.hh>
#include <core/options/keys/docking.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>


namespace protocols{
namespace enzdes{

static core::util::Tracer tr("protocols.enzdes.EnzdesBaseProtocol");

EnzdesBaseProtocol::EnzdesBaseProtocol():
	LigandBaseProtocol(),
	bb_min_allowed_dev_(0.5),
	lig_superposition_file_read_(false),
	exclude_protein_protein_hack_elec_(true)
{

		Mover::type( "EnzdesFixBBProtocol" );
		restype_set_ = core::chemical::ChemicalManager::get_instance()->residue_type_set( core::chemical::FA_STANDARD ) ;
		cst_io_ = new EnzConstraintIO(restype_set_);
		residue_calculators_.clear();
		native_compare_calculators_.clear();
		//catalytic_res_.clear();
		atoms_to_superimpose_on_.clear();
		silent_Es_.clear();
		bb_min_allowed_dev_ = core::options::option[ core::options::OptionKeys::enzdes::bb_min_allowed_dev];

		std::string score_patch = core::options::option[ core::options::OptionKeys::score::patch ];

		if( score_patch == "" )	reduced_sfxn_ = core::scoring::ScoreFunctionFactory::create_score_function( "enzdes_polyA_min" );
		else reduced_sfxn_ = core::scoring::ScoreFunctionFactory::create_score_function( "enzdes_polyA_min", score_patch );

		if( core::options::option[core::options::OptionKeys::enzdes::compare_native].user() ){
			native_comp_ = new protocols::enzdes::DesignVsNativeComparison();
		}

		//if( core::options::option[core::options::OptionKeys::score::weights].user() ){
		//	std::string weights_tag = core::options::option[core::options::OptionKeys::score::weights];
		//	scorefxn_->initialize_from_file( core::io::database::full_name( "scoring/weights/"+weights_tag+".wts" ) );
		//}
		if ( core::options::option[ core::options::OptionKeys::score::weights ].user() ) {
			scorefxn_ = core::scoring::getScoreFunction(); // This call handles the database vs working directory resolution -- DONT SUBVERT OR DUPLICATE IT
		}
		else{
			if( score_patch == "" ) scorefxn_ = core::scoring::ScoreFunctionFactory::create_score_function("enzdes");
			else scorefxn_ = core::scoring::ScoreFunctionFactory::create_score_function("enzdes", score_patch);
		}

		relevant_scoreterms_.push_back( "total_score" );
		relevant_scoreterms_.push_back( "fa_rep" );
		relevant_scoreterms_.push_back( "hbond_sc" );
		relevant_scoreterms_.push_back( "all_cst" );

		if (scorefxn_->has_zero_weight( core::scoring::coordinate_constraint ) ){
			constraint_weights_[core::scoring::coordinate_constraint] = 1.0;
		}
		else{
			constraint_weights_[core::scoring::coordinate_constraint] = scorefxn_->weights()[core::scoring::coordinate_constraint];
		}
		if (scorefxn_->has_zero_weight( core::scoring::atom_pair_constraint ) ){
			constraint_weights_[core::scoring::atom_pair_constraint] = 1.0;
		}
		else{
			constraint_weights_[core::scoring::atom_pair_constraint] = scorefxn_->weights()[core::scoring::atom_pair_constraint];
		}
		if (scorefxn_->has_zero_weight( core::scoring::angle_constraint ) ){
			constraint_weights_[core::scoring::angle_constraint] = 1.0;
		}
		else{
			constraint_weights_[core::scoring::angle_constraint] = scorefxn_->weights()[core::scoring::angle_constraint];
		}
		if (scorefxn_->has_zero_weight( core::scoring::dihedral_constraint ) ){
			constraint_weights_[core::scoring::dihedral_constraint] = 1.0;
		}
		else{
			constraint_weights_[core::scoring::dihedral_constraint] = scorefxn_->weights()[core::scoring::dihedral_constraint];
		}

		if( core::options::option[core::options::OptionKeys::enzdes::favor_native_res].user() ){
			if (scorefxn_->has_zero_weight( core::scoring::res_type_constraint ) ){
				constraint_weights_[core::scoring::res_type_constraint] = 1.0;
			}
			else{
				constraint_weights_[core::scoring::res_type_constraint] = scorefxn_->weights()[core::scoring::res_type_constraint];
			}
		}

		enable_constraint_scoreterms();

		if( core::options::option[ core::options::OptionKeys::docking::ligand::old_estat ].user() ){
			exclude_protein_protein_hack_elec_ = core::options::option[ core::options::OptionKeys::docking::ligand::old_estat ];
		}
		if( exclude_protein_protein_hack_elec_ ){
			core::scoring::methods::EnergyMethodOptions options( scorefxn_->energy_method_options() );
			options.exclude_protein_protein_hack_elec( true );
			scorefxn_->set_energy_method_options( options );
		}


		//set the native pose if requested
		if( core::options::option[core::options::OptionKeys::in::file::native].user() ){
			core::pose::PoseOP natpose = new core::pose::Pose();
			core::io::pdb::pose_from_pdb( *natpose, core::options::option[core::options::OptionKeys::in::file::native].value() );
			(*scorefxn_)( *natpose);
			this->set_native_pose( natpose );
		}

		//increase the chainbreak weight. 1.0 is apparently not enough for some constraints
		//scorefxn_->set_weight( core::scoring::chainbreak, 10.0 );

} //EnzdesBaseProtocol constructor


utility::vector1< core::Size >
EnzdesBaseProtocol::catalytic_res( core::pose::Pose const & pose ) const
{
	using namespace core;
	utility::vector1< Size > to_return = cst_io_->ordered_constrained_positions( pose );

	if( to_return.size() == 0 ){
		for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i){
			if( pose.residue_type( i ).is_ligand() ) to_return.push_back( i );
		}
	}
	return to_return;
}

std::set< core::Size > const &
EnzdesBaseProtocol::design_targets( core::pose::Pose const & pose ) const
{
	design_targets_.clear();

	for(core::Size i = 1; i <=  pose.total_residue(); ++i){
		if( is_catalytic_position( i ) ) design_targets_.insert( i );
	}
		//if no positions are constrained, we'll put the ligands into the array
	if( design_targets_.size() == 0 ){
		for(core::Size i = 1; i <=  pose.total_residue(); ++i){
			if( pose.residue_type(i).is_ligand() ) design_targets_.insert( i );
		}
	}
	return design_targets_;
}


core::Real
EnzdesBaseProtocol::sum_constraint_scoreterms(
	core::pose::Pose const & pose,
	int which_res
)
{

  using namespace core::scoring;

  EnergyMap all_weights = pose.energies().weights();
  EnergyMap scores;

  if( which_res == -1){ //means we want to know stuff for the whole pose
    scores = pose.energies().total_energies();
  }
  else{ scores = pose.energies().residue_total_energies( which_res ); }

  return scores[ coordinate_constraint ] * all_weights[ coordinate_constraint ] + scores[atom_pair_constraint] * all_weights[ atom_pair_constraint] +
    scores[ angle_constraint ] * all_weights[ angle_constraint ] + scores[ dihedral_constraint ] * all_weights[ dihedral_constraint ];

}


void
EnzdesBaseProtocol::register_options()
{

	using namespace core::options;
	using namespace OptionKeys;

	option.add_relevant( in::file::native );
	option.add_relevant( in::file::s );
	option.add_relevant( in::file::l );
	option.add_relevant( OptionKeys::score::weights );
	option.add_relevant( OptionKeys::score::patch );
	option.add_relevant( OptionKeys::packing::resfile  );
	option.add_relevant( OptionKeys::packing::soft_rep_design );

	protocols::moves::MinMover::register_options();
	protocols::moves::PackRotamersMover::register_options();

	option.add_relevant( OptionKeys::enzdes::detect_design_interface );
	option.add_relevant( OptionKeys::enzdes::cut1 );
	option.add_relevant( OptionKeys::enzdes::cut2 );
	option.add_relevant( OptionKeys::enzdes::cut3 );
	option.add_relevant( OptionKeys::enzdes::cut4 );
	option.add_relevant( OptionKeys::enzdes::fix_catalytic_aa );
	option.add_relevant( OptionKeys::enzdes::enz_score );
	option.add_relevant( OptionKeys::enzdes::enz_repack );
	option.add_relevant( OptionKeys::enzdes::lig_packer_weight );
	option.add_relevant( OptionKeys::enzdes::cst_min );
	option.add_relevant( OptionKeys::enzdes::chi_min );
	option.add_relevant( OptionKeys::enzdes::bb_min );
	option.add_relevant( OptionKeys::enzdes::bb_min_allowed_dev );
	option.add_relevant( OptionKeys::enzdes::enz_debug );
	option.add_relevant( OptionKeys::enzdes::no_packstat_calculation );
	option.add_relevant( OptionKeys::enzdes::compare_native );
	option.add_relevant( OptionKeys::enzdes::favor_native_res );
	option.add_relevant( OptionKeys::docking::ligand::old_estat );

}


core::pack::task::PackerTaskOP
EnzdesBaseProtocol::create_enzdes_pack_task(
	core::pose::Pose const & pose,
	bool design
){

	using namespace core::pack::task;
	using namespace core::options;

	PackerTaskOP task = core::pack::task::TaskFactory::create_packer_task( pose );
	task->initialize_from_command_line();
	task->append_rotamerset_operation( unboundrot_ );

	//find the design interface and set up task
	if( core::options::option[core::options::OptionKeys::enzdes::detect_design_interface].user() ){

		utility::vector1< bool > repack_res( pose.total_residue() );
		utility::vector1< bool > design_res( pose.total_residue() );
		core::Real cut1 = core::options::option[core::options::OptionKeys::enzdes::cut1];
		core::Real cut2 = core::options::option[core::options::OptionKeys::enzdes::cut2];
		core::Real cut3 = core::options::option[core::options::OptionKeys::enzdes::cut3];
		core::Real cut4 = core::options::option[core::options::OptionKeys::enzdes::cut4];
		if( !design ){
			cut1 = 0.0;
			cut2 = 0.0;
		}

		find_design_interface( pose, pose.num_jump(), cut1, cut2, cut3, cut4, repack_res, design_res );

		//setup the task accordingly
		for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i){

			//catalytic res will be set to repacking/non-packing
			//utility::vector1< Size >::iterator find_res = find( catalytic_res_.begin(), catalytic_res_.end(), i );
			if( is_catalytic_position( i ) ){

				if( core::options::option[core::options::OptionKeys::enzdes::fix_catalytic_aa] ) {
					task->nonconst_residue_task( i ).prevent_repacking();
				}

				else{

					//if this happens to be a backbone only position, allow all the residues
					//specified in cstfile
					if( cst_io_->is_backbone_only_cst( i ) ){

						utility::vector1< bool > allowed_aas( core::chemical::num_canonical_aas, false );

						utility::vector1< std::string > allowed_name3 = cst_io_->allowed_res_name3_at_position( i );

						for( core::Size j =1; j <= allowed_name3.size(); ++j){
							allowed_aas[ core::chemical::aa_from_name( allowed_name3[ j ] ) ] = true;
						}

						task->nonconst_residue_task(i).restrict_absent_canonical_aas( allowed_aas );

					}

					else task->nonconst_residue_task(i).restrict_to_repacking();  }
			}

			//default behavior for design will be everything except cys, and of course we want to leave disulfide bonds untouched
			else if( design_res[i] == true) {

				if( pose.residue( i ).name3() == "CYS" && pose.residue( i ).has_variant_type( core::chemical::DISULFIDE ) ){
					task->nonconst_residue_task( i ).restrict_to_repacking();
				}
				else{
					utility::vector1< bool > keep_aas( core::chemical::num_canonical_aas, true );
					keep_aas[ core::chemical::aa_cys ] = false;
					task->nonconst_residue_task(i).restrict_absent_canonical_aas( keep_aas );
				}
			}

			else if( repack_res[i] == true ){
				task->nonconst_residue_task(i).restrict_to_repacking();
			}

			else if( pose.residue(i).is_ligand() && ( pose.residue_type(i).get_RotamerLibrary() != 0 ) ){
				task->nonconst_residue_task(i).restrict_to_repacking();
			}
			else { task->nonconst_residue_task( i ).prevent_repacking();}
		}

	} // detect design interface


	//in case there is a resfile, information in this resfile overrides the designed interface
	if( core::options::option[core::options::OptionKeys::packing::resfile].user() ){
		task->read_resfile();
		if( !design ){
			for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i){
				if(task->design_residue(i) ) task->nonconst_residue_task(i).restrict_to_repacking();
			}
		}
	}

	//we have to make sure that the ligand is set to repacking, otherwise upweighting will not work
	//for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {
	//	if ( ( pose.residue(i).is_ligand() ) ) //&& ( pose.residue_type(i).get_RotamerLibrary() == 0 ) )
	//		task->nonconst_residue_task( i ).restrict_to_repacking();
	//}

	//in case we are only interested in scoring
	if(core::options::option[core::options::OptionKeys::enzdes::enz_score]) {
		for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i){
			task->nonconst_residue_task(i).prevent_repacking();
		}
	}

	//in case we are only interested in repacking
	else if(core::options::option[core::options::OptionKeys::enzdes::enz_repack]) {
		for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i){
			if(task->design_residue(i)) {
					task->nonconst_residue_task(i).restrict_to_repacking();
			}
		}
	}

	//and if we're designing, set the ligand weigths to the specified value
	else if( design && core::options::option[core::options::OptionKeys::enzdes::lig_packer_weight] != 1.0 ){
		core::Real lig_weight = core::options::option[core::options::OptionKeys::enzdes::lig_packer_weight];

		core::pack::task::IGEdgeReweighterOP lig_up = new protocols::toolbox::IGLigandDesignEdgeUpweighter( lig_weight );
		core::pack::task::IGEdgeReweightContainerOP IGreweight = task->set_IGEdgeReweights();
		IGreweight->add_reweighter( lig_up );

		tr.Info << "Packer Energies between ligand and design residues are upweighted by factor " << lig_weight << "." << std::endl;

	} //if different ligand weights are asked for

	//debug shit
	//for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i){
	//	if( task->design_residue(i) ) std::cerr << "TASKDEBUG Residue " << i << " set to redesign" <<std::endl;
	//	else if( task->pack_residue( i ) ) std::cerr << "TASKDEBUG Residue " << i << " set to repacking" <<std::endl;
	//}

	//make sure the design targets are up to date
	design_targets( pose );

	return task;

} //create_enzdes_pack_task


void
EnzdesBaseProtocol::enzdes_pack(
	core::pose::Pose & pose,
	core::pack::task::PackerTaskCOP task,
	core::scoring::ScoreFunctionCOP scorefxn,
	core::Size cycles,
	bool pack_unconstrained,
	bool minimize_unconstrained,
	bool favor_native
)
{

	if( pack_unconstrained ) remove_enzdes_constraints( pose, true );

	if( favor_native ){
		cst_io_->setup_favor_native_constraints( pose, task, *(this->get_native_pose()) );
	}

	core::pack::task::PackerTaskCOP usetask = task;
	core::scoring::ScoreFunctionCOP packsfxn;

	for( core::Size cycle = 1; cycle <= cycles; ++cycle){

		//soft_rep we'll only do this if the task is to be designed and not the last cycle,
		//because we really don't want any clashes
		bool soft_rep ( core::options::option[core::options::OptionKeys::packing::soft_rep_design] && usetask->design_any() && (cycle < cycles ) );

		if( soft_rep ) packsfxn = soft_scorefxn_;
		else packsfxn = scorefxn;

		protocols::moves::PackRotamersMoverOP enzdes_pack = new protocols::moves::PackRotamersMover(packsfxn, usetask);

		enzdes_pack->apply(pose);

		if(core::options::option[core::options::OptionKeys::enzdes::cst_min]) cst_minimize(pose, task);

		usetask = enzutil::recreate_task( pose, *task );

	} //cycle loop

	if( favor_native ){
		cst_io_->remove_favor_native_constraints( pose );
	}

	if( pack_unconstrained) add_pregenerated_enzdes_constraints( pose );

	(*scorefxn)( pose );

	if(core::options::option[core::options::OptionKeys::enzdes::cst_min]){
		//tr.Info << "Starting after design/pack unconstrained minimization... " << std::endl;
		if( minimize_unconstrained ) remove_enzdes_constraints( pose, true );
		cst_minimize(pose, task);
		if( minimize_unconstrained ) add_pregenerated_enzdes_constraints( pose );

		(*scorefxn_)( pose );
		//tr.Info << "done after design unconstrained minimization." << std::endl;
	}

	if( task->design_any() ) cst_io_->update_pdb_remarks_for_backbone_params( pose );

} //design function


void
EnzdesBaseProtocol::find_design_interface(
	core::pose::Pose const & pose,
	Size jump_id,
	core::Real cut1,
	core::Real cut2,
	core::Real cut3,
	core::Real cut4,
	utility::vector1< bool > & repack_res,
	utility::vector1< bool > & design_res
){

	//runtime_assert( cutoff_dist > 0 );
	//double const cutoff2 = cutoff_dist * cutoff_dist;
	core::Real cut1_sq = cut1 * cut1;
	core::Real cut2_sq = cut2 * cut2;
	core::Real cut3_sq = cut3 * cut3;
	core::Real cut4_sq = cut4 * cut4;

	Size num_repack = 0;
	Size num_design = 0;
	//is_interface.dimension( pose.total_residue(), false ); // init all positions to false
	FArray1D_bool is_upstream ( pose.total_residue(), false );
	pose.fold_tree().partition_by_jump( jump_id, is_upstream );
	for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {

		// all residues on ligand side are part of repack interface
		if ( ! is_upstream(i) ) {
			repack_res[i] = true;
			design_res[i] = false;
			//			num_in_interface += 1;
			continue;
		}
		else{
			repack_res[i] = false;
			design_res[i] = false;
		}
		// on protein side, have to do distance check
		core::conformation::Residue const & prot_rsd = pose.residue(i);
		for(core::Size j = 1, j_end = pose.total_residue(); j <= j_end; ++j) {
			if ( is_upstream(j) ) continue; // compare against only ligand residues
			core::conformation::Residue const & lig_rsd = pose.residue(j);
			for(core::Size k = 1, k_end = lig_rsd.nheavyatoms(); k <= k_end; ++k) {
				core::Vector prot_cb, prot_ca;
				if( prot_rsd.has("CB") ) prot_cb = prot_rsd.xyz("CB");
				if( prot_rsd.has("CA") ) prot_ca = prot_rsd.xyz("CA"); // GLY
				core::Real ca_dist2 = lig_rsd.xyz(k).distance_squared( prot_ca );
				if( ca_dist2 <= cut4_sq ) {
					if( ca_dist2 <= cut3_sq ) {
						if( ca_dist2 <= cut2_sq ) {
							if( ca_dist2 <= cut1_sq) {
								design_res[i] = true;
								repack_res[i] = false;
								goto END_LIGRES_LOOP; // C++ lacks multi-level break  :(
							} // cut1
							else if( prot_rsd.has("CB") ) {
								core::Real cb_dist2 = lig_rsd.xyz(k).distance_squared( prot_cb );
								//								tr.Info << "cb_dist2 is " << cb_dist2 << "; ";
								if( cb_dist2 < ca_dist2 ) {
									design_res[i] = true;
									repack_res[i] = false;
									goto END_LIGRES_LOOP; // C++ lacks multi-level break  :(
								}
								else {
									repack_res[i] = true;
								}
							}
							else {   //glycine, we'll be generous and allow redesigning
								design_res[i] = true;
								repack_res[i] = false;
								goto END_LIGRES_LOOP; // C++ lacks multi-level break  :(
							}
						} //cut2
						else {
							repack_res[i] = true;
						}
					} //cut3

					else if( prot_rsd.has("CB") ) {
						core::Real cb_dist2 = lig_rsd.xyz(k).distance_squared( prot_cb );
						if( cb_dist2 < ca_dist2 ) {
							repack_res[i] = true;
						}
					}
				} //cut4
			}
		}
		END_LIGRES_LOOP: ; // compiler needs ; as a no-op before end of loop
		if( repack_res[i] == true ) { num_repack++ ; }
		if( design_res[i] == true ) { num_design++ ; }
		if( ( repack_res[i] == true)  && ( design_res[i] == true ) ) { tr.Info << "Huh? this should not happen. " << std::endl; }
	}

	tr.Info << "Design Interface: detected " << num_design << " design-shell residues and " << num_repack << " repack-shell residues, shell sizes cut1-4 used were " << cut1 << " " << cut2 << " " << cut3 << " " << cut4 << std::endl << "Design-shell Residues(pose-numbering): " ;
	for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {
		if(design_res[i] == true) { tr.Info << i << "+";
		}
	}
	tr.Info << std::endl << "Repack-shell Residues(pose-numbering): ";

	for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {
		if(repack_res[i] == true) { tr.Info << i << "+";
		}
	}
	tr.Info << std::endl;

} //find_design_interface


void
EnzdesBaseProtocol::setup_enzdes_constraints(
	core::pose::Pose & pose,
	bool allow_missing_remark_blocks
){

	cst_io_->add_constraints_to_pose(pose, scorefxn_, allow_missing_remark_blocks);

	tr.Info << "Catalytic residues (pose numbering) are: ";
	for( core::Size i = 1; i <= pose.total_residue(); ++i){
		if( is_catalytic_position( i ) ) tr.Info << i << " ";
	}
	tr.Info << std::endl;


} //setup enzdes constraints function


void
EnzdesBaseProtocol::remove_enzdes_constraints(
	core::pose::Pose & pose,
	bool keep_covalent
)
{ cst_io_->remove_constraints_from_pose( pose, keep_covalent, true ); }


void
EnzdesBaseProtocol::add_pregenerated_enzdes_constraints(
	core::pose::Pose & pose
)
{ cst_io_->add_pregenerated_constraints_to_pose( pose, scorefxn_ ); }


void
EnzdesBaseProtocol::cst_minimize(
	core::pose::Pose & pose,
	core::pack::task::PackerTaskCOP task,
	bool cst_opt
){


	core::pose::Pose old_Pose = pose; //copy old pose
	core::scoring::ScoreFunctionOP min_scorefxn;
	//core::scoring::constraints::ConstraintSetOP saved_constraints;
	utility::vector1< bool > allow_move_bb(pose.total_residue(), false );

	//for the jump id we'll figure out something smarter later
	int jump_id = pose.num_jump(); // assume ligand attached by last jump
	if ( jump_id == 0 ) {
		utility_exit_with_message("Error: Pose has no jumps!");
	}


	if(cst_opt){

		min_scorefxn = reduced_sfxn_;

		utility::vector1< core::Size >positions_to_replace;

		for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {

			if( task->pack_residue(i) && ( ! is_catalytic_position( i ) ) ) positions_to_replace.push_back( i );
		}

		protocols::toolbox::pose_manipulation::construct_poly_ala_pose( pose, positions_to_replace, true, true, true );

	}

	else { min_scorefxn = scorefxn_; }

	//set up the movemap. eventually this could be integrated with ian's stuff
	//in src/protocols/ligand_docking/
	core::kinematics::MoveMapOP movemap = new core::kinematics::MoveMap();
	movemap->set_jump(jump_id, true);
	movemap->set_chi( false );
	movemap->set_bb( false );

	for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {
		if ( task->pack_residue(i) && pose.residue(i).is_polymer() ){
			if(core::options::option[core::options::OptionKeys::enzdes::chi_min] ) movemap->set_chi(i, true);
			if(core::options::option[core::options::OptionKeys::enzdes::bb_min] ) {
				allow_move_bb[i] = true;
			}
		}
	}

	//for the minimization to work properly, each residue that is minimized has to be part of a stretch of at least 4 other residues
	//so we need to allow a couple of more residues to move
	//core::Real initial_cbreak;
	if(core::options::option[core::options::OptionKeys::enzdes::bb_min] ){
		core::Size window = 4;

		enzutil::make_continuous_true_regions_in_bool_vector( allow_move_bb, window );
		tr.Info << "Doing a pose minimization... the backbone is allowed to move at positions: ";
		for(core::Size i = 1; i <= pose.total_residue(); ++i){
			if( allow_move_bb[i] && pose.residue(i).is_polymer() ){
				movemap->set_bb(i, true);
				tr.Info << i <<", ";
			}
		}
		tr.Info << std::endl;

		core::Size const lig_id = get_ligand_id(pose, jump_id);
		//restraining function for Calphas. should allow fairly liberal movement ~0.5A from the original position,
		//but severly limits movement beyond this
		core::scoring::constraints::FuncOP ca_restr_func = new core::scoring::constraints::BoundFunc( 0, bb_min_allowed_dev_, 0.1, "CAdis");
		reorder_foldtree_around_mobile_regions( pose, jump_id, allow_move_bb, lig_id );
		restrain_protein_Calphas(pose, allow_move_bb, ca_restr_func );

		//(*min_scorefxn)( pose );
		//initial_cbreak = pose.energies().total_energies()[ core::scoring::chainbreak ];
		//std::cerr << "chainbreak score premin " << initial_cbreak << std::endl;
	}

	if(core::options::option[core::options::OptionKeys::enzdes::enz_debug] ){
		//debug stage: only interested in constraints minimization for now
		//restrict move map to only move ligand along jump and the three
		//catalytic res
		min_scorefxn = reduced_sfxn_->clone();
		min_scorefxn->reset();
		min_scorefxn->set_weight(core::scoring::chainbreak, 1.0);
		min_scorefxn->set_weight(core::scoring::omega, 0.0 );
		min_scorefxn->set_weight(core::scoring::coordinate_constraint, 1.0);
		min_scorefxn->set_weight(core::scoring::atom_pair_constraint, 1.0);
		min_scorefxn->set_weight(core::scoring::angle_constraint, 1.0);
		min_scorefxn->set_weight(core::scoring::dihedral_constraint, 1.0);

	}

	//temporarily set chainbreak score very high, to prevent chainbreaks from opening
	//this has sometimes been observed in case of highly constrained systems
	core::Real orig_cbreak_weight = min_scorefxn->weights()[ core::scoring::chainbreak ];
	if( orig_cbreak_weight < 10.0 ) min_scorefxn->set_weight( core::scoring::chainbreak, 10.0);

	//setting up move map done, now do minimization
	protocols::moves::MinMoverOP dfpMinTightTol = new protocols::moves::MinMover( movemap, min_scorefxn, "dfpmin_armijo_nonmonotone_atol", 0.02, true /*use_nblist*/ );
	dfpMinTightTol->apply(pose);

	min_scorefxn->set_weight( core::scoring::chainbreak, orig_cbreak_weight);
	(*min_scorefxn)( pose );
	//core::Real post_cbreak = pose.energies().total_energies()[ core::scoring::chainbreak ];
	//std::cerr << "chainbreak score postmin " << post_cbreak << std::endl;


	//now we have to reinstate the orginial pose, so the task is still valid
	if(cst_opt){

		for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {

			//bool this_res_catalytic(false);
			//utility::vector1< Size >::iterator find_res = find( catalytic_res_.begin(), catalytic_res_.end(), i );
			//if( find_res != catalytic_res_.end() ){
			//	this_res_catalytic = true;
			//}
			if ( task->pack_residue(i) && ( ! is_catalytic_position( i ) )  ){
				pose.replace_residue( i, old_Pose.residue(i), true);
			}
		}
	}

	if(core::options::option[core::options::OptionKeys::enzdes::bb_min] ){

		pose.constraint_set( old_Pose.constraint_set()->clone() );
		pose.fold_tree( old_Pose.fold_tree() );

		//put back the right variants
		for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {

			if( !pose.residue_type( i ).variants_match( old_Pose.residue_type( i ) ) ){

				utility::vector1< std::string > const new_var_types( pose.residue_type( i ).variant_types() );
				utility::vector1< std::string > const old_var_types( old_Pose.residue_type( i ).variant_types() );
				for( utility::vector1< std::string >::const_iterator newvars = new_var_types.begin(); newvars  != new_var_types.end(); ++newvars ){
					if( !old_Pose.residue_type( i ).has_variant_type( *newvars ) ) core::chemical::remove_variant_type_from_pose_residue( pose, *newvars, i );
				}

				for( utility::vector1< std::string >::const_iterator oldvars = old_var_types.begin(); oldvars  != old_var_types.end(); ++oldvars ){
					if( !pose.residue_type( i ).has_variant_type( *oldvars ) ) core::chemical::add_variant_type_to_pose_residue( pose, *oldvars, i );
				}


			} //if variants don't match
		}
	} //if bb_min

	if(core::options::option[core::options::OptionKeys::enzdes::enz_debug] ){
		pose.dump_scored_pdb("aftermin_pose.pdb", *min_scorefxn);
	}

}

bool
EnzdesBaseProtocol::is_catalytic_position( core::Size const seqpos ) const
{
	return cst_io_->contains_position( seqpos );
}

void
EnzdesBaseProtocol::enable_constraint_scoreterms(){

	scorefxn_->set_weight(core::scoring::coordinate_constraint, constraint_weights_[core::scoring::coordinate_constraint] );
	scorefxn_->set_weight(core::scoring::atom_pair_constraint, constraint_weights_[core::scoring::atom_pair_constraint] );
	scorefxn_->set_weight(core::scoring::angle_constraint, constraint_weights_[core::scoring::angle_constraint] );
	scorefxn_->set_weight(core::scoring::dihedral_constraint, constraint_weights_[core::scoring::dihedral_constraint] );
	scorefxn_->set_weight(core::scoring::res_type_constraint, constraint_weights_[core::scoring::res_type_constraint] );

}

void
EnzdesBaseProtocol::disable_constraint_scoreterms(){

	constraint_weights_.clear();
	constraint_weights_[core::scoring::coordinate_constraint] = scorefxn_->weights()[core::scoring::coordinate_constraint];
	constraint_weights_[core::scoring::atom_pair_constraint] = scorefxn_->weights()[core::scoring::atom_pair_constraint];
	constraint_weights_[core::scoring::angle_constraint] = scorefxn_->weights()[core::scoring::angle_constraint];
	constraint_weights_[core::scoring::dihedral_constraint] = scorefxn_->weights()[core::scoring::dihedral_constraint];
	constraint_weights_[core::scoring::res_type_constraint] = scorefxn_->weights()[core::scoring::res_type_constraint];

	scorefxn_->set_weight(core::scoring::coordinate_constraint, 0.0 );
	scorefxn_->set_weight(core::scoring::atom_pair_constraint, 0.0 );
	scorefxn_->set_weight(core::scoring::angle_constraint, 0.0 );
	scorefxn_->set_weight(core::scoring::dihedral_constraint, 0.0 );
	scorefxn_->set_weight(core::scoring::res_type_constraint, 0.0 );

}



/// @brief function to setup residue specific calculators for all the catalytic residues and the ligand
void
EnzdesBaseProtocol::setup_pose_metric_calculators ( core::pose::Pose const & pose ){


	using namespace core::pose::metrics;
	residue_calculators_.clear();
	native_compare_calculators_.clear();

	//calculators setup by this function ( under development )
	//1. packstat for the complete pose (with and without ligand)
	//2. SASA and interface delta for ligand
	//3. number of Hbonds for catalytic residues + ligand
	//4. number of buried polars for catalytic residues + ligand
	//5. packstat for every catalytic residue
	//6. further down the road maybe some stuff to compare native and designed structure

	std::string hbond_calc_name = "hbond_pm";
	std::string burunsat_calc_name = "burunsat_pm";
	std::string packstat_calc_name = "pstat_pm";
	std::string noligpackstat_calc_name = "nlpstat_pm";
	std::string nonlocalcontacts_calc_name = "NLconts_pm";

	//before starting, we make sure that all necessary calculators are instantiated
	if( !CalculatorFactory::Instance().check_calculator_exists( hbond_calc_name ) ){
		core::pose::metrics::PoseMetricCalculatorOP hb_calc = new protocols::toolbox::PoseMetricCalculators::NumberHBondsCalculator();
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( hbond_calc_name, hb_calc );
	}

	if( !CalculatorFactory::Instance().check_calculator_exists( burunsat_calc_name ) ){
		core::pose::metrics::PoseMetricCalculatorOP burunsat_calc = new protocols::toolbox::PoseMetricCalculators::BuriedUnsatisfiedPolarsCalculator("default", hbond_calc_name);
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( burunsat_calc_name, burunsat_calc );
	}

	if( !CalculatorFactory::Instance().check_calculator_exists( packstat_calc_name ) ){
		core::pose::metrics::PoseMetricCalculatorOP pstat_calc = new protocols::toolbox::PoseMetricCalculators::PackstatCalculator();
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( packstat_calc_name, pstat_calc );
	}

	if( !CalculatorFactory::Instance().check_calculator_exists( noligpackstat_calc_name ) ){
		core::pose::metrics::PoseMetricCalculatorOP noligpstat_calc = new protocols::toolbox::PoseMetricCalculators::PackstatCalculator( true );
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( noligpackstat_calc_name, noligpstat_calc );
	}

	if( !CalculatorFactory::Instance().check_calculator_exists( nonlocalcontacts_calc_name ) ){
		core::pose::metrics::PoseMetricCalculatorOP nlcontacts_calc = new protocols::toolbox::PoseMetricCalculators::NonlocalContactsCalculator();
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( nonlocalcontacts_calc_name, nlcontacts_calc );
	}


	//first general pose calculators

	utility::vector1< std::pair< std::string, std::string > > total_pose_calculators;

	if( !core::options::option[core::options::OptionKeys::enzdes::no_packstat_calculation] ){
		total_pose_calculators.push_back( std::pair< std::string, std::string > ( packstat_calc_name, "total_packstat") );
		total_pose_calculators.push_back( std::pair< std::string, std::string > ( noligpackstat_calc_name, "total_packstat") );
	}

	total_pose_calculators.push_back( std::pair< std::string, std::string > ( burunsat_calc_name, "all_bur_unsat_polars") );
	total_pose_calculators.push_back( std::pair< std::string, std::string > ( hbond_calc_name, "all_Hbonds") );
	total_pose_calculators.push_back( std::pair< std::string, std::string > ( nonlocalcontacts_calc_name, "total_nlcontacts") );

	residue_calculators_.insert( std::pair< Size, utility::vector1< std::pair< std::string, std::string > > > ( 0, total_pose_calculators ) );

	//if native compare is requested
	if( core::options::option[core::options::OptionKeys::enzdes::compare_native].user() ){
		native_compare_calculators_.push_back( std::pair< std::string, std::string > ( burunsat_calc_name, "all_bur_unsat_polars") );
		if( !core::options::option[core::options::OptionKeys::enzdes::no_packstat_calculation] ){
			native_compare_calculators_.push_back( std::pair< std::string, std::string > ( packstat_calc_name, "total_packstat") );
		}
		native_compare_calculators_.push_back( std::pair< std::string, std::string > ( hbond_calc_name, "all_Hbonds") );
		native_compare_calculators_.push_back( std::pair< std::string, std::string > ( nonlocalcontacts_calc_name, "total_nlcontacts") );
	}

	//general pose calculators done, now onto the residue specific calculators

	for( core::Size i = 1;  i<= pose.total_residue(); ++i){

		if( ! is_catalytic_position( i ) ) continue;

		utility::vector1< std::pair< std::string, std::string > > calculators_this_res;

		//first a couple of ligand specific calculators ( interface SASA and interface energetics)
		if( pose.residue_type( i ).is_ligand() ){

			if( pose.conformation().num_chains() != 2){
				utility_exit_with_message( "Error: the input pose has more than two chains, code not ready to deal with this yet.");
			}
			Size lig_chain = pose.chain( i );
			Size prot_chain = pose.chain( 1 );   //we're making the not so wild assumption that the the first residue in the pose belongs to the protein
			std::string lig_ch_string = utility::to_string( lig_chain );
			std::string prot_ch_string = utility::to_string( prot_chain );
			if( lig_chain == prot_chain ) { utility_exit_with_message( "WTF?!? ligand and residue 1 are on the same chain... " );}

			std::string lig_interface_neighbor_calc_name = "neighbor_def_" + prot_ch_string + "_" + lig_ch_string;
			std::string lig_dsasa_calc_name = "dsasa_" + prot_ch_string + "_" + lig_ch_string;
			std::string lig_interface_e_calc_name = "interf_E_" + prot_ch_string + "_" + lig_ch_string;

			if( !CalculatorFactory::Instance().check_calculator_exists( lig_interface_neighbor_calc_name ) ){
				PoseMetricCalculatorOP lig_neighbor_calc = new protocols::toolbox::PoseMetricCalculators::InterfaceNeighborDefinitionCalculator( prot_chain, lig_chain );
				CalculatorFactory::Instance().register_calculator( lig_interface_neighbor_calc_name, lig_neighbor_calc );
			}
			if( !CalculatorFactory::Instance().check_calculator_exists( lig_dsasa_calc_name ) ){
				PoseMetricCalculatorOP lig_dsasa_calc = new protocols::toolbox::PoseMetricCalculators::InterfaceSasaDefinitionCalculator( prot_chain, lig_chain );
				CalculatorFactory::Instance().register_calculator( lig_dsasa_calc_name, lig_dsasa_calc );
			}
			if( !CalculatorFactory::Instance().check_calculator_exists( lig_interface_e_calc_name ) ){
				PoseMetricCalculatorOP lig_interf_E_calc = new protocols::toolbox::PoseMetricCalculators::InterfaceDeltaEnergeticsCalculator( lig_interface_neighbor_calc_name );
				CalculatorFactory::Instance().register_calculator( lig_interface_e_calc_name, lig_interf_E_calc );
			}

			calculators_this_res.push_back( std::pair< std::string, std::string > ( lig_interface_e_calc_name, "weighted_total") );
			calculators_this_res.push_back( std::pair< std::string, std::string > ( lig_dsasa_calc_name, "frac_ch2_dsasa") );

		} //ligand specific calculators set up

		//now calculators for every residue, for starters number of Hbonds and number of buried polars
		calculators_this_res.push_back( std::pair< std::string, std::string > ( hbond_calc_name, "residue_Hbonds") );
		calculators_this_res.push_back( std::pair< std::string, std::string > ( burunsat_calc_name, "residue_bur_unsat_polars") );

		//and for protein residues, we also want to know the packstat
		if( pose.residue_type( i ).is_protein() && ( !core::options::option[core::options::OptionKeys::enzdes::no_packstat_calculation] ) ){
			calculators_this_res.push_back( std::pair< std::string, std::string > ( packstat_calc_name, "residue_packstat") );
			calculators_this_res.push_back( std::pair< std::string, std::string > ( noligpackstat_calc_name, "residue_packstat") );
		}
		//debug
		//for( utility::vector1< std::pair< std::string, std::string > >::iterator calc_it = calculators_this_res.begin();
		//		 calc_it != calculators_this_res.end(); calc_it++ ){
		//	std::cerr << "calculator " << calc_it->first << " created for residue " << *res_it << std::endl;
		//}
		//debug over

		residue_calculators_.insert( std::pair< Size, utility::vector1< std::pair< std::string, std::string > > > ( i, calculators_this_res ) );

	} // loop over all (catalytic) residues

}



bool
EnzdesBaseProtocol::exchange_ligands_in_pose(
	core::pose::Pose & pose,
	bool check_bb_clashes,
	core::scoring::ScoreFunctionCOP scofx
){

	if( !lig_superposition_file_read_ ) read_ligand_superposition_file(core::options::option[core::options::OptionKeys::enzdes::change_lig].value() );

	utility::vector1< core::Size > ligs_to_exchange;
	for( core::Size i = 1; i <= pose.total_residue(); ++i ){
		if( pose.residue(i).name3() == res_to_superimpose_.first ) ligs_to_exchange.push_back( i );
	}

	core::conformation::Residue new_res( restype_set_->name_map( res_to_superimpose_.second ), true );

	for( utility::vector1< core::Size >::const_iterator pos_it = ligs_to_exchange.begin();
			 pos_it != ligs_to_exchange.end(); ++pos_it )
		{

			pose.replace_residue( *pos_it, new_res, atoms_to_superimpose_on_ );

			//we should probably do this
			pose.update_residue_neighbors();

			// if we don't care for backbone clashes, we are already done
			if( check_bb_clashes ){

				//otherwise, we now have to check for bb_clashes with the new residue

				utility::vector1< core::conformation::ResidueOP > accepted_rotamers;
				get_non_bb_clashing_rotamers( pose, *pos_it, scofx, accepted_rotamers );

				//DEBUG SHIT
				//core::pose::Pose debugpose = pose;
				//core::Size dei(0);
				//for( utility::vector1< core::conformation::ResidueOP >::iterator rot_it = accepted_rotamers.begin(); rot_it != accepted_rotamers.end(); ++rot_it ){
				//dei++;
				//debugpose.replace_residue( *pos_it, **rot_it, true );
				//debugpose.dump_pdb( "rodebug"+utility::to_string( dei ) );
				//}
				//DEBUG SHIT over

				if( accepted_rotamers.size() == 0 ) return false;

				tr << "There are " << accepted_rotamers.size() << " rotamers that don't clash with the backbone." << std::endl;

				//we should put one of the non_clashing rotamers into the pose, just to make sure...
				pose.replace_residue( *pos_it, *accepted_rotamers[1], true );
			}

			//now we also have to change the remarks
			core::io::pdb::Remarks & remarks = pose.pdb_info()->remarks();
			for( std::vector< core::io::pdb::RemarkInfo >::iterator rem_it = remarks.begin(); rem_it != remarks.end(); ++rem_it ){

				std::string chainA(""), resA(""),chainB(""),resB("");
				core::Size cst_block(0), exgeom_id(0);
				int seqposA(0), seqposB(0);
				if( enzutil::split_up_remark_line( rem_it->value, chainA, resA, seqposA, chainB, resB, seqposB, cst_block, exgeom_id ) ){

					bool line_changed( false );

					if( resA == res_to_superimpose_.first ){
						if( seqposA == (int) *pos_it || seqposA == 0 ) {
							resA = res_to_superimpose_.second;
							line_changed=true;
						}
					}
					else if( resB == res_to_superimpose_.first ){
						if( seqposB == (int) *pos_it || seqposB == 0 ) {
							resB = res_to_superimpose_.second;
							line_changed=true;
						}
					}

					if( line_changed ){
						rem_it->value = enzutil::assemble_remark_line( chainA, resA, seqposA, chainB, resB, seqposB, cst_block, exgeom_id );
					}
				}
			}
		} //lig_positions to exchange


	return true;

} //exchange_ligands_in_pose


utility::vector1< std::pair< std::string, std::string > > const &
EnzdesBaseProtocol::calculators_for_residue( core::Size res ) const
{

	std::map< Size, utility::vector1< std::pair< std::string, std::string > > >::const_iterator calc_it = residue_calculators_.find( res );

	if( calc_it == residue_calculators_.end() ){
		utility_exit_with_message("Error: no specific calculators for requested residue number were found");
	}

	return calc_it->second;
}


core::Real
EnzdesBaseProtocol::design_targets_score(
	core::pose::Pose const & pose
) const
{
	core::Real return_val(0.0);
	using namespace core::scoring;

	for( std::set< core::Size >::const_iterator des_it = design_targets_.begin();
			 des_it != design_targets_.end(); ++des_it ){

		return_val += pose.energies().residue_total_energy( *des_it );
	}

	return return_val;
}//design targets score


void
EnzdesBaseProtocol::examine_pose(
	core::pose::Pose const & pose,
	utility::vector1< core::io::silent::SilentEnergy > & silent_Es
)
{

	using namespace core::io::silent;

	silent_Es.clear();

	setup_pose_metric_calculators( pose );

	bool separate_out_constraints = false;
	utility::vector1< std::string >::const_iterator cstfind = find( relevant_scoreterms_.begin(), relevant_scoreterms_.end(),"all_cst");
	if( cstfind != relevant_scoreterms_.end() ) separate_out_constraints = true;

  //first write out the relevant score terms for the pose total
  for( utility::vector1< std::string >::const_iterator sco_it = relevant_scoreterms_.begin();
       sco_it != relevant_scoreterms_.end(); ++sco_it ){
    std::string sco_name = *sco_it;
    int width = std::max( 10, (int) sco_name.length() + 3 );

    SilentEnergy new_se;
    if( *sco_it == "all_cst" ) {
      new_se = SilentEnergy ( sco_name, sum_constraint_scoreterms(pose, -1 ), 1, width);
    }
		else if( separate_out_constraints && ( *sco_it == "total_score" ) ){
			core::Real desired_value = pose.energies().total_energies()[ core::scoring::score_type_from_name( *sco_it ) ] - sum_constraint_scoreterms(pose, -1 );
			new_se = SilentEnergy(sco_name, desired_value, 1, width);
		}
    else{
      new_se = SilentEnergy ( sco_name, pose.energies().total_energies()[ core::scoring::score_type_from_name( *sco_it ) ] * pose.energies().weights()[ core::scoring::score_type_from_name( *sco_it ) ], 1 ,width);
    }
    silent_Es.push_back( new_se );

  }

	//pose metric calculators for pose total
	std::map< Size, utility::vector1< std::pair< std::string, std::string > > >::const_iterator totcalc_it = residue_calculators_.find( 0 );
	if( totcalc_it != residue_calculators_.end() ){

    utility::vector1< std::pair< std::string, std::string > > const & tot_calculators = totcalc_it->second;
		for( utility::vector1< std::pair< std::string, std::string > >::const_iterator calc_it = tot_calculators.begin();
				 calc_it != tot_calculators.end(); ++calc_it){

			std::string calc_name = "tot_" + calc_it->first;
			int width = std::max( 10, (int) calc_name.length() + 3 );

			core::Real calc_value;

			//following lines fairly hacky, but don't know a better solution at the moment
			if( calc_it->first == "hbond_pm" || calc_it->first == "burunsat_pm" || calc_it->first == "NLconts_pm" ){
				core::util::MetricValue< core::Size > mval_size;
				pose.metric( calc_it->first, calc_it->second, mval_size );
				calc_value = mval_size.value();
			}
			else{
				core::util::MetricValue< core::Real > mval_real;
				pose.metric( calc_it->first, calc_it->second, mval_real );
				calc_value = mval_real.value();
			}

			SilentEnergy new_se( calc_name, calc_value, 1, width);
			silent_Es.push_back( new_se );
		}

	}

	//done with pose totals

  //then write out the relevant scoreterms (and potentially pose metrics) for each of the special residues
	Size spec_res_counter(0);
	utility::vector1< Size > special_res = catalytic_res( pose );

  for( utility::vector1<Size>::const_iterator res_it = special_res.begin(); res_it != special_res.end(); res_it++ ){

    spec_res_counter++;
    //for convenience, the sequence number of the residue will be written out
    std::stringstream temp;
    temp << spec_res_counter;
    std::string spec_res_name = "SR_" + temp.str();
    //std::cerr << "name for res " << *res_it << " is " << spec_res_name ;
    SilentEnergy res_name(spec_res_name, *res_it, 1, 10);
    silent_Es.push_back( res_name );

    for( utility::vector1< std::string >::const_iterator sco_it = relevant_scoreterms_.begin();
       sco_it != relevant_scoreterms_.end(); ++sco_it ){

      std::string sco_name = spec_res_name + "_" +  *sco_it ;
      int width = std::max( 10, (int) sco_name.length() + 3 );

      SilentEnergy new_se;
      if( *sco_it == "all_cst" ) {
				new_se = SilentEnergy ( sco_name, sum_constraint_scoreterms(pose, *res_it), 1, width);
      }
			else if( separate_out_constraints && ( *sco_it == "total_score" ) ){
				core::Real desired_value = pose.energies().residue_total_energies( *res_it )[ core::scoring::score_type_from_name( *sco_it ) ] - sum_constraint_scoreterms(pose, *res_it );
				new_se = SilentEnergy(sco_name, desired_value, 1, width);
			}
      else{
      new_se = SilentEnergy ( sco_name, pose.energies().residue_total_energies( *res_it )[ core::scoring::score_type_from_name( *sco_it ) ] * pose.energies().weights()[ core::scoring::score_type_from_name( *sco_it ) ], 1 ,width);
      }

     silent_Es.push_back( new_se );
    }//loop over relevant scoreterms


    //if there are calculators that need to be evaluated for this residue, let's do that now
    //note: section still under development, right now only calculators that return reals are supported
    std::map< Size, utility::vector1< std::pair< std::string, std::string > > >::const_iterator res_calc_it = residue_calculators_.find( *res_it );
    if( res_calc_it != residue_calculators_.end() ){

      utility::vector1< std::pair< std::string, std::string > > calculators_this_res = res_calc_it->second;
      for( utility::vector1< std::pair< std::string, std::string > >::iterator calc_it = calculators_this_res.begin();
					 calc_it != calculators_this_res.end(); calc_it++ ){

				std::string res_calc_name = spec_res_name + "_" + calc_it->first;
				int width = std::max( 10, (int) res_calc_name.length() + 3 );

				core::Real calc_value;

				core::util::MetricValue< core::Real > mval_real;
				core::util::MetricValue< utility::vector1< core::Size > >mval_sizevec;
				core::util::MetricValue< utility::vector1< core::Real > >mval_realvec;

				//following lines fairly hacky, but don't know a better solution at the moment
				if( ( calc_it->first == "hbond_pm") || ( calc_it->first == "burunsat_pm") ){
					pose.metric( calc_it->first, calc_it->second, mval_sizevec );
					calc_value = mval_sizevec.value()[*res_it];
				}
				else if( (calc_it->first == "pstat_pm") || (calc_it->first == "nlpstat_pm" ) ) {
					pose.metric( calc_it->first, calc_it->second, mval_realvec );
					calc_value = mval_realvec.value()[*res_it];
				}
				else {
					pose.metric( calc_it->first, calc_it->second, mval_real );
					calc_value = mval_real.value();
				}
				//std::cerr << " hehe, just executed pose metric for " << calc_it->first << " calculator   ";


				//special case: if this is an interface calculation, we do not want to include constraint terms
				//hacky at the moment, haven't figured out yet how to do this really clean
				if( separate_out_constraints && ( calc_it->first == "interf_E_1_2") ){
					calc_value = calc_value - ( 2 * sum_constraint_scoreterms(pose, *res_it) );
				}
				SilentEnergy new_se( res_calc_name, calc_value, 1, width);
				silent_Es.push_back( new_se );

      } // for calculators this res

    }// if calculators for this res
    //else std::cerr << "resi " << *res_it << " has no calcs." << std::endl;

  }//loop over special residues


	//finally, if comparison to native is requested

	if( core::options::option[core::options::OptionKeys::enzdes::compare_native].user() ){
		native_comp_->compare_to_native( pose, native_compare_calculators_, scorefxn_, silent_Es );
	}

} //examine pose


void
EnzdesBaseProtocol::remap_resid(
	core::pose::Pose const & pose,
	core::sequence::SequenceMapping const & smap
)
{
	cst_io_->remap_resid( smap );
	design_targets( pose );
}



void
EnzdesBaseProtocol::generate_explicit_ligand_rotamer_poses(
	core::pose::Pose const & orig_pose,
	utility::vector1< core::pose::PoseOP > & ligrot_poses,
	core::scoring::ScoreFunctionCOP scofx
)
{

	ligrot_poses.clear();

	//std::cerr << "staring generate explictligrot function, looking for residues of type " << core::options::option[ core::options::OptionKeys::enzdes::process_ligrot_separately ].value() << std::endl;

	for( core::Size i = 1; i <= orig_pose.total_residue(); ++ i ){

		if( orig_pose.residue( i ).name3() != core::options::option[ core::options::OptionKeys::enzdes::process_ligrot_separately ].value() ) continue;

		utility::vector1< core::conformation::ResidueOP > accepted_rotamers;
		get_non_bb_clashing_rotamers( orig_pose, i, scofx, accepted_rotamers );

		for( utility::vector1< core::conformation::ResidueOP >::iterator rot_it = accepted_rotamers.begin();
				 rot_it != accepted_rotamers.end(); ++rot_it ){

			core::pose::PoseOP lig_pose = new core::pose::Pose( orig_pose );
			lig_pose->replace_residue( i, **rot_it, true );

			ligrot_poses.push_back( lig_pose );

		}
		//std::cerr << "There are " << accepted_rotamers.size() << " non_bb clashing residues at positon " << i << std::endl;
	}

} //generate_explicit_ligand_rotamer_poses


void
EnzdesBaseProtocol::read_ligand_superposition_file( std::string filename )
{

	lig_superposition_file_read_ = false;
	atoms_to_superimpose_on_.clear();

	bool switch_info_found( false );

	utility::io::izstream filedata( filename.c_str() );
	std::istringstream line_stream;
	std::string line("");

	if ( !filedata ) {
		std::cerr << "ERROR:: Unable to open ligand superposition info file: "
				<< filename << std::endl;
		std::exit( 1 );
	}

	while( !filedata.eof() ) {
		std::string key("");
		getline(filedata,line);
		line_stream.clear();
		line_stream.str(line);
		line_stream >> key;

		if( key == "SWITCH_NAME" ){

			line_stream >> res_to_superimpose_.first >> res_to_superimpose_.second;
			switch_info_found = true;

		}

		if( key == "SUPERIMPOSE" ){

			std::string buf1(""), buf2("");
			line_stream >> buf1 >> buf2;
			atoms_to_superimpose_on_.push_back( std::pair< std::string, std::string > ( buf1, buf2 ) );

		}
	}

	if( !switch_info_found ){
		utility_exit_with_message("Error: "+filename+" does not specify which residues to switch!");
	}

	if( atoms_to_superimpose_on_.size() < 3 ){
		utility_exit_with_message("Error: "+filename+" specifies less than 3 atoms to superimpose on, unambiguous superposition not possible");
	}

	tr << "read superposition info from " << filename << " ...";
	lig_superposition_file_read_ = true;

} //read_ligand_superposition_file

}//namespace enzdes
}//namespace protocols
