// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/enzdes/EnzdesFixBBProtocol.cc
///
/// @brief
/// @author Florian Richter


#include <protocols/enzdes/EnzdesFixBBProtocol.hh>
#include <protocols/enzdes/EnzdesBaseProtocol.hh>
#include <protocols/enzdes/EnzConstraintIO.hh>
#include <protocols/enzdes/EnzdesMovers.hh>
#include <core/chemical/ResidueType.hh>
#include <core/options/option.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunction.hh>
#include <protocols/ligand_docking/LigandDockProtocol.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/PackRotamersMover.hh>

#include <core/util/Tracer.hh>


// option key includes

#include <core/options/keys/enzdes.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>


namespace protocols{
namespace enzdes{

static core::util::Tracer tr("protocols.enzdes.EnzdesFixBBProtocol");

EnzdesFixBBProtocol::EnzdesFixBBProtocol() : EnzdesBaseProtocol() {}

void
EnzdesFixBBProtocol::apply(
	core::pose::Pose & pose
){


	using namespace protocols::moves;
	using namespace core::pack::task;


	silent_Es_.clear();
	// Scoring function already set up by superclass
	tr.Info << "starting apply function..." << std::endl;

	//set the native pose if requested
	if( ! core::options::option[core::options::OptionKeys::in::file::native].user() ){

		core::pose::PoseOP natpose = new core::pose::Pose( pose );
		(*scorefxn_)( *natpose );
		this->set_native_pose( natpose );
	}


	//set up constraints (read cstfile, do mapping, etc, then add to pose)
	if( core::options::option[core::options::OptionKeys::enzdes::cstfile].user() ){
		enable_constraint_scoreterms();
		setup_enzdes_constraints( pose, false );
	}

	//create packer task (read resfile, etc)
	PackerTaskOP design_pack_task;

	tr.Info << "Done setting up the task and constraints... " << std::endl;
	//score pose to make sure everything is initialised correctly
	(*scorefxn_)( pose );

	//cst opt stage, if demanded
	if(core::options::option[core::options::OptionKeys::enzdes::cst_opt]){
		design_pack_task =  create_enzdes_pack_task( pose );
		tr.Info << "starting cst_opt minimization..." << std::endl;
		cst_minimize(pose, design_pack_task, true);
		(*scorefxn_)( pose );
		tr.Info << "done cst_opt minimization." << std::endl;
	}


	if(core::options::option[core::options::OptionKeys::enzdes::cst_predock]){
		design_pack_task =  create_enzdes_pack_task( pose );
		PredesignPerturbMoverOP predock = new PredesignPerturbMover(design_pack_task,this);
    predock->set_ligand( get_ligand_id(pose, pose.num_jump()) );
    predock->apply(pose);
    (*scorefxn_)( pose );
	}


	if(core::options::option[core::options::OptionKeys::enzdes::cst_design]){

		design_pack_task = create_enzdes_pack_task( pose ); //make a new task in case the ligand has moved a lot
		tr.Info << "starting cst_design, " << core::options::option[core::options::OptionKeys::enzdes::design_min_cycles] << " cycles of design/minimization ... " << std::endl;

		core::Size design_min_cycles = core::options::option[core::options::OptionKeys::enzdes::design_min_cycles];

		bool favor_native_res(false);
		if( core::options::option[core::options::OptionKeys::enzdes::favor_native_res].user() ) favor_native_res = true;

		enzdes_pack( pose, design_pack_task, scorefxn_, design_min_cycles, false, true, favor_native_res );

	} //if cst_design

	else if( core::options::option[core::options::OptionKeys::enzdes::cst_min] ){

		design_pack_task = create_enzdes_pack_task( pose );

		cst_minimize(pose, design_pack_task);

		(*scorefxn_)( pose );
	}

	PackerTaskOP repack_task;


	//do a repack without constraints
	if( ! core::options::option[core::options::OptionKeys::enzdes::no_unconstrained_repack]){

		remove_enzdes_constraints( pose, true );
		(*scorefxn_)( pose );
		repack_task = create_enzdes_pack_task( pose, false ); //remake task in case the ligand has moved a lot
		tr.Info << "Starting after design unconstrained repack/minimization... " << std::endl;
		PackRotamersMoverOP enzdes_repack = new PackRotamersMover(scorefxn_, repack_task);
		enzdes_repack->apply( pose );

		if(core::options::option[core::options::OptionKeys::enzdes::cst_min]) cst_minimize(pose, repack_task);

		//and turn constraints back on for the final scoring
		add_pregenerated_enzdes_constraints( pose );
		tr.Info <<"Finished after design unconstrained repack/minimization... " << std::endl;

		(*scorefxn_)( pose );

	}

	if(core::options::option[core::options::OptionKeys::enzdes::cst_dock] ){


		//note: this is not really ready to go yet, still to be developed
		//to do: write a wrapper that executes the docking protocol a number
		//of times, either till a maximum number of decoys is produced, or
		//until a decoy is found that has a better energy than the designed pose

		tr.Info << "Starting ligand docking... " << std::endl;
		remove_enzdes_constraints( pose, false );
		//write stuff for ligand docking protocol
		protocols::ligand_docking::LigandDockProtocolOP dock_lig_protocol = new protocols::ligand_docking::LigandDockProtocol();
		dock_lig_protocol->apply( pose );

		add_pregenerated_enzdes_constraints( pose );
		(*scorefxn_)( pose );

	}


} //apply function


void
EnzdesFixBBProtocol::register_options()
{
	using namespace core::options;

	option.add_relevant( OptionKeys::enzdes::cstfile );
	option.add_relevant( OptionKeys::enzdes::cst_opt );
	option.add_relevant( OptionKeys::enzdes::cst_predock );

	PredesignPerturbMover::register_options();

	option.add_relevant( OptionKeys::enzdes::cst_design );
	option.add_relevant( OptionKeys::enzdes::design_min_cycles );
	option.add_relevant( OptionKeys::enzdes::no_unconstrained_repack );

}

} //namespace enzdes
} //namespace protocols

