// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/protocols/enzdes/EnzdesMovers.hh
/// @brief a collection of movers that are used at different stages in enzyme design
/// @author Sinisa Bjelic sinibjelic@gmail.com, Florian Richter, floric@u.washington.edu

#include <protocols/enzdes/EnzdesMovers.hh>
#include <protocols/enzdes/EnzdesFixBBProtocol.hh>
#include <protocols/enzdes/EnzdesBaseProtocol.hh>
#include <protocols/enzdes/EnzConstraintIO.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/toolbox/pose_manipulation.hh>

#include <core/chemical/ResidueTypeSet.fwd.hh>
#include <core/conformation/Residue.fwd.hh>
#include <core/chemical/util.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/options/option.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/pose/Pose.hh>
#include <core/pack/task/PackerTask.fwd.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/Constraints.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/MultiConstraint.hh>
#include <core/scoring/constraints/MultiConstraint.fwd.hh>
#include <core/scoring/constraints/Func.fwd.hh>
#include <core/scoring/constraints/AtomPairConstraint.hh>
#include <core/scoring/ScoreFunction.fwd.hh>
#include <core/util/Tracer.hh>
#include <core/id/AtomID.hh>

#include <utility/string_util.hh>
#include <utility/io/izstream.hh>

// option key includes
#include <core/options/keys/enzdes.OptionKeys.gen.hh>

namespace protocols {
namespace enzdes {

static core::util::Tracer mv_tr("protocols.enzdes.PredesignPerturbMover");

	PredesignPerturbMover::PredesignPerturbMover():protocols::moves::RigidBodyPerturbMover() {}

	PredesignPerturbMover::PredesignPerturbMover(core::pack::task::PackerTaskCOP task_in,  EnzdesBaseProtocolCOP enzdes_prot_in) : protocols::moves::RigidBodyPerturbMover(),task_(task_in),enzdes_prot_(enzdes_prot_in) {}

void
PredesignPerturbMover::set_docking_pose(
	core::pose::Pose &pose,
	core::pack::task::PackerTaskCOP task,
	EnzdesBaseProtocolCOP enzdes_prot)
{
	for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {
		if( task->design_residue(i) && (! enzdes_prot->is_catalytic_position( i ))) positions_to_replace_.push_back( i );
	}
	protocols::toolbox::pose_manipulation::construct_poly_ala_pose( pose, positions_to_replace_, true, true, true );
}

void
PredesignPerturbMover::reinstate_pose(
	core::pose::Pose &pose,
	core::pose::Pose const &old_Pose)
{
	core::Size ires;
  for(core::Size i=1, i_end=positions_to_replace_.size(); i<=i_end; ++i) {
		ires=positions_to_replace_[i];
 		pose.replace_residue( ires, old_Pose.residue(ires), true);
	}
}

void
PredesignPerturbMover::add_constrained_lig_atoms_from_multiconstraint(
  core::scoring::constraints::MultiConstraintCOP real_multi_constraint
) {
  core::scoring::constraints::ConstraintCOPs multi_constraint_members;
  multi_constraint_members=real_multi_constraint->member_constraints();
	for (core::scoring::constraints::ConstraintCOPs::const_iterator MC_it=multi_constraint_members.begin(); MC_it!=multi_constraint_members.end(); MC_it++) {
		if ( ((*MC_it)->type()) == "AtomPair") {
			( ((*MC_it)->atom(1)).rsd() == ligand_seqpos_) ? add_constrained_lig_atom( ((*MC_it)->atom(1)).atomno() ) : add_constrained_lig_atom( ((*MC_it)->atom(2)).atomno() ) ;
     // mv_tr.Info << "adding AtomPair" << std::endl;
     // mv_tr.Info << ((*MC_it)->atom(1)).atomno() << std::endl;
		} else if ( ((*MC_it)->type()) == "MultiConstraint") {
			add_constrained_lig_atoms_from_multiconstraint( (dynamic_cast <core::scoring::constraints::MultiConstraint const * > ((*MC_it)())) );
    } else if ( ((*MC_it)->type()) == "AmbiguousConstraint") {
      add_constrained_lig_atoms_from_multiconstraint( (dynamic_cast <core::scoring::constraints::MultiConstraint const * > ((*
MC_it)())) );
		}
	}
}

void
PredesignPerturbMover::add_constrained_lig_atom(core::Size atom_no){
  for(utility::vector1< core::Size >::const_iterator it=constrained_lig_atoms_.begin(); it !=constrained_lig_atoms_.end(); it++){
    if ( (*it) == atom_no)
			return;
	}
	constrained_lig_atoms_.push_back(atom_no);
  mv_tr.Info << "Constrained ligand atom: " << std::endl;
  mv_tr.Info << atom_no << std::endl;
}

void
PredesignPerturbMover::find_constraints_to_ligand(
core::pose::Pose const & pose
){
  using namespace core::scoring::constraints;

  //Get a set of constraints for a ligand.
  //It is a vector of ResidueConstraints.
  ConstraintSetCOP constraint_set;
  constraint_set=pose.constraint_set();
  ConstraintSet::ResiduePairConstraintsIterator rpc_start=constraint_set->residue_pair_constraints_begin(ligand_seqpos_);
  ConstraintSet::ResiduePairConstraintsIterator rpc_end=constraint_set->residue_pair_constraints_end(ligand_seqpos_);

  // Each residue constraint is a map container
  // of Size and ConstraintsOP, defining a residue number
 	//and constraints object respectively.
  for (; rpc_start !=rpc_end; rpc_start++) {
    //Constraints object is a pointer vector of constraint objects.
    // Each contraint object contains type of constraints.
    ConstraintsOP constraints;
		constraints=rpc_start->second;
		Constraints::const_iterator it_begin=constraints->begin();
  	Constraints::const_iterator it_end=constraints->end();
    for (; it_begin!=it_end; it_begin++) {
		   //mv_tr.Info <<(*it_begin)->type() << std::endl;
			  if ( ((*it_begin)->type()) == "MultiConstraint") {
  //        MultiConstraintCOP real_multi_constraint = dynamic_cast <MultiConstraint const * > ((*it_begin)());
  //        add_constrained_lig_atoms_from_multiconstraint(real_multi_constraint);
          add_constrained_lig_atoms_from_multiconstraint( dynamic_cast <MultiConstraint const * > ((*it_begin)()) );
			  } else if ( ((*it_begin)->type()) == "AmbiguousConstraint") {
          add_constrained_lig_atoms_from_multiconstraint( dynamic_cast <MultiConstraint const * > ((*it_begin)()) );
				} else if ( ((*it_begin)->type()) == "AtomPair") {
        // if ligand equals residue of atom1 then assign atom1
        // otherwise asign the other atom
      ( ((*it_begin)->atom(1)).rsd() == ligand_seqpos_ ) ? add_constrained_lig_atom( ((*it_begin)->atom(1)).atomno() ) : add_constrained_lig_atom( ((*it_begin)->atom(2)).atomno() ) ;
			  }
 		}
	}
}

core::Vector
PredesignPerturbMover::find_geometric_center_for_constrained_lig_atoms(
core::pose::Pose const &pose
){
  // geometric_center=1000000; //for testing
	core::Vector geometric_center;
  geometric_center=0.0;
  for(utility::vector1< core::Size >::const_iterator it=constrained_lig_atoms_.begin(); it !=constrained_lig_atoms_.end(); it++){
   geometric_center+=pose.residue(ligand_seqpos_).xyz(*it);
  }

  //check if there are any atoms in constraind_lig_atoms_
  if (constrained_lig_atoms_.size()!=0) {
	  geometric_center /=constrained_lig_atoms_.size();
	} else {
  	std::cerr << "There are no atom pair constraints. Predocking is done relative to a constrained ligand atom!" << std::endl;
	}
  return geometric_center;
}

void
PredesignPerturbMover::apply(
core::pose::Pose & pose
){
  core::pose::Pose org_Pose = pose; //copy old pose
  //make a poly ala of the designable
  set_docking_pose( pose, task_, enzdes_prot_ );
  trans_magnitude(core::options::option[core::options::OptionKeys::enzdes::trans_magnitude]);
  rot_magnitude(core::options::option[core::options::OptionKeys::enzdes::rot_magnitude]);
  //core::scoring::ScoreFunction reduced_scorefxn;
  //reduced_scorefxn=*(enzdes_prot_->reduced_scorefxn());
  protocols::moves::MonteCarloOP MCpredock = new protocols::moves::MonteCarlo(pose, *(enzdes_prot_->reduced_scorefxn()), 2.0 /* temperature, from RosettaLigand paper */);
  MCpredock->reset( pose );
  MCpredock->reset_counters();
  core::Size n_dock=core::options::option[core::options::OptionKeys::enzdes::dock_trials];
  mv_tr.Info << "starting predocking ... " << std::endl;
  //itereate through all constraints in the pose and check for the constrained atoms
  find_constraints_to_ligand(pose);
  for(core::Size i=1; i<= n_dock; i++) {
    rot_center_= find_geometric_center_for_constrained_lig_atoms(pose);

    core::kinematics::Jump flexible_jump = pose.jump( rb_jump_ );
    core::kinematics::Stub downstream_stub = pose.conformation().downstream_jump_stub( rb_jump_ );
    flexible_jump.set_rb_center( dir_, downstream_stub, rot_center_ );

    flexible_jump.gaussian_move( dir_, trans_mag_, rot_mag_ );
    pose.set_jump( rb_jump_, flexible_jump );
    MCpredock->boltzmann(pose);
  }
  MCpredock->show_counters();
  MCpredock->recover_low( pose );
  mv_tr.Info << "... done predocking" << std::endl;
  //put back the old pose
  reinstate_pose(pose,org_Pose);
}

} //enzdes
} //protocols

