// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file PoseEvaluator
/// @brief PoseEvaluator
/// @detailed
///
///
/// @author Oliver Lange



// Unit Headers
#include <protocols/evaluation/ConstraintEvaluator.hh>

// Package Headers
#include <protocols/abinitio/MaxSeqSepConstraintSet.hh>

// Project Headers
#include <core/io/silent/SilentStruct.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/ConstraintIO.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/NamedAtomPairConstraint.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreType.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>

// Utility headers
#include <core/util/Tracer.hh>
#include <core/scoring/rms_util.hh>
#include <utility/file/file_sys_util.hh>
// C++ headers

static core::util::Tracer tr("protocols.evaluation.ConstraintEvaluator");

namespace protocols {
namespace evaluation {

using namespace core;
using namespace scoring;
using namespace constraints;

ConstraintEvaluator::ConstraintEvaluator( std::string tag, ConstraintSet const& cst, Size viol_level, Real threshold,  Size max_seq_sep )
	: name_( tag ),
		constraints_( new ConstraintSet( cst ) ),
		file_name_( "" ),
		viol_level_ ( viol_level  ),
		threshold_( threshold ),
		max_seq_sep_( max_seq_sep )
{}

ConstraintEvaluator::ConstraintEvaluator( std::string tag, ConstraintCOPs const& csts, Size viol_level, Real threshold, Size max_seq_sep )
	: name_( tag ),
		constraints_( new ConstraintSet() ),
		file_name_( "" ),
		viol_level_ ( viol_level  ),
		threshold_( threshold ),
		max_seq_sep_( max_seq_sep )
{
	constraints_->add_constraints( csts );
}

ConstraintEvaluator::ConstraintEvaluator( std::string tag, std::string file_name, Size viol_level, Real threshold, Size max_seq_sep )
	: name_( tag ),
		constraints_( NULL ),
		file_name_( file_name ),
		viol_level_ ( viol_level  ),
		threshold_( threshold ),
		max_seq_sep_( max_seq_sep )
{
	//check file exists
	if ( !utility::file::file_exists( file_name_ ) ) {
		utility_exit_with_message(" could not find file " + file_name_ );
	}
}


void ConstraintEvaluator::prepare_pose( core::pose::Pose const& pose_in, core::pose::Pose& pose ) const {

	if ( !constraints_ ) {
		runtime_assert( utility::file::file_exists( file_name_) );

		ConstraintOP new_atom_pair_type( new NamedAtomPairConstraint( id::NamedAtomID(), id::NamedAtomID(), NULL) );
		ConstraintOP orig_atom_pair_type( ConstraintIO::get_cst_factory().newConstraint( new_atom_pair_type->type() ));
		ConstraintIO::get_cst_factory().add_type( new_atom_pair_type );

		constraints_ = ConstraintIO::get_instance()->read_constraints( file_name_, new ConstraintSet, pose );

		//restore original cst-type
		ConstraintIO::get_cst_factory().add_type( orig_atom_pair_type );

	}

	runtime_assert( constraints_ );

	abinitio::MaxSeqSepConstraintSetOP new_cst( NULL );
	if ( max_seq_sep_ > 0 ) {
		new_cst = new abinitio::MaxSeqSepConstraintSet( *constraints_, pose.fold_tree() );
		new_cst->set_max_seq_sep( max_seq_sep_ );
	} else {
		//if not specified  we copy the max_seq_separation if present
		abinitio::MaxSeqSepConstraintSetCOP ms_set = dynamic_cast< abinitio::MaxSeqSepConstraintSet const* >( pose_in.constraint_set().get() );
		if ( ms_set ) {
			new_cst = new abinitio::MaxSeqSepConstraintSet( *constraints_, pose.fold_tree()  );
			new_cst->set_max_seq_sep( ms_set->max_seq_sep() );
		}
	}

	if ( new_cst ) {
		pose.constraint_set( new_cst );
	} else {
		pose.constraint_set( constraints_ );
	}

}

Real ConstraintEvaluator::apply( core::pose::Pose& pose_in ) const {
	pose::Pose pose( pose_in );
	prepare_pose( pose_in, pose );
	ScoreFunction scfxn;
	scfxn.set_weight( atom_pair_constraint, 1.0 );
	return scfxn( pose );
}

std::string ConstraintEvaluator::name( core::Size i ) const {
	if ( i == 1 ) { return name_ + "_cst"; }
	if ( i == 2 ) { return name_ + "_viol"; }
	runtime_assert( i <= 2 && i > 0 );
	return ""; //make compiler happy
}


void ConstraintEvaluator::apply( core::pose::Pose& pose_in, std::string, core::io::silent::SilentStruct &pss ) const {
	pose::Pose pose( pose_in );

	prepare_pose( pose_in, pose );

	ScoreFunction scfxn;
	scfxn.set_weight( atom_pair_constraint, 1.0 );

	pss.add_energy( name( 1 ), scfxn( pose ) );
	pss.add_energy( name( 2 ), pose.constraint_set()->show_violations( tr.Info, pose, viol_level_, threshold_ ) );
	//	return scfxn( pose );

}


// core::Real ConstraintEvaluator::apply( core::pose::Pose& pose_in  ) const {
// 	pose::Pose pose( pose_in );

// 	runtime_assert( constraints_ );
// 	pose.constraint_set( constraints_ );

// 	ScoreFunction scfxn;
// 	scfxn.set_weight( atom_pair_constraint, 1.0 );
// 	return scfxn( pose );

// }


}
}
