// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file PoseEvaluator
/// @brief PoseEvaluator
/// @detailed
///
///
/// @author Oliver Lange



// Unit Headers
#include <protocols/evaluation/RmsdEvaluator.hh>

// Package Headers

// Project Headers
#include <core/io/silent/SilentStruct.hh>
#include <core/pose/Pose.hh>

#include <core/options/util.hh>
#include <core/options/after_opts.hh>
// ObjexxFCL Headers

// Utility headers
#include <core/util/Tracer.hh>
#include <core/scoring/rms_util.hh>

// option key includes

#include <core/options/keys/in.OptionKeys.gen.hh>


// C++ headers

static core::util::Tracer tr("protocols.evalution.RMSD");

namespace protocols {
namespace evaluation {
using namespace core;

void invert_exclude_residues( Size nres, utility::vector1<int> const& exclude_list, ResidueSelection& residue_selection ) {

	residue_selection.clear();

	for( Size ir = 1; ir <= nres; ++ir ) {
		bool exclude_residue = false;
		for( Size ex = 1; ex <= exclude_list.size(); ex ++ ){
			if( int(exclude_list[ex]) == int(ir) ) {
				exclude_residue = true;
				break;
			}
		}

		if ( !exclude_residue ) {
			residue_selection.push_back( ir );
		}
	} // for ( Size ir = 1; ir <= native_pose.total_residue(); ++ir )
}


Real native_CA_rmsd( const core::pose::Pose & native_pose, const core::pose::Pose & pose ){
	using namespace core::options;
	using namespace core::options::OptionKeys;

	if ( option[ in::file::native_exclude_res ].user() ) {
		ResidueSelection residues;
		invert_exclude_residues( native_pose.total_residue(), option[ in::file::native_exclude_res ](), residues );
		return core::scoring::CA_rmsd( native_pose, pose, residues );
	} else {
		// no residues excluded from the native.
		return core::scoring::CA_rmsd( native_pose, pose );
	}
} // native_CA_rmsd




RmsdEvaluator::RmsdEvaluator( core::pose::PoseCOP pose, Size start, Size end, std::string tag, bool bGDT /*default true*/ )
  : SingleValuePoseEvaluator< Real > ("rms"+tag ),
		rmsd_pose_( pose ),
    start_( start ),
    end_( end ),
    bGDT_ ( bGDT ),
    tag_ ( tag ),
		report_gdt_components_ ( false )
{
  runtime_assert( start >=1 );
  runtime_assert( end <= pose -> total_residue() );
}

RmsdEvaluator::RmsdEvaluator( core::pose::PoseCOP pose, std::string tag, bool bGDT /* default true */ )
  : SingleValuePoseEvaluator< Real > ("rms"+tag),
		rmsd_pose_( pose ),
    start_( 1 ),
    end_( pose->total_residue() ),
    bGDT_ ( bGDT ),
    tag_( tag ),
		report_gdt_components_ ( false )
{}


void
RmsdEvaluator::apply( core::pose::Pose& pose, std::string tag, core::io::silent::SilentStruct &pss) const {
  static util::Tracer tr("protocols.Evaluator.RMSD");
	tr.Debug << "compute RMSD for " << tag << " for residues " << start_ << "..." << end_ << std::endl;

	core::Real rmsd( apply( pose ) );
  pss.add_energy( "rms"+tag_, rmsd );


  if ( bGDT_ ) { //move into GDT_Evaluator
    tr.Debug << "compute GDT-score for " << tag << std::endl;
    core::Real m_1_1, m_2_2, m_3_3, m_4_3, m_7_4;
    core::Real gdtmm = core::scoring::CA_gdtmm( *rmsd_pose_, pose, m_1_1, m_2_2, m_3_3, m_4_3, m_7_4 );
    pss.add_energy ( "gdtmm"+tag_, gdtmm );
		if ( report_gdt_components_ ) {
			pss.add_energy ( "m11", m_1_1 );
			pss.add_energy ( "m22", m_2_2 );
			pss.add_energy ( "m33", m_3_3 );
			pss.add_energy ( "m43", m_4_3 );
			pss.add_energy ( "m74", m_7_4 );
		}
    tr.Debug << "compute maxsub for " << tag << std::endl;
    int maxsub = core::scoring::CA_maxsub( *rmsd_pose_, pose );
    pss.add_energy( "maxsub", maxsub );
  }
}

Real RmsdEvaluator::apply( core::pose::Pose& pose ) const {
	runtime_assert( rmsd_pose_ );
	core::Real rmsd;
  if ( start_ == 1 && end_ == rmsd_pose_->total_residue() ) {
    runtime_assert( pose.total_residue() >= end_ );
    rmsd = core::scoring::CA_rmsd( *rmsd_pose_, pose );
  } else {
    runtime_assert( pose.total_residue() >= end_ );
    rmsd = core::scoring::CA_rmsd( *rmsd_pose_, pose, start_, end_ );
  }
	return rmsd;
}


SelectRmsdEvaluator::SelectRmsdEvaluator( core::pose::PoseCOP pose, std::list< Size > const& selection, std::string tag )
  : SingleValuePoseEvaluator< Real >( "rms"+tag ),
		rmsd_pose_( pose ),
		selection_( selection ),
    tag_ ( tag )
{

}

SelectRmsdEvaluator::SelectRmsdEvaluator( core::pose::PoseCOP pose, utility::vector1< Size> const& selection, std::string tag )
  : SingleValuePoseEvaluator< Real >( "rms"+tag ),
		rmsd_pose_( pose ),
		tag_( tag )
{
	copy( selection.begin(), selection.end(), std::back_inserter( selection_ ) );
}


SelectRmsdEvaluator::SelectRmsdEvaluator( core::pose::PoseCOP pose, std::string tag )
	: SingleValuePoseEvaluator< Real >( "rms"+tag ),
		rmsd_pose_( pose ),
		tag_( tag )
{
	find_existing_residues( pose, tag, selection_ );
}

SelectRmsdEvaluator::SelectRmsdEvaluator( core::pose::Pose const& pose, std::string tag )
	: SingleValuePoseEvaluator< Real >( "rms"+tag ),
		rmsd_pose_( new core::pose::Pose( pose ) ),
		tag_( tag )
{
	find_existing_residues( rmsd_pose_, tag, selection_ );
}

Real
SelectRmsdEvaluator::apply( core::pose::Pose& pose ) const {
  runtime_assert( rmsd_pose_ );
  core::Real rmsd;
	rmsd = core::scoring::CA_rmsd( *rmsd_pose_, pose, selection_ );
	return rmsd;
}


SelectGdtEvaluator::SelectGdtEvaluator( core::pose::PoseCOP pose, std::list< Size > const& selection, std::string tag )
  : SingleValuePoseEvaluator< Real >( "gdtmm"+tag ),
		rmsd_pose_( pose ),
		selection_( selection ),
    tag_ ( tag )
{

}

SelectGdtEvaluator::SelectGdtEvaluator( core::pose::PoseCOP pose, utility::vector1< Size> const& selection, std::string tag )
  : SingleValuePoseEvaluator< Real >( "gdtmm"+tag ),
		rmsd_pose_( pose ),
		tag_( tag )
{
	copy( selection.begin(), selection.end(), std::back_inserter( selection_ ) );
}

SelectGdtEvaluator::SelectGdtEvaluator( core::pose::PoseCOP pose, std::string tag )
	: SingleValuePoseEvaluator< Real >( "gdtmm"+tag ),
		rmsd_pose_( pose ),
		tag_( tag )
{
	find_existing_residues( pose, tag, selection_ );
}

SelectGdtEvaluator::SelectGdtEvaluator( core::pose::Pose const& pose, std::string tag )
	: SingleValuePoseEvaluator< Real >( "gdtmm"+tag ),
		rmsd_pose_( new core::pose::Pose( pose ) ),
		tag_( tag )
{
	find_existing_residues( rmsd_pose_, tag, selection_ );
}


Real
SelectGdtEvaluator::apply( core::pose::Pose& pose ) const {
  runtime_assert( rmsd_pose_ );
  core::Real m_1_1, m_2_2, m_3_3, m_4_3, m_7_4;
	core::Real gdtmm = core::scoring::CA_gdtmm( *rmsd_pose_, pose, selection_, m_1_1, m_2_2, m_3_3, m_4_3, m_7_4 );
	return gdtmm;
}


SelectMaxsubEvaluator::SelectMaxsubEvaluator( core::pose::PoseCOP pose, std::list< Size > const& selection, std::string tag )
  : SingleValuePoseEvaluator< Real >( "maxsub"+tag ),
		rmsd_pose_( pose ),
		selection_( selection ),
    tag_ ( tag )
{

}

SelectMaxsubEvaluator::SelectMaxsubEvaluator( core::pose::PoseCOP pose, utility::vector1< Size> const& selection, std::string tag )
  : SingleValuePoseEvaluator< Real >( "maxsub"+tag ),
		rmsd_pose_( pose ),
		tag_( tag )
{
	copy( selection.begin(), selection.end(), std::back_inserter( selection_ ) );
}

SelectMaxsubEvaluator::SelectMaxsubEvaluator( core::pose::PoseCOP pose, std::string tag )
	: SingleValuePoseEvaluator< Real >( "maxsub"+tag ),
		rmsd_pose_( pose ),
		tag_( tag )
{
	find_existing_residues( pose, tag, selection_ );
}

SelectMaxsubEvaluator::SelectMaxsubEvaluator( core::pose::Pose const& pose, std::string tag )
	: SingleValuePoseEvaluator< Real >( "maxsub"+tag ),
		rmsd_pose_( new core::pose::Pose( pose ) ),
		tag_( tag )
{
	find_existing_residues( rmsd_pose_, tag, selection_ );
}


Real
SelectMaxsubEvaluator::apply( core::pose::Pose& pose ) const {
  runtime_assert( rmsd_pose_ );
	//  core::Real m_1_1, m_2_2, m_3_3, m_4_3, m_7_4;
	core::Real maxsub = core::scoring::CA_maxsub( *rmsd_pose_, pose, selection_ );
	return maxsub;
}


SymmetricRmsdEvaluator::SymmetricRmsdEvaluator( core::pose::PoseCOP pose, std::string tag )
	: SingleValuePoseEvaluator< Real > ("symmetric rms"+tag ),
		rmsd_pose_( pose ) {}

Real
SymmetricRmsdEvaluator::apply( core::pose::Pose& pose ) const {

	return core::scoring::CA_rmsd_symmetric( *rmsd_pose_, pose );
}


}
}
