// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file relax_initialization_protocols
/// @brief initialization protocols for relax
/// @detailed
///	  Contains currently: Classic Abinitio
///
///
/// @author Oliver Lange


// Unit Headers
#include <protocols/evaluation/ScoreEvaluator.hh>

// Package Headers
#include <protocols/evaluation/PoseEvaluator.hh>

// Project Headers
#include <core/io/silent/SilentStruct.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/options/option.hh>
#include <iterator>

// Utility headers
#include <core/util/Tracer.hh>
#include <core/util/prof.hh>
static core::util::Tracer tr("protocols.evalution.Score");


// option key includes
// option key includes

#include <core/options/keys/constraints.OptionKeys.gen.hh>


// ObjexxFCL Headers

// Utility headers

//// C++ headers

namespace protocols {
namespace evaluation {

using namespace core;

ScoreEvaluator::ScoreEvaluator( std::string tag, core::scoring::ScoreFunctionOP scorefxn ) :
	SingleValuePoseEvaluator< Real >( "score"+tag ),
  scorefxn_ ( scorefxn->clone() )
{}

core::Real
ScoreEvaluator::apply(
  core::pose::Pose& pose
) const {
	return ( *scorefxn_ )( pose );
}

TruncatedScoreEvaluator::TruncatedScoreEvaluator(
	 std::string tag,
   ResidueSelectionVector const& selection,
 	 core::scoring::ScoreFunctionOP scorefxn
) :
	SingleValuePoseEvaluator< Real >( "score"+tag ),
	scorefxn_( scorefxn ),
	selection_( selection )
{
	nres_ = 200; //no worries: if the pose turns out to be larger we repeat that inversion in apply().
	invert_include_residues( 200, selection_, exclude_list_ );
}

Real TruncatedScoreEvaluator::apply( core::pose::Pose& pose ) const {
	PROF_START( core::util::TRUNCATED_SCORE_EVALUATOR );
	scoring::ScoreFunctionOP scorefxn( scorefxn_ );
	if ( !scorefxn ) {
		if ( pose.is_fullatom() ) {
			scorefxn = core::scoring::getScoreFunction();
		} else {
			scorefxn = core::scoring::ScoreFunctionFactory::create_score_function( "score3" );
		}
		scorefxn->set_weight( scoring::linear_chainbreak, 1.0 );
		scorefxn->set_weight( scoring::overlap_chainbreak, 1.0 );
		if ( pose.constraint_set()->has_residue_pair_constraints()  ) {
			scorefxn->set_weight( scoring::atom_pair_constraint, options::option[ options::OptionKeys::constraints::cst_weight ]() );
			scorefxn->set_weight( scoring::angle_constraint, options::option[ options::OptionKeys::constraints::cst_weight ]() );
			scorefxn->set_weight( scoring::dihedral_constraint, options::option[ options::OptionKeys::constraints::cst_weight ]() );
		}
	}
	if ( pose.total_residue() != nres_ ) {
		nres_ = pose.total_residue();
		invert_include_residues( nres_, selection_, exclude_list_ );
	}
	tr.Debug << "compute score without using these residues: ";
	std::copy( exclude_list_.begin(), exclude_list_.end(), std::ostream_iterator< Size >( tr.Debug, " " ));
	tr.Debug << std::endl;
	core::Real val = scorefxn->get_sub_score_exclude_res( pose, exclude_list_ );
	PROF_STOP( core::util::TRUNCATED_SCORE_EVALUATOR );
	return val;
}


}
}
