// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file relax_initialization_protocols
/// @brief initialization protocols for relax
/// @detailed
///	  Contains currently: Classic Abinitio
///
///
/// @author Oliver Lange


// Unit Headers

// Package Headers
#include <protocols/evaluation/PoseEvaluator.fwd.hh>
#include <protocols/evaluation/PoseEvaluator.hh>
#include <protocols/evaluation/RmsdEvaluator.hh>
#include <protocols/evaluation/RDC_Evaluator.hh>
#include <protocols/evaluation/ScoreEvaluator.hh>
#include <protocols/evaluation/TimeEvaluator.hh>
#include <protocols/evaluation/JumpEvaluator.hh>
#include <protocols/evaluation/PoseMetricEvaluator.hh>
#include <protocols/evaluation/ConstraintEvaluator.hh>
#include <protocols/evaluation/ChemicalShiftEvaluator.hh>
#include <protocols/evaluation/BatchNrEvaluator.hh>
#include <protocols/loops/LoopClass.hh>
#include <core/scoring/constraints/util.hh>

// Project Headers
#include <core/chemical/ResidueType.hh>
#include <core/conformation/Residue.hh>

#include <core/io/silent/silent.fwd.hh>
#include <core/pose/Pose.hh>
#include <core/io/pdb/pose_io.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>

// Utility headers
#include <utility/pointer/ReferenceCount.hh>
#include <utility/vector1.hh>

#include <utility/file/FileName.hh>

#include <core/options/option.hh>
#include <core/options/after_opts.hh>
#include <core/util/Tracer.hh>
#include <core/io/silent/SilentStructFactory.hh>
//// C++ headers

// due to template function
#include <core/io/silent/SilentStruct.hh>


// option key includes

#include <core/options/keys/evaluation.OptionKeys.gen.hh>
#include <core/options/keys/abinitio.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>


static core::util::Tracer tr("protocols.evalution");

namespace protocols {
namespace evaluation {
using namespace core;

void invert_include_residues( Size nres, ResidueSelectionVector const& include_list, ResidueSelectionVector& exclude_list ) {

	exclude_list.clear();

	for ( Size ir = 1; ir <= nres; ++ir ) {
		bool include_residue = false;
		for ( Size ex = 1; ex <= include_list.size(); ex ++ ) {
			if ( include_list[ex] == ir ) {
				include_residue = true;
				break;
			}
		}

		if ( !include_residue ) {
			exclude_list.push_back( ir );
		}
	} // for ( Size ir = 1; ir <= native_pose.total_residue(); ++ir )
}


void read_common_evaluator_options( MetaPoseEvaluator& eval ) {

	using namespace core;
	using namespace core::options;
	using namespace OptionKeys;
  if ( option[ OptionKeys::evaluation::rmsd_target ].user() ) {
    /*
      this creates Evaluators to get RMSD and GDTMM for as many structures as you specify in this FileVector option.
      pls: provide also as many column-names to match your rmsd-targets.
      ---
      missing density in the input file will be ignored, e.g., this is your way to signal on which residues the rmsd should be computed
    */
    utility::vector1< std::string > const& rmsd_target( option[ OptionKeys::evaluation::rmsd_target ]() );
    utility::vector1< std::string > const& rmsd_col_name( option[ OptionKeys::evaluation::rmsd_column]() );
    for ( Size ct = 1; ct <= rmsd_target.size(); ct ++ ) {
      pose::PoseOP rmsd_pose = new pose::Pose;
      io::pdb::pose_from_pdb( *rmsd_pose, rmsd_target[ ct ] );
      std::string tag( ObjexxFCL::string_of( ct ) );
      if ( rmsd_col_name.size() >= ct ) tag = rmsd_col_name[ ct ];
      eval.add_evaluation( new evaluation::SelectRmsdEvaluator( rmsd_pose, tag ) );
      if ( option[ OptionKeys::evaluation::gdtmm ]() ) eval.add_evaluation( new evaluation::SelectGdtEvaluator( rmsd_pose, tag ) );
			if ( option[ OptionKeys::evaluation::score_with_rmsd ]() ){
				ResidueSelection selection;
				find_existing_residues( rmsd_pose, tag, selection );
				ResidueSelectionVector vector;
				std::copy( selection.begin(), selection.end(), std::back_inserter( vector ) );
				eval.add_evaluation( new evaluation::TruncatedScoreEvaluator( tag, vector ) );
			}
			if ( option[ OptionKeys::evaluation::symmetric_rmsd ].user() ) {
				eval.add_evaluation( new evaluation::SymmetricRmsdEvaluator( rmsd_pose, tag ) );
			}
    }
  }
	if ( option[ OptionKeys::evaluation::rmsd_select ].user() ) {
    utility::vector1< utility::file::FileName > const& rmsd_core( option[ OptionKeys::evaluation::rmsd_select ]() );
		if ( !option[ in::file::native ].user() ) utility_exit_with_message( "need to specify in:file:native together with rmsd_select " );

		core::pose::PoseOP native_pose = new core::pose::Pose;
		io::pdb::pose_from_pdb( *native_pose, option[ in::file::native ]() );

	  for ( Size ct = 1; ct <= rmsd_core.size(); ct ++ ) {
			loops::Loops core;
			core.read_loop_file( rmsd_core[ ct ], false, "RIGID" );
			utility::vector1< Size> selection;
			core.get_residues( selection );
			if ( native_pose ) eval.add_evaluation( new evaluation::SelectRmsdEvaluator( native_pose, selection, rmsd_core[ ct ].base() ) );
			if ( option[ OptionKeys::evaluation::score_with_rmsd ]() ) {
				eval.add_evaluation( new evaluation::TruncatedScoreEvaluator( rmsd_core[ ct ].base(), selection ) );
			}
		}
	}

  if ( option[ OptionKeys::evaluation::rdc_target ].user() ) {
    /*
      this creates Evaluators to get RMSD and GDTMM for as many structures as you specify in this FileVector option.
      pls: provide also as many column-names to match your rmsd-targets.
      ---
      missing density in the input file will be ignored, e.g., this is your way to signal on which residues the rmsd should be computed
    */
    utility::vector1< std::string > const& rdc_target( option[ OptionKeys::evaluation::rdc_target ]() );
    utility::vector1< std::string > const& rdc_col_name( option[ OptionKeys::evaluation::rdc_column ]() );
    for ( Size ct = 1; ct <= rdc_target.size(); ct ++ ) {
      pose::PoseOP rdc_pose = new pose::Pose;
      io::pdb::pose_from_pdb( *rdc_pose, rdc_target[ ct ] );
      std::string tag( ObjexxFCL::string_of( ct ) );
      if ( rdc_col_name.size() >= ct ) tag = rdc_col_name[ ct ];
      eval.add_evaluation( new evaluation::SelectRDC_Evaluator( rdc_pose, tag ) );
    }
  }

	if ( option[ OptionKeys::evaluation::rdc_select ].user() ) {
    utility::vector1< utility::file::FileName > const& rdc_core( option[ OptionKeys::evaluation::rdc_select ]() );

	  for ( Size ct = 1; ct <= rdc_core.size(); ct ++ ) {
			loops::Loops core;
			core.read_loop_file( rdc_core[ ct ], false, "RIGID" );
			utility::vector1< Size> selection;
			core.get_residues( selection );
			eval.add_evaluation( new evaluation::SelectRDC_Evaluator( selection, rdc_core[ ct ].base() ) );
		}
	}


  if ( option[ OptionKeys::evaluation::constraints ].user() ) {
    /*
      this creates Evaluators to evaluate different constraint sets against your decoys
      pls: provide also as many column-names to match your constraint sets
      ---
    */
    utility::vector1< std::string > const& cst_target( option[ OptionKeys::evaluation::constraints ]() );
    utility::vector1< std::string > const& cst_col_name( option[ OptionKeys::evaluation::constraints_column ]() );
    for ( Size ct = 1; ct <= cst_target.size(); ct ++ ) {
      std::string tag( ObjexxFCL::string_of( ct ) );
      if ( cst_col_name.size() >= ct ) tag = cst_col_name[ ct ];
      eval.add_evaluation( new evaluation::ConstraintEvaluator( tag, cst_target[ ct ] ) );
    }
  }

	// set rmsd native
	if ( option[ in::file::native ].user() ) {
		core::pose::PoseOP native_pose = new core::pose::Pose;
		io::pdb::pose_from_pdb( *native_pose, option[ in::file::native ]() );

		if ( option[ in::file::native_exclude_res ].user() ) {
			eval.add_evaluation( new SelectRmsdEvaluator(
						native_pose,
						invert_exclude_residues( native_pose->total_residue(), option[ in::file::native_exclude_res ]()),
						"" )
			);
			if ( option[ OptionKeys::evaluation::gdtmm ]() ) {
				eval.add_evaluation( new SelectGdtEvaluator(
						native_pose,
						invert_exclude_residues( native_pose->total_residue(), option[ in::file::native_exclude_res ]()),
						"" )
			);
			}
		} else if ( option[ OptionKeys::abinitio::rmsd_residues ].user() ){
			core::Size start = option[ OptionKeys::abinitio::rmsd_residues ]()[ 1 ];
			Size end = option[ OptionKeys::abinitio::rmsd_residues ]()[ 2 ];
			eval.add_evaluation( new RmsdEvaluator( native_pose, start, end,  "", option[ OptionKeys::abinitio::bGDT ]() ) );
		} else {
			eval.add_evaluation( new SelectRmsdEvaluator( native_pose, "" ) );
			if ( option[ OptionKeys::evaluation::gdtmm ]() ) eval.add_evaluation( new SelectGdtEvaluator( native_pose, "" ) );
      eval.add_evaluation( new SelectMaxsubEvaluator( native_pose, "" ) );
		}
	} // if ( native_pose_ )

	if ( option[ OptionKeys::evaluation::chemical_shifts ].user() ) {
		eval.add_evaluation( new ChemicalShiftEvaluator( "cs_score", option[ OptionKeys::evaluation::chemical_shifts ]() ) );
	}

	if ( option[ OptionKeys::evaluation::extra_score ].user() ) {
		using namespace core::scoring;
		utility::vector1< std::string > const& extra_scores( option[ OptionKeys::evaluation::extra_score ]() );
    utility::vector1< std::string > const& extra_score_names( option[ OptionKeys::evaluation::extra_score_column]() );
		if ( extra_scores.size() != extra_score_names.size() ) {
			utility_exit_with_message("-extra_score: you need to provide as much extra_score_names as extra_scores! ");
		}
		for ( Size ct = 1; ct <= extra_scores.size(); ct ++ ) {
			std::string const& tag = extra_score_names[ ct ];
			std::string patch( "NOPATCH" );
			if ( option[ OptionKeys::evaluation::extra_score_patch ].user() ) {
				if ( option[ OptionKeys::evaluation::extra_score_patch ]().size() != extra_scores.size() ) {
					utility_exit_with_message("-extra_score: you need to provide as much extra_score_patch(es) as \
                    extra_scores! use NOPATCH as placeholder");
				}
				patch = option[ OptionKeys::evaluation::extra_score_patch ]()[ ct ];
			}
			ScoreFunctionOP scfxn( NULL );
			if ( patch != "NOPATCH" ) {
				scfxn = ScoreFunctionFactory::create_score_function( extra_scores[ ct ], patch );
			} else {
				scfxn = ScoreFunctionFactory::create_score_function( extra_scores[ ct ] );
			}
			std::string name( extra_scores[ ct ] );
			if ( (name == "score0") ||
				(name == "score2") ||
				(name == "score3") ||
				(name == "score5") ) {
				core::scoring::constraints::add_constraints_from_cmdline_to_scorefxn( *scfxn );
			} else {
				core::scoring::constraints::add_fa_constraints_from_cmdline_to_scorefxn( *scfxn );
			}
			eval.add_evaluation( new ScoreEvaluator( tag, scfxn ) );
			eval.add_evaluation( new ScoreEvaluator( tag, scfxn ) );
		}
	}


	if ( option[ OptionKeys::evaluation::jump_nr ]() ) {
		eval.add_evaluation( new JumpNrEvaluator );
	}
  //return eval;
} //read_common_options


//@detail find residues that don't have missing density
void find_existing_residues(  core::pose::PoseCOP pose, std::string tag, ResidueSelection& selection ) {
	for ( Size pos = 1; pos <= pose->total_residue(); pos++ ) {
		if ( pose->residue_type( pos ).is_protein() && pose->residue_type( pos ).has("CA") ) {
			numeric::xyzVector< core::Real> ca_pos = pose->residue( pos ).atom("CA").xyz();
			bool good ( true );
			for ( Size j=1; j<= pose->residue( pos ).natoms(); ++j ) {
				if ( ( ca_pos - pose->residue( pos ).atom(j).xyz() ).length() > 20 ) {
					good = false;
				}
			}
			if ( good ) selection.push_back( pos );
		}
	}
	if ( tr.Trace.visible() ) {
		tr.Trace << "selection of residues for rmsd of " << tag << std::endl;
		for ( std::list< core::Size >::const_iterator it = selection.begin(), eit = selection.end();
					it != eit; ++it ) {
			tr.Trace << " " << *it;
		}
		tr.Trace << std::endl;
	}
}

void evaluate_pose( core::pose::Pose& pose, PoseEvaluator& eval, std::ostream& os ) {
		//		ProteinSilentStruct pss;
		io::silent::SilentStructOP pss = io::silent::SilentStructFactory::get_silent_struct_out();
		pss->fill_struct( pose, "eval" );
		eval.apply( pose, "eval", *pss );
		os << "\n";
		pss->print_score_header( os );
		os << "\n";
		pss->print_scores( os );
		os << std::endl;
}

}
}
