// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 sw=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file devel/ProteinInterfaceDesign/filters/DisulfideFilter.hh
/// @brief Filters for interfaces which could form a disulfide bond between
/// docking partners.
/// @author Sarel Fleishman (sarelf@uw.edu)

#include <protocols/filters/AtomicContactFilter.hh>


// Project Headers
#include <core/types.hh>
#include <core/pose/Pose.hh>

//parsing
#include <utility/Tag/Tag.hh>
#include <core/conformation/Residue.hh>
#include <protocols/moves/DataMap.hh>
#include <protocols/filters/Filter.hh>
#include <protocols/moves/Mover.fwd.hh> //Movers_map
#include <protocols/ProteinInterfaceDesign/util.hh>

namespace protocols {
namespace filters {

static core::util::Tracer TR( "protocols.filters.AtomicContactFilter" );

///@brief default ctor
AtomicContactFilter::AtomicContactFilter() :
	parent( "AtomicContact" )
{}

///@brief Constructor with a single target residue
AtomicContactFilter::AtomicContactFilter( core::Size const res1, core::Size const res2, core::Real const distance, bool const sidechain, bool const backbone, bool const protons ) :
	parent( "AtomicContactFilter" ),
	residue1_( res1 ),
	residue2_( res2 ),
	distance_( distance ),
	sidechain_( sidechain ),
	backbone_( backbone ),
	protons_( protons )
{}

/// @return Whether a disulfide bond is possible between any of the targets
bool AtomicContactFilter::apply(core::pose::Pose const & pose ) const
{
	core::Real const dist( compute( pose ) );
	if( dist <= distance_ ) return true;
	return false;
}

core::Real
AtomicContactFilter::compute( core::pose::Pose const & pose ) const
{
	using namespace core::conformation;

	core::Real nearest_distance( 10000 );
	Residue const res1( pose.residue( residue1_ ) ), res2( pose.residue( residue2_ ) );

	Atoms::const_iterator atom1_begin( res1.atom_begin() ), atom1_end( res1.atom_end() ), atom2_begin( res2.atom_begin() ), atom2_end( res2.atom_end() );
	if( sidechain_ && !backbone_ ){
		atom1_begin = res1.sidechainAtoms_begin();
		atom2_begin = res2.sidechainAtoms_begin();
	}
	if( !sidechain_ && backbone_ ){
		atom1_end = res1.sidechainAtoms_begin();
		atom2_end = res2.sidechainAtoms_begin();
	}
	if( !protons_ ){
		atom1_end = res1.heavyAtoms_end();
		atom2_end = res2.heavyAtoms_end();
	}
	for( Atoms::const_iterator atom1=atom1_begin; atom1!=atom1_end; ++atom1 ){
		for( Atoms::const_iterator atom2=atom2_begin; atom2!=atom2_end; ++atom2 ){
			core::Real const dist( atom1->xyz().distance( atom2->xyz() ) );
			if( dist <= nearest_distance ) nearest_distance = dist;
		}//foreach atom2
	}//foreach atom1
	return( nearest_distance );
}

core::Real
AtomicContactFilter::report_sm( core::pose::Pose const & pose ) const
{
	core::Real const dist( compute( pose ) );
	return( dist );
}

void AtomicContactFilter::report( std::ostream & out, core::pose::Pose const & pose ) const
{
	core::Real const dist( compute( pose ) );
	out<<"Minimal distance between residues "<<residue1_<<" and "<<residue2_<<" is "<<dist<<std::endl;
}

void AtomicContactFilter::parse_my_tag( utility::Tag::TagPtr const tag,
		protocols::moves::DataMap &,
		protocols::filters::Filters_map const &,
		protocols::moves::Movers_map const &,
		core::pose::Pose const & pose)
{
	using namespace protocols::ProteinInterfaceDesign;

	distance_ = tag->getOption< core::Real >( "distance", 4.0 );
	std::string const res1( tag->getOption< std::string >( "residue1" ) );
	std::string const res2( tag->getOption< std::string >( "residue2" ) );
	residue1_ = parse_resnum( res1, pose );
	residue2_ = parse_resnum( res2, pose );
	sidechain_ = tag->getOption< bool >( "sidechain", 1 );
	backbone_  = tag->getOption< bool >( "backbone",  0 );
	protons_   = tag->getOption< bool >( "protons",   0 );

	TR<<"AtomicContact filter between residues "<<residue1_<<" and "<<residue2_<<" with distance cutoff of "<<distance_<<std::endl;
}

} // filters
} // protocols
