// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/filters/Filter.cc
/// @brief
/// @detailed
///	  Contains currently:
///
///
/// @author Florian Richter, Sarel Fleishman (sarelf@uw.edu)

// Unit Headers
#include <protocols/filters/Filter.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/moves/DataMap.hh>
#include <utility/Tag/Tag.hh>
#include <core/util/Tracer.hh>

// Package Headers

// Project Headers
//#include <core/pose/Pose.hh>
//#include <core/types.hh>

// ObjexxFCL Headers

// Utility headers
#include <core/util/Tracer.hh>
#include <numeric/random/random.hh>

//// C++ headers
static core::util::Tracer TR("protocols.filters.Filter");
static numeric::random::RandomGenerator RG( 140789 ); // <- Magic number, do not change it!!!

namespace protocols {
namespace filters {

using namespace core;
typedef std::pair< std::string const, FilterCOP > StringFilter_pair;
typedef utility::Tag::TagPtr TagPtr;
typedef core::pose::Pose Pose;

bool
FilterCollection::apply( core::pose::Pose const & pose ) const
{

	for( utility::vector1< protocols::filters::FilterCOP >::const_iterator filter_it = filters_.begin();
			 filter_it != filters_.end(); ++filter_it ){

		if( ! (*filter_it)->apply( pose ) ){
			return false;
		}
	}

	return true;
}

Filter::Filter()
	: utility::pointer::ReferenceCount(),
		type_( "UNDEFINED TYPE" )
{}

Filter::Filter( std::string const & type )
	: utility::pointer::ReferenceCount(),
		type_( type )
{}

Filter::Filter( Filter const & init )
	:	utility::pointer::ReferenceCount(),
		type_( init.type_ ),
		user_defined_name_( init.user_defined_name_ )
{}

Filter::~Filter() {}

void
Filter::parse_my_tag(
	TagPtr const,
	moves::DataMap &,
	Filters_map const &,
	moves::Movers_map const &,
	Pose const & )
{}


////////////////////////////////////////////////////////////////////////////////////////////////////

StochasticFilter::StochasticFilter() : Filter( "Stochastic" ) {}
StochasticFilter::~StochasticFilter() {}

StochasticFilter::StochasticFilter( core::Real const confidence )
	: Filter( "Stochastic" ), confidence_( confidence )
{}

bool
StochasticFilter::apply( Pose const & ) const
{
	if( confidence_ >= 0.999 ) return true;

	core::Real const random_number( RG.uniform() );
	if( random_number <= confidence_ ) {
		TR<<"stochastic filter returning false"<<std::endl;
		return false;
	}
	TR<<"stochastic filter returning true"<<std::endl;
	return true;
}


FilterOP
StochasticFilter::clone() const
{
	return new StochasticFilter( *this );
}

FilterOP
StochasticFilter::fresh_instance() const
{
	return new StochasticFilter();
}

void
StochasticFilter::parse_my_tag(
	TagPtr const tag,
	moves::DataMap &,
	Filters_map const &,
	moves::Movers_map const &,
	Pose const & )
{
	confidence_ = tag->getOption< core::Real >( "confidence", 1.0 );
	TR<<"stochastic filter with confidence "<<confidence_<<std::endl;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// @brief Used to define a compound logical statement involving other filters with
// AND, OR and XOR
CompoundFilter::CompoundFilter() : Filter( "CompoundStatement" ) {}
CompoundFilter::~CompoundFilter() {}

CompoundFilter::CompoundFilter( CompoundStatement const & compound_statement ) :
	Filter( "CompoundStatement" ),
	compound_statement_( compound_statement )
{}

bool
CompoundFilter::apply( Pose const & pose ) const
{
	bool const value( compute( pose ) );

	TR<<"Compound logical statement is "<<value<<"."<<std::endl;
	return( value );
}

FilterOP
CompoundFilter::clone() const
{
	return new CompoundFilter( *this );
}

FilterOP
CompoundFilter::fresh_instance() const
{
	return new CompoundFilter();
}

void
CompoundFilter::report( std::ostream & out, Pose const & pose ) const
{
	if( compound_statement_.size() == 2 ){
	//special case for filters that are defined with a confidence value. In that case, we want to report the value of the filter regardless of the stochastic filter
		bool confidence( false );
		CompoundStatement::const_iterator non_stochastic_filter;
		for( CompoundStatement::const_iterator it=compound_statement_.begin(); it!=compound_statement_.end(); ++it ){
			if( it->first->get_type() == "Stochastic" ) confidence = true;
			else non_stochastic_filter = it;
		}
		if( confidence ) non_stochastic_filter->first->report( out, pose );
	}
	bool const value( compute( pose ) );

	out<<"Compound filter returns: "<<value<<'\n';
}

core::Real
CompoundFilter::report_sm( Pose const & pose ) const
{
	if( compound_statement_.size() == 2 ){
	//special case for filters that are defined with a confidence value. In that case, we want to report the value of the filter regardless of the stochastic filter
		bool confidence( false );
		CompoundStatement::const_iterator non_stochastic_filter;
		for( CompoundStatement::const_iterator it=compound_statement_.begin(); it!=compound_statement_.end(); ++it ){
			if( it->first->get_type() == "Stochastic" ) confidence = true;
			else non_stochastic_filter = it;
		}
		if( confidence ) return( non_stochastic_filter->first->report_sm( pose ) );
	}
	bool const value( compute( pose ) );
	return( value );
}

bool
CompoundFilter::compute( Pose const & pose ) const
{
	bool value( true );

	for( CompoundStatement::const_iterator it=compound_statement_.begin(); it!=compound_statement_.end(); ++it ) {
		if( it - compound_statement_.begin() == 0 ){ // ignore first logical operation
			value = it->first->apply( pose );
			continue;
		}
		switch( it->second  ) {
			case ( AND ) : value = value && it->first->apply( pose ); break;
			case ( OR  ) : value = value || it->first->apply( pose ); break;
			case ( XOR ) : value = value ^ it->first->apply( pose ); break;
			case ( NOR ) : value = !( value || it->first->apply( pose ) ); break;
			case (NAND ) : value = !( value && it->first->apply( pose ) ); break;
		}
	}
	return( value );
}

void
CompoundFilter::clear()
{
	compound_statement_.clear();
}

void
CompoundFilter::parse_my_tag(
	TagPtr const tag,
	moves::DataMap &,
	Filters_map const & filters,
	moves::Movers_map const &,
	Pose const & )
{
	TR<<"CompoundStatement"<<std::endl;
	std::vector< TagPtr > const compound_tags( tag->getTags() );
	for( std::vector< TagPtr >::const_iterator cmp_it=compound_tags.begin(); cmp_it!=compound_tags.end(); ++cmp_it ) {
		TagPtr const cmp_tag_ptr = *cmp_it;
		std::string const operation( cmp_tag_ptr->getName() );
		std::pair< FilterCOP, boolean_operations > filter_pair;
		if( operation == "AND" ) filter_pair.second = AND;
		else if( operation == "OR" ) filter_pair.second = OR;
		else if( operation == "XOR" ) filter_pair.second = XOR;
		else if( operation == "NOR" ) filter_pair.second = NOR;
		else if( operation =="NAND" ) filter_pair.second = NAND;
		else {
			TR<<"***WARNING WARNING: Boolean operation in tag is undefined. Skipping:\n"<<tag<<std::endl;
			runtime_assert( 0 );
		}
		std::string const filter_name( cmp_tag_ptr->getOption<std::string>( "filter_name" ) );

		std::map< std::string const, FilterCOP >::const_iterator find_filter( filters.find( filter_name ));
		bool const filter_found( find_filter!=filters.end() );
		if( filter_found )
			filter_pair.first = find_filter->second->clone();
		else {
			TR<<"***WARNING WARNING! Filter defined for CompoundStatement not found in filter_list!!!! Defaulting to truefilter***"<<std::endl;
			filter_pair.first = new filters::TrueFilter;
		}
		runtime_assert( filter_found );
		compound_statement_.push_back( filter_pair );
	}
}

} // filters
} // protocols
