// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/protocols/filters/Filter.hh
/// @brief header file for Filter base class
/// @detailed
///
///
///
/// @author Florian Richter, floric@u.washington.edu (feb 09 ), Sarel Fleishman sarelf@u.washington.edu

#ifndef INCLUDED_protocols_filters_Filter_HH
#define INCLUDED_protocols_filters_Filter_HH

// Unit Headers
#include <protocols/filters/Filter.fwd.hh>
#include <protocols/moves/Mover.fwd.hh>

// Project Headers
#include <core/pose/Pose.fwd.hh>

// Utility headers
#include <utility/pointer/ReferenceCount.hh>
#include <utility/vector1.hh>
#include <utility/Tag/Tag.fwd.hh>
#include <protocols/moves/DataMap.fwd.hh>
#include <core/types.hh>


//// C++ headers
#include <string>

namespace protocols {
namespace filters {

class Filter : public utility::pointer::ReferenceCount {
public:
	Filter();
	Filter( std::string const  & );
	Filter( Filter const & );
	virtual ~Filter();
  // allows reporting of filter values to a stream
	virtual void report( std::ostream &, core::pose::Pose const & ) const {}
/// @brief used to report filter internals through a score or silent file
  // to determine that derived class has not overridden }
	virtual core::Real report_sm( core::pose::Pose const & ) const { return -99999; }

	virtual std::string get_type() const{ return type_; }
	std::string get_user_defined_name() const { return user_defined_name_; }
	void set_user_defined_name( std::string const & name ) { user_defined_name_ = name; };

	/// @brief used to clear internal variables if needed. Using fresh_instance is preferred since it's a pure virtual
	virtual void clear() {};
	virtual void parse_my_tag(
		utility::Tag::TagPtr const,
		moves::DataMap &,
		Filters_map const &,
		moves::Movers_map const &,
		core::pose::Pose const & );
	virtual FilterOP clone() const = 0;

	virtual FilterOP fresh_instance() const = 0;

	/// @brief Returns true if the given pose passes the filter, false otherwise.
	virtual bool apply( core::pose::Pose const & pose ) const = 0;

	virtual
	std::string name() const { return "BaseFilter"; };

private:
	std::string const type_;
	std::string user_defined_name_;
};


/// @brief Wrapper-class that contains a vector1 of Filters
/// @brief apply function returns true if all member filters return true
class FilterCollection : public utility::pointer::ReferenceCount {

public:

	/// @brief Returns true if the given pose passes all filters, false otherwise.
	bool apply( core::pose::Pose const & pose ) const;

	void
	add_filter( FilterCOP filter_in ){
		filters_.push_back( filter_in ); }

	void
	clear(){
		filters_.clear(); }

private:

	utility::vector1< FilterCOP > filters_;

};

class TrueFilter : public Filter {
public:
	TrueFilter() : Filter( "TrueFilter" ) {}
	bool apply( core::pose::Pose const & ) const { return true; }
	FilterOP clone() const { return new TrueFilter; }
	FilterOP fresh_instance() const { return new TrueFilter; }
};

class FalseFilter : public Filter {
public:
	FalseFilter() : Filter( "FalseFilter" ) {}
	bool apply( core::pose::Pose const & ) const { return false; }
	FilterOP clone() const { return new FalseFilter; }
	FilterOP fresh_instance() const { return new FalseFilter; }
};

class StochasticFilter : public Filter {

public:
	StochasticFilter();
	virtual ~StochasticFilter();
	StochasticFilter( core::Real const confidence );
	bool apply( core::pose::Pose const & pose ) const;
	FilterOP clone() const;
	FilterOP fresh_instance() const;
	void report( std::ostream &, core::pose::Pose const & ) const {}

	void parse_my_tag(
		utility::Tag::TagPtr const tag,
		moves::DataMap &,
		Filters_map const &,
		moves::Movers_map const &,
		core::pose::Pose const & );

private:
	core::Real confidence_;
};

// @brief Used to define a compound logical statement involving other filters with
// AND, OR and XOR
class CompoundFilter : public Filter {

public:
	typedef std::vector< std::pair< FilterCOP, boolean_operations > > CompoundStatement;

public:
	CompoundFilter();
	virtual ~CompoundFilter();
	CompoundFilter( CompoundStatement const & );
	bool apply( core::pose::Pose const & ) const;
	FilterOP clone() const;
	FilterOP fresh_instance() const;
	void report( std::ostream &, core::pose::Pose const & ) const;
	core::Real report_sm( core::pose::Pose const & ) const;
	bool compute( core::pose::Pose const & ) const;
	void clear();

	void parse_my_tag(
		utility::Tag::TagPtr const,
		moves::DataMap &,
		Filters_map const &,
		moves::Movers_map const &,
		core::pose::Pose const & );

private:
	CompoundStatement compound_statement_;
};



} // filters
} // protocols

#endif
