// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/filters/ScoreCutoffFilter.cc
/// @brief
/// @detailed
///	  Contains currently:
///
///
/// @author Florian Richter

// Unit Headers
#include <protocols/filters/ScoreCutoffFilter.hh>

// Package Headers

// Project Headers
#include <core/pose/Pose.hh>
#include <core/types.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/Energies.hh>

// ObjexxFCL Headers

// Utility headers
#include <core/util/Tracer.hh>


//// C++ headers
static core::util::Tracer tr("protocols.filters.ScoreCutoffFilter");

namespace protocols {
namespace filters {


ScoreCutoffFilter::ScoreCutoffFilter() :
	cutoff_( 0.0 ), total_score_( true )
{
	score_types_.push_back( core::scoring::total_score );
	positions_.clear();
}

ScoreCutoffFilter::ScoreCutoffFilter( core::Real cutoff_in ) :
	cutoff_(cutoff_in), total_score_( true )
{
	score_types_.push_back( core::scoring::total_score );
	positions_.clear();
}


void
ScoreCutoffFilter::set_score_type( core::scoring::ScoreType scotype )
{
	score_types_.clear();
	add_score_type( scotype );
}

void
ScoreCutoffFilter::add_score_type( core::scoring::ScoreType scotype )
{
	using namespace core::scoring;

	if( scotype == total_score ){

		total_score_ = true;

		if( (score_types_.size() > 1) || ( score_types_[1] != total_score ) ){

			score_types_.clear();
			tr << "WARNING: setting score_type to total score even though other score types have previously been set. these will be erased." << std::endl;
		}

	}
	else{

		if( total_score_ == true ){

			score_types_.clear();
			total_score_ = false;
		}
	}

	score_types_.push_back( scotype );
}


bool
ScoreCutoffFilter::apply( core::pose::Pose const & pose ) const {

	core::Real cur_score = get_score( pose );

	if ( cur_score <= cutoff() ) return true;

	return false;

} // apply_filter


core::Real
ScoreCutoffFilter::get_score( core::pose::Pose const & pose ) const {

	using namespace core::scoring;
	EnergyMap emap;

	assert( score_types_.size() > 0 );

	if ( positions_.size() == 0 ) { //means we're interested in pose totals

		for ( utility::vector1< ScoreType >::const_iterator sco_it = score_types_.begin();
				 sco_it != score_types_.end(); ++sco_it ){

			emap[ *sco_it ] = pose.energies().total_energies()[ *sco_it ];
		}
	}

	else {
		for ( utility::vector1< core::Size >::const_iterator pos_it = positions_.begin(); pos_it != positions_.end(); ++pos_it ) {

			for ( utility::vector1< ScoreType >::const_iterator sco_it = score_types_.begin();
						 sco_it != score_types_.end(); ++sco_it ){
				emap[ *sco_it ] += pose.energies().residue_total_energies( *pos_it )[ *sco_it ];
			}
		}
	} // else

	//now accumulate into score
	if( total_score_ ) return emap[ total_score ];

	return emap.dot( pose.energies().weights() );

}

} // filters
} // protocols
