// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Nobuyasu Koga

#include <protocols/flxbb/BluePrint.hh>

//
#include <core/types.hh>

#include <core/options/util.hh>
#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;
static core::util::Tracer TR("protocols.flxbb.BluePrint");

// Utility headers
#include <utility/io/izstream.hh>
#include <utility/string_util.hh>
#include <boost/lexical_cast.hpp>

using namespace core;
using namespace protocols::flxbb;

BluePrint::BluePrint( std::string const & filename ){
  if ( read_blueprint( filename ) ) {
    identify_ss();
    set_respairing();
  }else{
    TR.Error << "Error in reading blueprint file " << filename << std::endl;
    exit(0);
  }
}


bool
BluePrint::read_blueprint( std::string const & filename ){

  utility::io::izstream data( filename );
  if ( !data ) {
    TR.Error << "can not open blueprint file " << filename << std::endl;
    return false;
  }

  beta_pairings_.clear();
  std::string line;
  Size linecount(0);
  while( getline( data, line ) ) {
    linecount++;
    std::vector< std::string > tokens ( utility::split( line ) );

    if( tokens[0][0] != '#' ){
      if( tokens[0] == "FOLDINFO" ){

	if( tokens.size() > 2 ){
	  TR.Error << "error parsing at " << linecount << " in " << filename << std::endl;
	  return false;
	}

	std::vector< std::string > bpairs( utility::string_split( tokens[1], '.' ) );

	for( std::vector< std::string >::const_iterator iter = bpairs.begin(); iter != bpairs.end() ; ++iter) {

	  std::string pair = *iter;
	  std::vector< std::string > parts( utility::string_split( pair, ':' ) );

	  if( parts.size() != 3 ){
	    TR.Error << "error parsing: " << pair << " at " << linecount << " in " << filename << std::endl;
	    return false;
	  }

	  std::vector< std::string > st( utility::string_split( parts[0], '-' ) );
	  Size st1 = boost::lexical_cast<Size>( st[0] );
	  Size st2 = boost::lexical_cast<Size>( st[1] );

	  Real rsf = boost::lexical_cast<Real>( parts[1] );

	  bool para;
	  if( parts[2][0] == 'P' ){
	    para = true;
	  }else if( parts[2][0] == 'A' ){
	    para = false;
	  }else{
	    TR.Error << "error parsing: " << pair << " at " << linecount << " in " << filename << std::endl;
	    return false;
	  }

	  beta_pairings_.push_back( new BetaPair( st1, st2, rsf, para ) );

	}

      }else{

	if( tokens.size() == 3 ){

	  Size ii = boost::lexical_cast<Size>( tokens[0] );

	  assert( tokens[1].size() == 1 && tokens[2].size() == 1);
	  char aa ( tokens[1][0] );
	  char sec( tokens[2][0] );

	  //std::cout << aa << ' ' << sec << std::endl;

	  if ( sec != 'E' && sec != 'H' && sec != 'L' && sec != 'I'  && sec != '.' ) {
	    TR.Error << "unrecognized secstruct char : " << sec << " at lines " << linecount << " in " << filename << std::endl;
	    return false;
	  }

	  resnum_.push_back( ii );
	  resname_.push_back( aa );
	  sstype_.push_back( sec );

	}

      }
    }
  } //#

  assert( resname_.size() == sstype_.size() );
  total_residue_ = sstype_.size();

  assert( total_residue_ > 0 );

  for( utility::vector1< char >::const_iterator iter = sstype_.begin(); iter != sstype_.end() ; ++iter) {
    secstruct_ += *iter;
  }

  for( utility::vector1< char >::const_iterator iter = resname_.begin(); iter != resname_.end() ; ++iter) {
    sequence_ += *iter;
  }

  TR << secstruct_ << std::endl;
  TR << sequence_ << std::endl;

  helices_.resize( int( total_residue_ ) );
  strands_.resize( int( total_residue_ ) );

  return true;

}

void
BluePrint::identify_ss()
{

  bool flag_E( false );
  bool flag_H( false );
  int istrand( 0 );
  int ihelix( 0 );

  for ( int i=1; i<= int( total_residue_ ); ++i ) {
    char ss( sstype_[ i ] );

    if( ss =='E' && flag_E == false ){
      istrand ++;
      strands_.SS_strand_end( 1, istrand ) = i;
      flag_E = true;
    }

    if( ss !='E' && flag_E == true ){
      strands_.SS_strand_end( 2, istrand ) = i - 1;
      flag_E = false;
    }

    if( ss =='H' && flag_H == false ){
      ihelix ++;
      helices_.HH_helix_end( 1, ihelix ) = i;
      flag_H = true;
    }

    if( ss !='H' && flag_H == true ){
      helices_.HH_helix_end( 2, ihelix ) = i - 1;
      flag_H = false;
    }

  }

  if( flag_H == true ){
    helices_.HH_helix_end( 2, ihelix ) = int(total_residue_);
  }

  if( flag_E == true ){
    strands_.SS_strand_end( 2, istrand ) = int(total_residue_);
  }

  strands_.total_strands = istrand;
  helices_.total_helices = ihelix;

}

void
BluePrint::set_respairing(){

  for( utility::vector1< BetaPairOP >::const_iterator
	 iter = beta_pairings_.begin(); iter != beta_pairings_.end() ; ++iter) {

    BetaPairOP bpair = *iter;

    int ist = int( bpair->strandi() );
    int jst = int( bpair->strandj() );

    Real rsf = bpair->rshift();
    bool para = bpair->parallel();

    Real ir_ist = strands_.SS_strand_end( 1, ist );
    Real er_ist = strands_.SS_strand_end( 2, ist );
    Real len_ist = er_ist - ir_ist + 1;

    Real ir_jst = strands_.SS_strand_end( 1, jst );
    Real er_jst = strands_.SS_strand_end( 2, jst );
    Real len_jst = er_jst - ir_jst + 1;

    Real start_ist, start_jst;
    Real len,inc;

    /*
    std::cout << "ist-jst:" << ist << ' ' << jst << ' ' << std::endl;
    std::cout << "ires-eres:" << ir_ist << ' ' << er_ist << ' ' << std::endl;
    std::cout << "ires-eres:" << ir_jst << ' ' << er_jst << ' ' << std::endl;
    */

    if( para == true ){
      if( rsf >= 0 ){

	start_ist = ir_ist + rsf;
	start_jst = ir_jst;

	if( len_ist >= (len_jst + rsf) ){
	  len = len_jst;
	}else{
	  len = len_ist - rsf;
	}

      }else{

	start_ist = ir_ist;
	start_jst = ir_jst - rsf;

	if( len_ist >= (len_jst + rsf) ){
	  len = len_jst - rsf;
	}else{
	  len = len_ist ;
	}

      }

      inc = 1;

    }else{

      if( rsf >= 0 ){

	start_ist = ir_ist;
	start_jst = er_jst - rsf;

	if( len_ist >= (len_jst - rsf) ){
	  len = len_jst - rsf;
	}else{
	  len = len_ist;
	}

      }else{

	start_ist = ir_ist - rsf;
	start_jst = er_jst;

	if( len_ist >= (len_jst - rsf) ){
	  len = len_jst ;
	}else{
	  len = len_ist + rsf;
	}

      }

      inc = -1;

    }

    bpair->pair_res_.resize(2);
    for( Size i=1; i<=len ;i++ ){
      //std::cout << start_ist << ' ' << start_jst << ' ' << std::endl;

      bpair->pair_res_[0].push_back( Size(start_ist) );
      bpair->pair_res_[1].push_back( Size(start_jst) );

      start_ist ++;
      start_jst += inc;
    }

  }

}
