// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Nobuyasu Koga
///

#ifndef INCLUDED_protocols_flxbb_DesignTask_HH
#define INCLUDED_protocols_flxbb_DesignTask_HH

#include <protocols/flxbb/DesignLayerOperation.hh>
#include <protocols/flxbb/FilterStructs.hh>

#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunction.hh>

#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/task/TaskFactory.hh>

#include <protocols/moves/Mover.hh>

#include <utility/pointer/ReferenceCount.hh>
#include <core/options/option.hh>
#include <core/options/util.hh>

// option key includes
#include <core/options/keys/packing.OptionKeys.gen.hh>
using namespace core::options;
using namespace core::options::OptionKeys;

using namespace core;
using namespace protocols::moves;

namespace protocols {
namespace flxbb{


	///////////////////////////////////////////////////////////////////////////////////////////////////////
  class DesignTask : public utility::pointer::ReferenceCount {
  public:
    DesignTask(): ncycle_( 1 ){};
    DesignTask(
							 Size & ncycle,
							 scoring::ScoreFunctionOP & sfxn,
							 MoverOP const & mover=0,
							 FilterStructsOP const & filter_structs=0,
							 pack::task::PackerTaskOP const & taskf=0
								):
      ncycle_( ncycle ),
			scorefxn_( sfxn ),
			mover_( mover ),
			filter_structs_( filter_structs ),
			task_( taskf )
    {}

		virtual void setup( pose::Pose &, pack::task::PackerTaskOP & ) = 0;

    FilterStructsOP filter_structs() const
		{
			return filter_structs_;
		}

    core::Size ncycle() const
		{
			return ncycle_;
		}

    MoverOP mover() const
		{
			return mover_;
		}

    core::scoring::ScoreFunctionOP scorefxn() const
		{
			return scorefxn_;
		}

    core::pack::task::PackerTaskOP packertask() const
		{
			return task_;
		}

		void
		dump_packertask( std::ostream & os ){
			os << *task_;
		}

    void set_mover( MoverOP & value ){
			mover_  = value;
		}

    void set_filter_structs( FilterStructsOP const & value ){
			filter_structs_ = value;
		}

    void set_ncycle( core::Size const & ncycle ){
			ncycle_  = ncycle;
		}

    void set_scorefxn( core::scoring::ScoreFunctionOP const & sfxn ){
			scorefxn_  = sfxn;
		}

    void set_packertask( core::pack::task::PackerTaskOP const & taskf ){
			task_ = taskf;
		}

  private:

		Size ncycle_;
    scoring::ScoreFunctionOP scorefxn_;
		MoverOP mover_;
		FilterStructsOP filter_structs_;
    pack::task::PackerTaskOP task_;

	};

  typedef utility::pointer::owning_ptr< DesignTask > DesignTaskOP;
  typedef utility::vector1< DesignTaskOP > DesignTaskSet;

	///////////////////////////////////////////////////////////////////////////////////////////////////////
	class DesignTask_Layer: public DesignTask {
	public:
		DesignTask_Layer();

		DesignTask_Layer( bool dsgn_core, bool dsgn_boundary, bool dsgn_surface,
											bool use_original_seq,
											Size ncycle,
											scoring::ScoreFunctionOP sfxn,
											bool exala,
											bool exmet,
											MoverOP mover=0,
											FilterStructsOP filter_structs=0
											): DesignTask( ncycle, sfxn, mover, filter_structs ),
												 dsgn_core_(dsgn_core), dsgn_boundary_(dsgn_boundary), dsgn_surface_(dsgn_surface),
												 use_original_seq_( use_original_seq ), exclude_ala_( exala ), exclude_met_( exmet ){}


		void setup( pose::Pose & pose, pack::task::PackerTaskOP & task ){

			using namespace core::pack;
			operation::InitializeFromCommandlineOP cmop = new operation::InitializeFromCommandline;
			operation::ReadResfileOP rrop = new operation::ReadResfile;

			cmop->apply( pose, *task );

			DesignLayerOperationOP op = new DesignLayerOperation( dsgn_core_, dsgn_boundary_, dsgn_surface_ );
			if( exclude_ala_ ){
				op->set_exclude_ala();
			}
			if( exclude_met_ ){
				op->set_exclude_met();
			}
			if( use_original_seq_ ){
				op->use_original_seq();
			}
			op->apply( pose, *task );

			if ( option[ packing::resfile ].user() ) {
				rrop->apply( pose, *task );
			}

			this->set_packertask( task );

		}

	private:
		bool dsgn_core_, dsgn_boundary_, dsgn_surface_;
		bool use_original_seq_;
		bool exclude_ala_;
		bool exclude_met_;

	};
  typedef utility::pointer::owning_ptr< DesignTask_Layer > DesignTask_LayerOP;
	///////////////////////////////////////////////////////////////////////////////////////////////////////
	class DesignTask_Normal: public DesignTask {
	public:
		DesignTask_Normal();
		DesignTask_Normal( Size ncycle,
											 scoring::ScoreFunctionOP sfxn,
											 MoverOP mover=0,
											 FilterStructsOP filter_structs=0
											 ): DesignTask( ncycle, sfxn, mover, filter_structs ){}

		void setup( pose::Pose & pose, pack::task::PackerTaskOP & task ){

			using namespace core::pack;
			operation::InitializeFromCommandlineOP cmop = new operation::InitializeFromCommandline;
			operation::ReadResfileOP rrop = new operation::ReadResfile;

			cmop->apply( pose, *task );

			if ( option[ packing::resfile ].user() ) {
				rrop->apply( pose, *task );
			}

			this->set_packertask( task );
		}

	};
  typedef utility::pointer::owning_ptr< DesignTask_Normal > DesignTask_NormalOP;

	///////////////////////////////////////////////////////////////////////////////////////////////////////
	class DesignTask_AllVal: public DesignTask {
	public:
		DesignTask_AllVal();
		DesignTask_AllVal( Size ncycle,
											 scoring::ScoreFunctionOP sfxn,
											 MoverOP mover=0,
											 FilterStructsOP filter_structs=0
											 ): DesignTask( ncycle, sfxn, mover, filter_structs ){}

		void setup( pose::Pose & pose, pack::task::PackerTaskOP & task ){

			utility::vector1<bool> restrict_to_aa( 20, false );
			restrict_to_aa[chemical::aa_from_name( "VAL" )] = true;
			for( Size i=1; i<=pose.total_residue(); ++i ){
				task->nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );
			}

			this->set_packertask( task );
		}

	};
  typedef utility::pointer::owning_ptr< DesignTask_AllVal > DesignTask_AllValOP;


}
}

#endif
