// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Nobuyasu Koga

// Unit headers
#include <protocols/flxbb/FlxbbDesign.hh>
#include <protocols/flxbb/DesignTask.hh>
#include <protocols/flxbb/DesignLayerOperation.hh>
#include <protocols/flxbb/FilterStructs.hh>
#include <protocols/flxbb/utility.hh>
#include <protocols/flxbb/BluePrint.hh>

// Project headers
#include <core/pose/Pose.hh>
#include <core/pose/util.hh>
#include <protocols/moves/Mover.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/scoring/packstat/compute_sasa.hh>

#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>

#include <protocols/relax_protocols.hh>

#include <protocols/jobdist/standard_mains.hh>
#include <protocols/viewer/viewers.hh>

#include <core/options/option.hh>
#include <core/options/util.hh>
#include <core/init.hh>
#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.flxbb.FlxbbDesign");

// option key includes
#include <core/options/keys/flxbb.OptionKeys.gen.hh>
#include <core/options/keys/relax.OptionKeys.gen.hh>

using namespace core;
using namespace protocols::flxbb;
using namespace core::scoring::packstat;

using namespace core::options;
using namespace core::options::OptionKeys;
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
FlxbbDesignPack::FlxbbDesignPack()
	: PackRotamersMover()
{}

FlxbbDesignPack::~FlxbbDesignPack(){}

void
FlxbbDesignPack::apply( pose::Pose & pose ){

	pose.update_residue_neighbors();
	this->setup( pose );

	if( filter_ == 0 ){

		this->run( pose );

	}else{

		while( filter_->filter_on() ){

			this->run( pose );
			filter_->apply( pose );

		}
		pose = filter_->get_bestpose();
	}

}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
FlxbbDesign::~FlxbbDesign(){}

void
FlxbbDesign::apply( pose::Pose & pose )
{

	using namespace core::pack;
	using namespace core::scoring::packstat;
	using namespace protocols::moves;

#ifdef GL_GRAPHICS
	protocols::viewer::add_conformation_viewer( pose.conformation(), "FlxbbDesign" );
#endif

	//
	design_taskset_.clear();

	// set constraints in beta-sheet
	if( option[ OptionKeys::flxbb::constraints_sheet ] ){
		if( blueprint_ != 0 ){

			if( option[ OptionKeys::flxbb::constraints_sheet_ca ] ){
				constraints_sheet_ca( pose, scorefxn_relax_, blueprint_ );
			}else{
				constraints_sheet( pose, scorefxn_relax_, blueprint_ );
			}

		}else{
			constraints_sheet( pose, scorefxn_relax_ );
		}
	}

	// constraints between N and C
	if( option[ OptionKeys::flxbb::constraints_NtoC ] ){
		constraints_NtoC( pose, scorefxn_relax_ );
	}

	// setup design_taskset
	if( design_taskset_.empty() ){

		if( option[ OptionKeys::flxbb::layer::layer ]() == "fatty" ){

			make_design_taskset_fatty( design_taskset_, pose, scorefxn_design_, scorefxn_relax_, blueprint_ );

		}else if( option[ OptionKeys::flxbb::layer::layer ]() == "normal" ){

			make_design_taskset_normal( design_taskset_, pose, scorefxn_design_, scorefxn_relax_ );

		}else{

			MoverOP mover;
			if ( option[ OptionKeys::relax::fast]() ) {
				mover = new protocols::relax::SimpleMultiRelax( scorefxn_relax_ );
			} else{
				mover = new protocols::relax::ClassicRelax( scorefxn_relax_ );
			}

			design_taskset_.push_back( new DesignTask_Normal( nflxbb_cycle_, scorefxn_design_, mover ) );

		}

	}

	// run
	Size num_task( 0 );
	for ( DesignTaskSet::iterator it= design_taskset_.begin(), ite= design_taskset_.end(); it != ite; ++it ) {

		num_task ++;
		DesignTaskOP design_task( *it );
		for( Size i=1 ; i<=design_task->ncycle() ; i++ ){

			TR << "cycle/total_cycle: " << i << "/" << design_task->ncycle() << " in DesignTask: " << num_task << std::endl;

			task::PackerTaskOP task( task::TaskFactory::create_packer_task( pose ));
			design_task->setup( pose, task );
			design_task->dump_packertask( TR );

			FlxbbDesignPack pack( design_task->scorefxn(), design_task->packertask(), design_task->filter_structs() );
			pack.apply( pose );
			TR << "# score after designing by fixbb " << std::endl;
			//TR << pose.sequence() << std::endl;
			scorefxn_design_->show( TR, pose );
			TR.flush();

			if( design_task->mover() != 0 ){

				design_task->mover()->apply( pose );

				TR << "# score after mover " << std::endl;
				scorefxn_relax_->show( TR, pose );
				TR.flush();

			}

			//pose.dump_pdb( "tmp.pdb" ) ;

		}

	}

	core::Real packscore;
	packscore = compute_packing_score( pose );
	setPoseExtraScores( pose, "pack_stat", packscore );

}

