// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Nobuyasu Koga

#ifndef INCLUDED_protocols_flxbb_utility_HH
#define INCLUDED_protocols_flxbb_utility_HH

#include <protocols/flxbb/DesignTask.hh>
#include <protocols/flxbb/BluePrint.hh>

#include <core/pose/Pose.hh>
#include <core/scoring/SS_Info.hh>
#include <protocols/jumping/Dssp.hh>


#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/AtomPairConstraint.hh>
#include <core/scoring/constraints/BoundConstraint.hh>

#include <protocols/relax_protocols.hh>
#include <core/kinematics/MoveMap.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/ScoringManager.fwd.hh>

/// Numeric headers
#include <numeric/xyzVector.hh>

// Utility headers
#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR_flxbb_utility("protocols.flxbb.FlxbbDesign.utility");

using namespace core;
using namespace core::scoring;
using namespace core::scoring::constraints;
using namespace protocols::flxbb;

namespace protocols {
namespace flxbb {

////////////////////////////////////////////////////////////////////////////////////////////////
void
constraints_sheet( pose::Pose & pose, scoring::ScoreFunctionOP & scorefxn, BluePrintOP & blueprint_ ){

	// parameters for constraint
	Real condist( 3.5 );

	std::string tag( "constraints_in_beta_sheet" );
	Real lb( 0.0 );
	Real ub( condist );
	Real sd( 0.5 );
	Real coef( 50.0 );
	scorefxn->set_weight( atom_pair_constraint, coef );

	//
  Size nres( pose.total_residue() );
	if( nres != blueprint_->total_residue() ){
		TR_flxbb_utility.Error  << "# length of aa is different between pdb and blueprint: "
														<< nres << ' ' << blueprint_->total_residue() << std::endl;
		exit(0);
	}

	BetaPairings bpairs( blueprint_->beta_pairings() );
	for( BetaPairings::iterator it=bpairs.begin(), ite=bpairs.end(); it != ite; ++it ){
		BetaPairOP bop( *it );

		std::vector< std::vector< Size > > pair ( bop->pair_res() );

		for( Size i=0 ; i < pair[0].size(); ++i){

			Size iaa = pair[0][i];
			Size jaa = pair[1][i];

			conformation::Residue const & ires( pose.residue( iaa ) );
			conformation::Residue const & jres( pose.residue( jaa ) );

			std::cout << iaa << '-' << jaa << std::endl;

			Size iN = ires.atom_index( "N" );
			Size iCa= ires.atom_index( "CA" );
			Size iC = ires.atom_index( "C" );
			Size iO = ires.atom_index( "O" );

			Size jN = jres.atom_index( "N" );
			Size jCa= jres.atom_index( "CA" );
			Size jO = jres.atom_index( "O" );

			Vector vCaCa( jres.xyz( jCa ) - ires.xyz( iCa ) );
			Vector vCO( ires.xyz( iO ) - ires.xyz( iC ) );

			Vector uCaCa( vCaCa.normalized_or_zero() );
			Vector uCO( vCO.normalized_or_zero() );

			Real dp = uCaCa.dot( uCO );

			if( dp >= 0 ){

				if( jaa + 1 <= nres ) {

					id::AtomID atom1( pose.residue_type( iaa ).atom_index( "N" ), iaa );
					id::AtomID atom2( pose.residue_type( jaa-1 ).atom_index( "O" ), jaa-1 );
					pose.add_constraint( new AtomPairConstraint( atom1, atom2, new BoundFunc( lb, ub, sd, tag )) );

					id::AtomID atom3( pose.residue_type( iaa ).atom_index( "O" ), iaa );
					id::AtomID atom4( pose.residue_type( jaa+1 ).atom_index( "N" ), jaa+1 );
					pose.add_constraint( new AtomPairConstraint( atom3, atom4, new BoundFunc( lb, ub, sd, tag )) );

					conformation::Residue const & kres( pose.residue( jaa - 1 ) );
					conformation::Residue const & lres( pose.residue( jaa + 1 ) );
					Size kN = kres.atom_index( "O" );
					Size lO = lres.atom_index( "N" );

					Real const dsq1( distance_squared( ires.xyz( iO ), kres.xyz( kN ) ));
					Real const dsq2( distance_squared( ires.xyz( iN ), lres.xyz( lO ) ));

					std::cout << sqrt( dsq1 )  << ' ' << sqrt( dsq2 ) << std::endl;

				}

			}else{

				if( iaa - 1 >= 1) {

					id::AtomID atom1( pose.residue_type( iaa-1 ).atom_index( "O" ), iaa-1 );
					id::AtomID atom2( pose.residue_type( jaa ).atom_index( "N" ), jaa );
					pose.add_constraint( new AtomPairConstraint( atom1, atom2, new BoundFunc( lb, ub, sd, tag )) );

					id::AtomID atom3( pose.residue_type( iaa+1 ).atom_index( "N" ), iaa+1 );
					id::AtomID atom4( pose.residue_type( jaa ).atom_index( "O" ), jaa );
					pose.add_constraint( new AtomPairConstraint( atom3, atom4, new BoundFunc( lb, ub, sd, tag )) );

					conformation::Residue const & kres( pose.residue( iaa - 1 ) );
					conformation::Residue const & lres( pose.residue( iaa + 1 ) );
					Size kO = kres.atom_index( "O" );
					Size lN = lres.atom_index( "N" );

					Real const dsq1( distance_squared( jres.xyz( jN ), kres.xyz( kO ) ));
					Real const dsq2( distance_squared( jres.xyz( jO ), lres.xyz( lN ) ));

					std::cout << sqrt( dsq1 )  << ' ' << sqrt( dsq2 ) << std::endl;

				}

			}

		}

	}

}

////////////////////////////////////////////////////////////////////////////////////////////////
void
constraints_sheet_ca( pose::Pose & pose, scoring::ScoreFunctionOP & scorefxn, BluePrintOP & blueprint_ ){

	// parameters for constraint
	Real condist( 5.5 );

	std::string tag( "constraints_in_beta_sheet" );
	Real lb( 0.0 );
	Real ub( condist );
	Real sd( 0.5 );
	Real coef( 1000.0 );
	scorefxn->set_weight( atom_pair_constraint, coef );

	//
  Size nres( pose.total_residue() );
	if( nres != blueprint_->total_residue() ){
		TR_flxbb_utility.Error  << "# length of aa is different between pdb and blueprint: "
														<< nres << ' ' << blueprint_->total_residue() << std::endl;
		exit(0);
	}

	TR_flxbb_utility << "# Constrains between CA-CA atoms in sheet are applied for the following residues " << std::endl;
	BetaPairings bpairs( blueprint_->beta_pairings() );
	for( BetaPairings::iterator it=bpairs.begin(), ite=bpairs.end(); it != ite; ++it ){
		BetaPairOP bop( *it );

		std::vector< std::vector< Size > > pair ( bop->pair_res() );

		for( Size i=0 ; i < pair[0].size(); ++i){

			Size iaa = pair[0][i];
			Size jaa = pair[1][i];
			TR_flxbb_utility << iaa << ' ' << jaa << std::endl;

			id::AtomID atom1( pose.residue_type( iaa ).atom_index( "CA" ), iaa );
			id::AtomID atom2( pose.residue_type( jaa ).atom_index( "CA" ), jaa );
			pose.add_constraint( new AtomPairConstraint( atom1, atom2, new BoundFunc( lb, ub, sd, tag )) );

		}
	}

}

void
constraints_NtoC( pose::Pose & pose, scoring::ScoreFunctionOP & scorefxn ){

	// parameters for constraint
	Real condist( 11.0 );

	std::string tag( "constraints_in_beta_sheet" );
	Real lb( 0.0 );
	Real ub( condist );
	Real sd( 0.5 );
	Real coef( 10.0 );
	scorefxn->set_weight( atom_pair_constraint, coef );

	//
  Size nres( pose.total_residue() );

	id::AtomID atom1( pose.residue_type( 1 ).atom_index( "CA" ), 1 );
	id::AtomID atom2( pose.residue_type( nres ).atom_index( "CA" ), nres );
	pose.add_constraint( new AtomPairConstraint( atom1, atom2, new BoundFunc( lb, ub, sd, tag )) );

}

void
constraints_sheet( pose::Pose & pose, scoring::ScoreFunctionOP & scorefxn ){

	Real condist( 5.5 );
	Real condist2 = condist*condist;

	// parameters for constraint
	std::string tag( "constraints_in_beta_sheet" );
	Real lb( 0.0 );
	Real ub( condist );
	Real sd( 1.0 );
	Real coef( 1000.0 );
	scorefxn->set_weight( atom_pair_constraint, coef );
	//
  Size nres( pose.total_residue() );


	protocols::jumping::Dssp dssp( pose );
  dssp.insert_ss_into_pose( pose );

  // set strands
	bool flag( false );
  Size istrand ( 0 );
  Strands strands( nres );


  for ( Size i=1; i<= nres; ++i ) {
    char ss( pose.secstruct( i ) );

		//std::cout << i << ' ' << ss << std::endl;
    if( ss =='E' && flag == false ){
      istrand ++;
      strands.SS_strand_end( 1, istrand ) = i;
      flag = true;
    }

    if( ss !='E' && flag == true ){
      strands.SS_strand_end( 2, istrand ) = i - 1;
			flag = false;
    }

  }
  strands.total_strands = istrand;

	TR_flxbb_utility << "# Constrains between CA-CA atoms in sheet are applied for the following residues " << std::endl;

  for( int i=1; i<=strands.total_strands-1; ++i ){
    for( int j=i+1; j<=strands.total_strands; ++j ){

      for( int iresid=strands.SS_strand_end( 1, i ); iresid<=strands.SS_strand_end( 2, i ); iresid++){
				for( int jresid=strands.SS_strand_end( 1, j ); jresid<=strands.SS_strand_end( 2, j ); jresid++){

					conformation::Residue const & ires( pose.residue( iresid ) );
					conformation::Residue const & jres( pose.residue( jresid ) );

					Size ica = ires.atom_index( "CA" );
					Size jca = jres.atom_index( "CA" );

					Real const dsq( distance_squared( ires.xyz( ica ), jres.xyz( jca ) ));

					if( dsq<=condist2 ){

						TR_flxbb_utility << iresid << ' ' << jresid << std::endl;

						id::AtomID atom1( pose.residue_type( iresid ).atom_index( "CA" ), iresid );
						id::AtomID atom2( pose.residue_type( jresid ).atom_index( "CA" ), jresid );
						pose.add_constraint( new AtomPairConstraint( atom1, atom2, new BoundFunc( lb, ub, sd, tag )) );

					}
				}// jresid

      } // iresid

    } // j

  } // i

}
////////////////////////////////////////////////////////////////////////////////////////////////
void
restrict_bbmove( pose::Pose & pose, kinematics::MoveMapOP & movemap, BluePrintOP const & blueprint_ ){

	if( blueprint_ == 0 ){
		protocols::jumping::Dssp dssp( pose );
		dssp.insert_ss_into_pose( pose );
	}else{
		for ( Size i = 1; i <= pose.total_residue(); ++i ) {
			pose.set_secstruct( i, blueprint_->secstruct( i ) );
		}
	}

	bool flag_E( false );
	bool flag_H( false );
  for ( Size i=1; i<=pose.total_residue() ; ++i ) {
    char ss( pose.secstruct( i ) );

		std::cout << i << ' ' << ss << std::endl;

    if( ss =='E' ){

			movemap->set_bb( i, false );
			movemap->set_chi( i, true );

			if( flag_E == false && i != 1 ){
				//				movemap->set_bb( i-1, false );
				//				movemap->set_chi( i-1, true );
			}

			flag_E = true;

		}else if( ss == 'H' ){

			//movemap->set_bb( i, false );
			//movemap->set_chi( i, true );
			//flag_H = true;

		}else{

			flag_H = false;
			flag_E = false;
			movemap->set_bb( i, true );
			movemap->set_bb( i, true );

		}
	}

}

////////////////////////////////////////////////////////////////////////////////////////////////
void

make_design_taskset_normal( DesignTaskSet & design_taskset_,
														pose::Pose & pose,
														scoring::ScoreFunctionOP const & sfxn_dsgn,
														scoring::ScoreFunctionOP const & sfxn_rlx )
{

	scoring::ScoreFunctionOP sfxn = sfxn_rlx->clone();
	////////////////////////////////////////////////////////////////////////////////////////////////////////
	// 1st stage
	/*
	kinematics::MoveMapOP movemap = new core::kinematics::MoveMap();
	restrict_bbmove( pose, movemap, blueprint_ );

	protocols::ClassicRelaxOP myrelax = new protocols::ClassicRelax( sfxn, movemap );
	myrelax->get_checkpoints().set_disabled( true );
	myrelax->set_lj_ramp_cycles( 5 );
	myrelax->set_stage2_cycles( 0 );
	myrelax->set_stage3_cycles( 0 );

	//design_taskset_.push_back( new DesignTask_AllVal( 1, sfxn_dsgn, myrelax ) );


	design_taskset_.push_back( new DesignTask_Layer( true, false, false, false, 2, sfxn_dsgn,
																									 true, true,
																									 myrelax,
																									 new FilterStructs_Packstat( pose, 20 ) )); */

	////////////////////////////////////////////////////////////////////////////////////////////////////////
	// 2nd stage

	design_taskset_.push_back( new DesignTask_Layer( true, true, false, false, 2, sfxn_dsgn,
																									 false, false,
																									 new protocols::relax::SimpleMultiRelax( sfxn_rlx ),
																									 new FilterStructs_Packstat( pose, 20 ) ));

	////////////////////////////////////////////////////////////////////////////////////////////////////////
	// 2nd stage

	design_taskset_.push_back( new DesignTask_Layer( true, true, true, false, 2, sfxn_dsgn,
																									 false, false,
																									 new protocols::relax::SimpleMultiRelax( sfxn_rlx ),
																									 new FilterStructs_Packstat( pose, 20 ) ));

	////////////////////////////////////////////////////////////////////////////////////////////////////////
	// 3rd stage

	design_taskset_.push_back( new DesignTask_Layer( false, false, true, true, 1, sfxn_dsgn,
																									 false, false,
																									 0,
																									 new FilterStructs_TotalCharge( pose ) ));

}

void
make_design_taskset_fatty( DesignTaskSet & design_taskset_,
													 pose::Pose & pose,
													 scoring::ScoreFunctionOP const & sfxn_dsgn,
													 scoring::ScoreFunctionOP const & sfxn_rlx,
													 BluePrintOP const & blueprint_ )
{


	////////////////////////////////////////////////////////////////////////////////////////////////////////
	// 1st stage

	kinematics::MoveMapOP movemap = new core::kinematics::MoveMap();
	restrict_bbmove( pose, movemap, blueprint_ );

	scoring::ScoreFunctionOP sfxn = sfxn_rlx->clone();
	//core::Real original_weight = sfxn->get_weight( scoring::fa_rep );

	// sfxn->set_weight    ( scoring::fa_rep , original_weight * 0.08 );

	//                                               core, boundary, surface, ncycle, scorefxn,
	//                                               exala, exmet,
	//                                               mover,
	//                                               filter
	design_taskset_.push_back( new DesignTask_Layer( true, false, false, false, 1, sfxn_dsgn,
																									 true, true,
																									 new protocols::relax::ClassicRelax( sfxn, movemap ),
																									 new FilterStructs_Packstat( pose, 10 ) ));

	////////////////////////////////////////////////////////////////////////////////////////////////////////
	// 2nd stage

	//sfxn->set_weight    ( scoring::fa_rep , original_weight );
	design_taskset_.push_back( new DesignTask_Layer( true, true, false, false, 1, sfxn_dsgn,
																									 false, false,
																									 new protocols::relax::ClassicRelax( sfxn, movemap ),
																									 new FilterStructs_Packstat( pose, 20 ) ));

	/////////////////////////////////////////////////////////////////////////////////////////////////////////
	// 3rd stage

	design_taskset_.push_back( new DesignTask_Layer( true, true, true, false, 2, sfxn_dsgn,
																									 false, false,
																									 new protocols::relax::SimpleMultiRelax( sfxn_rlx ),
																									 new FilterStructs_Packstat( pose, 20 ) ));

	///////////////////////////////////////////////////////
	// 4th stage

	design_taskset_.push_back( new DesignTask_Layer( false, false, true, true, 1, sfxn_dsgn,
																									 false, false,
																									 0,
																									 new FilterStructs_TotalCharge( pose ) ));


}

/*
	void
	make_design_taskset_test( DesignTaskSet & design_taskset_,
	pose::Pose & pose,
	scoring::ScoreFunctionOP const & sfxn_dsgn,
	scoring::ScoreFunctionOP const & sfxn_rlx,
	BluePrintOP const & blueprint_	)
	{

	kinematics::MoveMapOP movemap = new core::kinematics::MoveMap();
	restrict_bbmove( pose, movemap, blueprint_ );

	scoring::ScoreFunctionOP sfxn = sfxn_rlx->clone();
	protocols::ClassicRelaxOP myrelax = new protocols::ClassicRelax( sfxn, movemap );

	//Size phase1_cycles_   = core::options::option[ core::options::OptionKeys::relax::fastrelax_rampcycles];

	myrelax->get_checkpoints().set_disabled( true );
	myrelax->set_lj_ramp_cycles( 5 );
	myrelax->set_stage2_cycles( 0 );
	myrelax->set_stage3_cycles( 0 );

	design_taskset_.push_back( new DesignTask_AllVal( 1, sfxn_dsgn, myrelax ) );

	design_taskset_.push_back( new DesignTask_Layer( true, false, false, false, 2, sfxn_dsgn,
	false, false,
	new protocols::SimpleMultiRelax( sfxn_rlx ),
	new FilterStructs_Packstat( pose, 20 ) ));

	}
*/


}
}

#endif
