// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/forge/methods/util.cc
/// @brief  miscellaneous utility functions for forge
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <protocols/forge/methods/util.hh>

// package headers
#include <protocols/forge/build/Interval.hh>
#include <protocols/forge/methods/fold_tree_functions.hh>

// project headers
#include <core/chemical/AA.hh>
#include <core/conformation/Conformation.hh>
#include <core/graph/DisjointSets.hh>
#include <core/id/types.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/pose/Pose.hh>
#include <core/util/Tracer.hh>

// numeric headers
#include <numeric/random/random.hh>

// C++ headers
#include <set>


namespace protocols {
namespace forge {
namespace methods {


// static
static core::util::Tracer TR( "protocols.forge.methods.util" );
static numeric::random::RandomGenerator RG( 2211990 ); // magic number, don't change


/// @brief perform union( root, i ) for all 'i' within the closed interval
///  [left, right]
/// @param[in] root position to union with; can be any number, does not have
///  to be a true root of a set
/// @param[in] left start of the interval
/// @param[in] right end of the interval
/// @param[in,out] uf
void
union_interval(
	core::Size const root,
	core::Size const left,
	core::Size const right,
	core::graph::DisjointSets & uf
)
{
	using core::Size;

	assert( left <= right );
	assert( root <= uf.n_nodes() );
	assert( left <= uf.n_nodes() );
	assert( right <= uf.n_nodes() );

	for ( Size i = left; i <= right; ++i ) {
		uf.ds_union( root, i );
	}
}


/// @brief moving left to right, find the first true cutpoint within specified
///  extent
/// @return the cutpoint position, otherwise 0 if not found
/// @details 0, pose.n_residue(), and any cutpoint with lower/upper terminus
///  are not counted as cutpoints
core::Size
find_cutpoint(
	core::pose::Pose const & pose,
	core::Size left,
	core::Size right
)
{
	using core::Size;
	using core::kinematics::FoldTree;

	FoldTree const & ft = pose.fold_tree();

	for ( Size i = left; i <= right; ++i ) {
		if ( ft.is_cutpoint( i ) && i < pose.n_residue() &&
		     !pose.residue( i ).is_lower_terminus() &&
		     !pose.residue( i ).is_upper_terminus()
		) {
			return i;
		}
	}

	return 0;
}


/// @brief moving left to right, count the number of true cutpoints within the
///  specified extent
/// @details 0, pose.n_residue(), and any cutpoint with lower/upper terminus
///  are not counted as cutpoints
core::Size
count_cutpoints(
	core::pose::Pose const & pose,
	core::Size left,
	core::Size right
)
{
	using core::Size;
	using core::kinematics::FoldTree;

	Size n = 0;
	FoldTree const & ft = pose.fold_tree();

	for ( Size i = left; i <= right; ++i ) {
		if ( ft.is_cutpoint( i ) && i < pose.n_residue() &&
		     !pose.residue( i ).is_lower_terminus() &&
		     !pose.residue( i ).is_upper_terminus()
		) {
			++n;
		}
	}

	return n;
}


/// @brief set omega to 180 for a range of residues [left, right]
void
trans_omega(
	core::Size const left,
	core::Size const right,
	core::pose::Pose & pose
)
{
	using core::Size;

	for ( Size i = left; i <= right; ++i ) {
		pose.set_omega( i, 180.0 );
	}
}


/// @brief create Loop object w/ random cutpoint from an Interval
protocols::loops::Loop
interval_to_loop( protocols::forge::build::Interval const & interval ) {
	using core::Size;
	using protocols::loops::Loop;

	// pick a cutpoint fully inside the loop so that there is always
	// at least one moveable residue on each side
	Size const cut = RG.random_range( interval.left, interval.right - 1 );

	return Loop( interval.left, interval.right, cut );
}


/// @brief create fold tree from loops
/// @remarks This is a generic replacement function for the one in protocols::loops
///  and will be moved there in the near future.
core::kinematics::FoldTree
fold_tree_from_loops(
	core::pose::Pose const & pose,
	protocols::loops::Loops const & loops
)
{
	using core::Size;
	using core::kinematics::FoldTree;
	using core::kinematics::MoveMap;
	using protocols::loops::Loop;
	using protocols::loops::Loops;

	using protocols::forge::methods::fold_tree_from_pose;

	// setup movemap, mark only loops moveable, everything else fixed
	MoveMap mm;
	loops.switch_movemap( mm, core::id::BB , true );
	loops.switch_movemap( mm, core::id::CHI , true );

	// Generate initial fold tree from pose.  If existing root is a virtual
	// residue use it, otherwise use the existing root if non-moveable,
	// otherwise use a random non-moveable residue.
	Size root;
	if ( pose.residue( pose.fold_tree().root() ).aa() == core::chemical::aa_vrt ) {
		root = pose.fold_tree().root();
	} else if ( !mm.get_bb( pose.fold_tree().root() ) ) {
		root = pose.fold_tree().root();
	} else { // need to pick a random non-moveable residue

		utility::vector1< Size > fixed_bb;
		for ( Size i = 1, ie = pose.n_residue(); i <= ie; ++i ) {
			if ( !mm.get_bb( i ) ) {
				fixed_bb.push_back( i );
			}
		}

		root = fixed_bb[ RG.random_range( 1, fixed_bb.size() ) ];
	}

	FoldTree ft = fold_tree_from_pose( pose, root, mm );

	// track chain termini to distinguish internal vs terminal loops
	std::set< Size > lower_termini;
	std::set< Size > upper_termini;
	for ( Size i = 1, ie = pose.conformation().num_chains(); i <= ie; ++i ) {
		lower_termini.insert( pose.conformation().chain_begin( i ) );
		upper_termini.insert( pose.conformation().chain_end( i ) );
	}

	// post-modify tree with new loop jump/cuts for internal loop modeling
	for ( Loops::const_iterator i = loops.begin(), ie = loops.end(); i != ie; ++i ) {
		Loop const & loop = *i;

		if ( lower_termini.find( loop.start() ) == lower_termini.end() && upper_termini.find( loop.stop() ) == upper_termini.end() ) {
			ft.new_jump( loop.start() - 1, loop.stop() + 1, loop.cut() );
		}
	}

	return ft;
}


/// @brief set a single loop fold tree
/// @remarks This is a generic replacement function for the one in protocols::loops
///  and will be moved there in the near future.
void
set_single_loop_fold_tree(
	core::pose::Pose & pose,
	protocols::loops::Loop const & loop
)
{
	using core::kinematics::FoldTree;
	using protocols::loops::Loops;

	Loops loops;
	loops.add_loop( loop );

	FoldTree ft = fold_tree_from_loops( pose, loops );
	pose.fold_tree( ft );
}


} // namespace methods
} // namespace forge
} // namespace protocols
