// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/forge/remodel/RemodelLoopMover.cc
/// @brief  Loop modeling protocol based on routines from Remodel and EpiGraft
///         packages in Rosetta++.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Possu Huang (possu@u.washington.edu)

// unit headers
#include <protocols/forge/remodel/RemodelLoopMover.hh>

// package headers
#include <protocols/forge/methods/chainbreak_eval.hh>
#include <protocols/forge/methods/fold_tree_functions.hh>
#include <protocols/forge/methods/util.hh>

// project headers
#include <core/id/TorsionID.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/util/Tracer.hh>
#include <protocols/abinitio/FragmentMover.hh>
#include <protocols/jd2/Job.hh>
#include <protocols/jd2/JobDistributor.hh>
#include <protocols/jd2/JobOutputter.hh>
#include <protocols/loops/ccd_closure.hh>
#include <protocols/loops/loops_main.hh>
#include <protocols/moves/MonteCarlo.hh>

// numeric headers
#include <numeric/random/random.hh>
#include <numeric/random/random_permutation.hh>

// boost headers
#include <boost/format.hpp>

// C++ headers
#include <algorithm>
#include <set>
#include <sstream>
#include <utility>


namespace protocols {
namespace forge {
namespace remodel {


// Tracer instance for this file
// Named after the original location of this code
static core::util::Tracer TR( "protocols.forge.remodel.RemodelLoopMover" );

// RNG
static numeric::random::RandomGenerator RG( 9788221 ); // magic number, don't change


/// @brief default constructor
RemodelLoopMover::RemodelLoopMover() :
	Super(),
	sfx_( core::scoring::ScoreFunctionFactory::create_score_function( "remodel_cen" ) ),
	max_linear_chainbreak_( 0.07 ),
	randomize_loops_( true ),
	allowed_closure_attempts_( 3 ),
	simultaneous_cycles_( 2 ),
	independent_cycles_( 8 ),
	lockdown_cycles_( 30 )
{}


/// @brief loops constructor
RemodelLoopMover::RemodelLoopMover( Loops const & loops ) :
	Super(),
	sfx_( core::scoring::ScoreFunctionFactory::create_score_function( "remodel_cen" ) ),
	loops_( loops ),
	max_linear_chainbreak_( 0.07 ),
	randomize_loops_( true ),
	allowed_closure_attempts_( 3 ),
	simultaneous_cycles_( 2 ),
	independent_cycles_( 8 ),
	lockdown_cycles_( 30 )
{}


/// @brief copy constructor
RemodelLoopMover::RemodelLoopMover( RemodelLoopMover const & rval ) :
	Super( rval ),
	sfx_( rval.sfx_->clone() ),
	false_movemap_( rval.false_movemap_ ),
	loops_( rval.loops_ ),
	max_linear_chainbreak_( rval.max_linear_chainbreak_ ),
	randomize_loops_( rval.randomize_loops_ ),
	allowed_closure_attempts_( rval.allowed_closure_attempts_ ),
	simultaneous_cycles_( rval.simultaneous_cycles_ ),
	independent_cycles_( rval.independent_cycles_ ),
	lockdown_cycles_( rval.lockdown_cycles_ ),
	fragsets_( rval.fragsets_ )
{}


/// @brief default destructor
RemodelLoopMover::~RemodelLoopMover() {}


/// @brief clone this object
RemodelLoopMover::MoverOP RemodelLoopMover::clone() {
	return new RemodelLoopMover( *this );
}


/// @brief create this type of object
RemodelLoopMover::MoverOP RemodelLoopMover::fresh_instance() {
	return new RemodelLoopMover();
}


/// @brief the ScoreFunction to use during modeling;
RemodelLoopMover::ScoreFunction const & RemodelLoopMover::scorefunction() const {
	return *sfx_;
}


/// @brief the ScoreFunction to use during modeling
void RemodelLoopMover::scorefunction( ScoreFunction const & sfx ) {
	sfx_ = new ScoreFunction( sfx );
}


/// @brief add a fragment set
void RemodelLoopMover::add_fragments( FragSetCOP fragset ) {
	if ( fragset->size() > 0 ) {
		fragsets_.push_back( fragset->clone() );
	}
}


/// @brief clear all fragment sets
void RemodelLoopMover::clear_fragments() {
	fragsets_.clear();
}


/// @brief apply defined moves to given Pose
/// @remarks Sets protocols::moves::MS_SUCCESS upon successful closure of
///  all loops, otherwise sets protocols::moves::FAIL_RETRY.
void RemodelLoopMover::apply( Pose & pose ) {
	using core::kinematics::FoldTree;
	using protocols::jd2::JobDistributor;

	using protocols::forge::methods::fold_tree_from_pose;

	// archive
	FoldTree const archive_ft = pose.fold_tree();
	FoldTree const sealed_ft = fold_tree_from_pose( pose, pose.fold_tree().root(), MoveMap() ); // used during structure accumulation

	// for accumulation of closed structures (only return the best)
	std::multimap< Real, PoseOP > accumulator;

	// setup parameters -- linearly scale the chainbreak weight
	Real const final_standard_cbreak_weight = 5.0;
	Real const cbreak_increment = final_standard_cbreak_weight / total_standard_cycles();

	// currently no scaling during lockdown
	Real const final_lockdown_cbreak_weight = 5.0;
	Real const lockdown_cbreak_increment = ( final_lockdown_cbreak_weight - final_standard_cbreak_weight ) / lockdown_cycles();

	assert( final_lockdown_cbreak_weight >= final_standard_cbreak_weight );

	// mark linear chainbreak in scoring function; this will be incremented
	// within simultaneous and independent stages
	ScoreFunction sfx = *sfx_;
	sfx.set_weight( core::scoring::linear_chainbreak, 0.0 );

	// randomize loops
	randomize_stage( pose );
	sfx( pose );

	// setup monte carlo
	Real const temp = 2.0;
	MonteCarlo mc( pose, sfx, temp );

	for ( Size attempt = 1; attempt <= allowed_closure_attempts_; ++attempt ) {
		TR << "* closure_attempt " << attempt << std::endl;

		// reset score function at the beginning of each attempt
		mc.score_function( sfx );

		// simultaneous loop movements using fragment + ccd_move (default 20% of the time)
		simultaneous_stage( pose, mc, cbreak_increment );

		// closure is hard, so attempt closure for each loop independently using
		// fragment + ccd_move (default 80% of the time)
		independent_stage( pose, mc, cbreak_increment );

		// "boost": if any loops are not closed but within a chainbreak interval, attempt
		// to close them with 1-mer + ccd_move.
		lockdown_stage( pose, mc, lockdown_cbreak_increment );

		// check to see if all loops closed, if so rescore w/out chainbreak
		// and store in accumulator
		if ( check_closure_criteria( pose ) ) {
			PoseOP pose_prime = new Pose( pose );
			pose_prime->fold_tree( sealed_ft );
			sfx( *pose_prime );
			accumulator.insert( std::make_pair( pose_prime->energies().total_energy(), pose_prime ) );

			// now randomize the loops again for a new starting point
			if ( attempt < allowed_closure_attempts_ ) {
				randomize_stage( pose );
				mc.reset( pose );
			}
		} else {
			// Still broken, so perform a random smallest-mer insertion into each
			// loop before cycling again, otherwise too easy for the trajectory to
			// get trapped.
			insert_random_smallestmer_per_loop( pose, true );
			mc.reset( pose );
		}

//		std::ostringstream ss;
//		ss << "rlm." << attempt << ".";
//		JobDistributor::get_instance()->job_outputter()->other_pose(
//			JobDistributor::get_instance()->current_job(),
//			pose,
//			ss.str()
//		);
	}

	TR << "* " << accumulator.size() << " / " << allowed_closure_attempts_ << "   closed / attempts " << std::endl;

	// return the best structure if available, otherwise mark failure
	if ( !accumulator.empty() ) {
		pose = *( accumulator.begin()->second );
		set_last_move_status( protocols::moves::MS_SUCCESS );
	} else {
		set_last_move_status( protocols::moves::FAIL_RETRY );
	}

	// set original topology
	pose.fold_tree( archive_ft );
}


/// @brief randomize loops
void RemodelLoopMover::randomize_stage( Pose & pose ) {
	TR << "** randomize_stage" << std::endl;

	// simul movemap -- all loops moveable
	MoveMap movemap;
	mark_loops_moveable( loops_, movemap, true );
	enforce_false_movemap( movemap );

	// set appropriate topology
	pose.fold_tree( protocols::forge::methods::fold_tree_from_loops( pose, loops_ ) );

	// init fragment mover for each fragment set
	FragmentMoverOPs frag_movers = create_fragment_movers( movemap );

	Size const n_moveable = count_moveable_residues( movemap, 1, pose.n_residue() );

	// insert random number of fragments = n_frag_movers * moveable_residues
	for ( FragmentMoverOPs::iterator i = frag_movers.begin(), ie = frag_movers.end(); i != ie; ++i ) {
		for ( Size j = 0; j < n_moveable; ++j ) {
			(*i)->apply( pose );
		}
	}

	check_closure_criteria( pose, true );
}


/// @brief find the smallest fragment size and insert a single such
///  smallmer into each loop; for breaking up trapped trajectories
/// @param[in,out] pose The pose to modify.
/// @param[in] only_broken_loop If true, only insert into broken loops,
///  otherwise insert into all. (default true)
void RemodelLoopMover::insert_random_smallestmer_per_loop(
	Pose & pose,
	bool const only_broken_loops
)
{
	using core::kinematics::FoldTree;

	// determine the right set of loops to insert fragments
	Loops loops_to_model;

	if ( only_broken_loops ) {
		loops_to_model = determine_loops_to_model( pose );
	} else {
		loops_to_model = loops_;
	}

	// set appropriate topology
	pose.fold_tree( protocols::forge::methods::fold_tree_from_loops( pose, loops_to_model ) );

	// find the size of the smallest fragments
	Size smallestmer_size = ( *fragsets_.begin() )->max_frag_length();
	for ( FragSetOPs::const_iterator f = fragsets_.begin(), fe = fragsets_.end(); f != fe; ++f ) {
		smallestmer_size = std::min( smallestmer_size, (*f)->max_frag_length() );
	}

	// insert fragments
	FragmentMoverOPs frag_movers = create_per_loop_fragment_movers( loops_to_model, smallestmer_size );

	for ( FragmentMoverOPs::iterator i = frag_movers.begin(), ie = frag_movers.end(); i != ie; ++i ) {
		(*i)->apply( pose );
	}
}


/// @brief simultaneous stage: multiple loop movement prior to MC accept/reject
void RemodelLoopMover::simultaneous_stage(
	Pose & pose,
	MonteCarlo & mc,
	Real const cbreak_increment
)
{
	using core::kinematics::FoldTree;

	using protocols::loops::add_cutpoint_variants;
	using protocols::loops::ccd_moves;
	using protocols::loops::remove_cutpoint_variants;
	using numeric::random::random_permutation;

	TR << "** simultaneous_stage" << std::endl;

	// Make a local copy of the Loops list.  At this stage all loops
	// are malleable so we don't use determine_loops_to_model().
	Loops loops_to_model = loops_;
	TR << "   n_loops = " << loops_to_model.size() << std::endl;

	if ( loops_to_model.size() == 0 ) { // nothing to do...
		return;
	}

	// Create fragment movers for each loop for each fragment set.  We want
	// to allow an equal probability of movement per-loop, rather than
	// per-residue.
	FragmentMoverOPs frag_movers = create_per_loop_fragment_movers( loops_to_model );
	assert( !frag_movers.empty() );

	// set appropriate topology
	pose.fold_tree( protocols::forge::methods::fold_tree_from_loops( pose, loops_to_model ) );

	// add cutpoint variants
	add_cutpoint_variants( pose );
	mc.reset( pose );

	// setup master movemap covering all loops -- used only for tracking purposes
	MoveMap movemap;
	mark_loops_moveable( loops_to_model, movemap, true );
	enforce_false_movemap( movemap );

	// parameters
	Size const n_moveable = count_moveable_residues( movemap, 1, pose.n_residue() );
	Size const n_standard_cycles = total_standard_cycles();
	Size const max_outer_cycles = simultaneous_cycles();
	Size const max_inner_cycles = std::max( 50 * loops_to_model.size(), 10 * n_moveable );

	// reset counters
	mc.reset_counters();

	// simul frag + ccd_move
	for ( Size outer = 1; outer <= max_outer_cycles; ++outer ) {
		// increment the chainbreak weight
		ScoreFunction sfx = mc.score_function();
		sfx.set_weight(
			core::scoring::linear_chainbreak,
			sfx.get_weight( core::scoring::linear_chainbreak ) + cbreak_increment
		);
		mc.score_function( sfx );

		// recover low
		pose = mc.lowest_score_pose();

		for ( Size inner = 1; inner <= max_inner_cycles; ++inner ) {

			if ( RG.uniform() * n_standard_cycles > outer ) {
				// fragments
				random_permutation( frag_movers.begin(), frag_movers.end(), RG );
				for ( FragmentMoverOPs::iterator i = frag_movers.begin(), ie = frag_movers.end(); i != ie; ++i ) {
					(*i)->apply( pose );
					mc.boltzmann( pose, "simul_frag" );
				}
			} else {
				// per-loop ccd
				random_permutation( loops_to_model.v_begin(), loops_to_model.v_end(), RG );
				for ( Loops::const_iterator l = loops_to_model.begin(), le = loops_to_model.end(); l != le; ++l ) {
					if ( !l->is_terminal( pose ) ) {
						ccd_moves( 10, pose, movemap, l->start(), l->stop(), l->cut() );
						mc.boltzmann( pose, "ccd_move" );
					}
				}
			}

		} // inner_cycles

	} // outer_cycles

	// recover low
	pose = mc.lowest_score_pose();

	// report status
	mc.score_function().show_line_headers( TR );
	TR << std::endl;
	mc.score_function().show_line( TR, pose );
	TR << std::endl;
	mc.show_state();

	check_closure_criteria( pose, true );
	remove_cutpoint_variants( pose );
}


/// @brief independent stage: single loop movement prior to MC accept/reject
void RemodelLoopMover::independent_stage(
	Pose & pose,
	MonteCarlo & mc,
	Real const cbreak_increment
)
{
	using protocols::forge::methods::linear_chainbreak;
	using protocols::loops::add_cutpoint_variants;
	using protocols::loops::ccd_moves;
	using protocols::loops::remove_cutpoint_variants;

	TR << "** independent_stage" << std::endl;

	// setup loops
	Loops loops_to_model = determine_loops_to_model( pose );
	TR << "   n_loops = " << loops_to_model.size() << std::endl;

	if ( loops_to_model.size() == 0 ) { // nothing to do...
		return;
	}

	// parameters
	Size const n_standard_cycles = total_standard_cycles();
	Size const max_outer_cycles = independent_cycles();

	// per-loop frag + ccd_move
	for ( Loops::const_iterator l = loops_to_model.begin(), le = loops_to_model.end(); l != le; ++l ) {
		Loop const & loop = *l;

		// movemap
		MoveMap movemap;
		mark_loop_moveable( loop, movemap, true );
		enforce_false_movemap( movemap );

		// fragment movers
		FragmentMoverOPs frag_movers = create_fragment_movers( movemap );
		assert( !frag_movers.empty() );

		// parameters
		Size const n_moveable = count_moveable_residues( movemap, loop.start(), loop.stop() );
		Size const max_inner_cycles = std::max( static_cast< Size >( 50 ), 10 * n_moveable );

		// set appropriate topology
		protocols::forge::methods::set_single_loop_fold_tree( pose, loop );

		// add cutpoint variants
		add_cutpoint_variants( pose );
		mc.reset( pose );

		// reset counters
		mc.reset_counters();

		// do closure
		for ( Size outer = 1; outer <= max_outer_cycles; ++outer ) {
			// increment the chainbreak weight
			ScoreFunction sfx = mc.score_function();
			sfx.set_weight(
				core::scoring::linear_chainbreak,
				sfx.get_weight( core::scoring::linear_chainbreak ) + cbreak_increment
			);
			mc.score_function( sfx );

			// recover low
			pose = mc.lowest_score_pose();

			for ( Size inner = 1; inner <= max_inner_cycles; ++inner ) {
				// fragments
				if ( loop.is_terminal( pose ) || RG.uniform() * n_standard_cycles > ( outer + simultaneous_cycles() ) ) {
					random_permutation( frag_movers.begin(), frag_movers.end(), RG );
					for ( FragmentMoverOPs::iterator i = frag_movers.begin(), ie = frag_movers.end(); i != ie; ++i ) {
						(*i)->apply( pose );
						mc.boltzmann( pose, "frag" );
					}
				} else { // ccd
					ccd_moves( 10, pose, movemap, loop.start(), loop.stop(), loop.cut() );
					mc.boltzmann( pose, "ccd_move" );
				}

			} // inner_cycles

		} // outer_cycles

		// recover low
		pose = mc.lowest_score_pose();

		// report status
		mc.score_function().show_line_headers( TR );
		TR << std::endl;
		mc.score_function().show_line( TR, pose );
		TR << std::endl;
		mc.show_state();

		// remove cutpoints
		remove_cutpoint_variants( pose );
	}

	check_closure_criteria( pose, true );
}


/// @brief lock down stage: close loops within some threshold
///  w/ smallest-mer (typically 1-mer) + ccd_move only
/// @details This stage differs from simultaneous_stage() and independent_stage()
///  in that once a loop is closed, it breaks out of the closure cycle and goes
///  to the next one.  The rationale is that once we hit the lockdown_stage()
///  we are desperate to close the loop, so we sacrifice diversity and instead
///  just seek a closed solution.
void RemodelLoopMover::lockdown_stage(
	Pose & pose,
	MonteCarlo & mc,
	Real const cbreak_increment
)
{
	// Notes: the current implementation of lockdown_stage() differs somewhat
	// from the equivalent, original "boost" stage within ++Remodel/EpiGraft.
	// In the old implementation, the boost cycles continued the
	// independent_stage().  As a result, the % time spent in fragment insertion
	// vs ccd continued to drop until the procedure was basically only doing
	// ccd when the total number of cycles reached 100.  In addition, the
	// chainbreak weight continued to increment until it reached 50 (at total
	// cycle 100).
	using protocols::abinitio::ClassicFragmentMover;

	using protocols::forge::methods::linear_chainbreak;
	using protocols::loops::add_cutpoint_variants;
	using protocols::loops::ccd_moves;
	using protocols::loops::remove_cutpoint_variants;

	TR << "** lockdown_stage" << std::endl;

	// setup loops
	Loops pre_loops_to_model = determine_loops_to_model( pose );
	Loops loops_to_model;

	// filter for non-terminal loops that are within tolerance
	Real const cbreak_tolerance = 1.0;
	for ( Loops::const_iterator l = pre_loops_to_model.begin(), le = pre_loops_to_model.end(); l != le; ++l ) {
		if ( !l->is_terminal( pose ) ) {
			Real const cbreak = linear_chainbreak( pose, l->cut() );
			if ( cbreak < cbreak_tolerance ) {
				loops_to_model.add_loop( *l );
			}
		}
	}

	TR << "   n_loops = " << loops_to_model.size() << std::endl;
	if ( loops_to_model.size() == 0 ) { // nothing to do...
		return;
	}

	// find the size of the smallest fragments
	Size smallestmer_size = ( *fragsets_.begin() )->max_frag_length();
	for ( FragSetOPs::const_iterator f = fragsets_.begin(), fe = fragsets_.end(); f != fe; ++f ) {
		smallestmer_size = std::min( smallestmer_size, (*f)->max_frag_length() );
	}
	TR << "** lockdown stage: using fragment size = " << smallestmer_size << std::endl;

	// Parameters.  Note that fragments often get rejected at this stage, so
	// recommend keeping the number of insertions low and the number of ccd_move
	// high.
	Size const max_outer_cycles = lockdown_cycles();
	Real const frag_mover_probability = 0.25; // do 1-mer insertions only 25% of the time

	// 1-mer frag + ccd_move
	for ( Loops::const_iterator l = loops_to_model.begin(), le = loops_to_model.end(); l != le; ++l ) {
		Loop const & loop = *l;

		// movemap
		MoveMap movemap;
		mark_loop_moveable( loop, movemap, true );
		enforce_false_movemap( movemap );

		// prepare smallest-mer fragment movers
		FragmentMoverOPs frag1_movers = create_fragment_movers( movemap, smallestmer_size );

		// parameters
		Size const n_moveable = count_moveable_residues( movemap, loop.start(), loop.stop() );
		Size const max_inner_cycles = std::max( static_cast< Size >( 50 ), 10 * n_moveable );

		// set appropriate topology
		protocols::forge::methods::set_single_loop_fold_tree( pose, loop );

		// add cutpoint variants
		add_cutpoint_variants( pose );
		mc.reset( pose );

		// reset counters
		mc.reset_counters();

		// do closure
		for ( Size outer = 1; outer <= max_outer_cycles; ++outer ) {
			// increment the chainbreak weight
			ScoreFunction sfx = mc.score_function();
			sfx.set_weight(
				core::scoring::linear_chainbreak,
				sfx.get_weight( core::scoring::linear_chainbreak ) + cbreak_increment
			);
			mc.score_function( sfx );

			pose = mc.lowest_score_pose();

			// Going into the lockdown stage implies we are "desperate" to close
			// the loop and don't care about diversity anymore, so continue to
			// cycle only until the loop is closed.
			if ( linear_chainbreak( pose, loop.cut() ) <= max_linear_chainbreak_ ) {
				break;
			}

			for ( Size inner = 1; inner <= max_inner_cycles; ++inner ) {
				if ( !frag1_movers.empty() && RG.uniform() < frag_mover_probability ) { // 1-mer insertions

					random_permutation( frag1_movers.begin(), frag1_movers.end(), RG );
					for ( FragmentMoverOPs::iterator i = frag1_movers.begin(), ie = frag1_movers.end(); i != ie; ++i ) {
						(*i)->apply( pose );
						mc.boltzmann( pose, "frag1" );
					}

				} else { // ccd_move
					ccd_moves( 10, pose, movemap, loop.start(), loop.stop(), loop.cut() );
					mc.boltzmann( pose, "ccd_move" );
				}
			} // inner_cycles

		} // outer_cycles

		// recover low
		pose = mc.lowest_score_pose();

		// report status
		mc.score_function().show_line_headers( TR );
		TR << std::endl;
		mc.score_function().show_line( TR, pose );
		TR << std::endl;
		mc.show_state();

		// remove cutpoint variants
		remove_cutpoint_variants( pose );
	}

	check_closure_criteria( pose, true );
}


/// @brief determine which loops need modeling wrt to given Pose
/// @remarks Skips closed loops and shuffles the order of the remaining
///  loops.
RemodelLoopMover::Loops RemodelLoopMover::determine_loops_to_model( Pose & pose ) {
	using protocols::forge::methods::linear_chainbreak;
	Loops loops_to_model;

	for ( Loops::const_iterator l = loops_.begin(), le = loops_.end(); l != le; ++l ) {
		bool skip_loop = false;

		if ( !l->is_terminal( pose ) ) {
			skip_loop |= ( linear_chainbreak( pose, l->cut() ) <= max_linear_chainbreak_ ); // loop already closed?
		}

		if ( !skip_loop ) {
			loops_to_model.add_loop( *l );
		}
	}

	// shuffle the order
	random_permutation( loops_to_model.v_begin(), loops_to_model.v_end(), RG );

	return loops_to_model;
}


/// @brief check all loops for closure criteria
/// @param[in] pose The pose being checked.
/// @param[in] show_in_tracer Output state of each loop to tracer?
/// @return true if all criteria pass, false otherwise
bool RemodelLoopMover::check_closure_criteria(
	Pose & pose,
	bool const show_in_tracer
)
{
	using protocols::forge::methods::linear_chainbreak;

	// boost::format here does not appear to be doing what I want it to do...
	// The format string is probably borked.
	boost::format fmt( "%|5t|%1% %|5t|%2% %|5t|%3% %|8t|%4%" );
	if ( show_in_tracer ) {
		TR << fmt % "start" % "stop" % "cut" % "cbreak" << std::endl;
	}

	bool all_loops_pass = true;

	for ( Loops::const_iterator l = loops_.begin(), le = loops_.end(); l != le; ++l ) {
		Real cbreak = 0.0;
		if ( !l->is_terminal( pose ) ) {
			cbreak = linear_chainbreak( pose, l->cut() );
			all_loops_pass &= ( cbreak <= max_linear_chainbreak_ );
		}

		if ( show_in_tracer ) {
			TR << fmt % l->start() % l->stop() % l->cut() % cbreak << std::endl;
		}
	}

	return all_loops_pass;
}


/// @brief return fragment movers for the list of internally kept fragment sets
/// @param[in] movemap Use this movemap when initializing fragment movers.
/// @param[in] largest_frag_size Only use fragment sets whose largest fragment
///  size is this number.  If zero, uses all fragment sets.
RemodelLoopMover::FragmentMoverOPs
RemodelLoopMover::create_fragment_movers(
	MoveMap const & movemap,
	Size const largest_frag_size
)
{
	using protocols::abinitio::ClassicFragmentMover;

	FragmentMoverOPs frag_movers;
	for ( FragSetOPs::const_iterator f = fragsets_.begin(), fe = fragsets_.end(); f != fe; ++f ) {

		if ( largest_frag_size == 0 || (*f)->max_frag_length() <= largest_frag_size ) {
			ClassicFragmentMover * cfm = new ClassicFragmentMover( *f, movemap.clone() );
			cfm->set_check_ss( false );
			cfm->enable_end_bias_check( false );
			frag_movers.push_back( cfm );
		}

	}

	return frag_movers;
}


/// @brief append fragment movers for the list of internally kept fragment sets
/// @param[in] movemap Use this movemap when initializing fragment movers.
/// @param[out] frag_movers Append fragment movers to this list.
/// @param[in] largest_frag_size Only use fragment sets whose largest fragment
///  size is this number.  If zero, uses all fragment sets.
void RemodelLoopMover::create_fragment_movers(
	MoveMap const & movemap,
	FragmentMoverOPs & frag_movers,
	Size const largest_frag_size
) {
	using protocols::abinitio::ClassicFragmentMover;

	for ( FragSetOPs::const_iterator f = fragsets_.begin(), fe = fragsets_.end(); f != fe; ++f ) {

		if ( largest_frag_size == 0 || (*f)->max_frag_length() <= largest_frag_size ) {
			ClassicFragmentMover * cfm = new ClassicFragmentMover( *f, movemap.clone() );
			cfm->set_check_ss( false );
			cfm->enable_end_bias_check( false );
			frag_movers.push_back( cfm );
		}

	}
}


/// @brief create per-loop fragment movers: 1 fragment mover for each loop (uses
///  movemaps to lock down non-loop residues)
/// @param[in] loops The loops to use.
/// @param[in] largest_frag_size Only use fragment sets whose largest fragment
///  size is this number.  If zero, uses all fragment sets.
RemodelLoopMover::FragmentMoverOPs RemodelLoopMover::create_per_loop_fragment_movers(
	Loops const & loops,
	Size const largest_frag_size
)
{
	// Create fragment movers for each loop for each fragment set.  Here we
	// want to allow an equal probability of movement per-loop, rather than
	// per-residue.
	FragmentMoverOPs frag_movers;
	for ( Loops::const_iterator l = loops.begin(), le = loops.end(); l != le; ++l ) {
		MoveMap mm;
		mark_loop_moveable( *l, mm, true );
		enforce_false_movemap( mm );
		create_fragment_movers( mm, frag_movers, largest_frag_size );
	}

	return frag_movers;
}


/// @brief enforce settings in the false movemap
void RemodelLoopMover::enforce_false_movemap( MoveMap & movemap ) {
	// enforce everything in the false movemap
	movemap.import_false( false_movemap_ );
}


/// @brief mark bb/chi torsions of multiple loops moveable in a movemap
/// @param[in] loops The loops to use.
/// @param[out] movemap The movemap to modify.
/// @param[in] allow_omega Allow bb omega to move? (should be yes when
///  doing either fragment insertion or scoring function has omega
///  tether, otherwise should probably be no)
void RemodelLoopMover::mark_loops_moveable(
	Loops const & loops,
	MoveMap & movemap,
	bool const allow_omega
)
{
	for ( Loops::const_iterator l = loops.begin(), le = loops.end(); l != le; ++l ) {
		mark_loop_moveable( *l, movemap, allow_omega );
	}
}


/// @brief mark bb/chi torsion of a single loop moveable in movemap
/// @param[in] loops The loop to use.
/// @param[out] movemap The movemap to modify.
/// @param[in] allow_omega Allow bb omega to move? (should be yes when
///  doing either fragment insertion or scoring function has omega
///  tether, otherwise should probably be no)
void RemodelLoopMover::mark_loop_moveable(
	Loop const & loop,
	MoveMap & movemap,
	bool const allow_omega
)
{
	using core::id::BB;
	using core::id::omega_torsion;
	using core::id::TorsionID;

	for ( Size i = loop.start(), ie = loop.stop(); i <= ie; ++i ) {
		movemap.set_bb( i, true );
		movemap.set_chi( i, true );

		if ( !allow_omega ) {
			movemap.set( TorsionID( i, BB, omega_torsion ), false );
		}
	}
}


/// @brief count number of residues with moveable backbone torsions in the
///  given range [left, right]
RemodelLoopMover::Size RemodelLoopMover::count_moveable_residues(
	MoveMap const & movemap,
	Size const left,
	Size const right
)
{
	Size n_moveable = 0;

	// Count total number of residues w/ moveable backbone in movemap.
	// Depending on types of fragments (possible non-backbone?) consider
	// changing this in the future to check chi/other dof as well.
	for ( Size i = left; i <= right; ++i ) {
		if ( movemap.get_bb( i ) ) {
			++n_moveable;
		}
	}

	return n_moveable;
}


} // remodel
} // forge
} // protocols

