// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file Entity.hh
/// @brief the unit employed/optimized by GeneticAlgorithm
/// @author ashworth

#ifndef INCLUDED_protocols_genetic_algorithm_Entity_HH
#define INCLUDED_protocols_genetic_algorithm_Entity_HH

#include <core/types.hh>

#include <utility/exit.hh>
#include <utility/pointer/ReferenceCount.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/pointer/access_ptr.hh>
#include <utility/vector1.hh>

#include <boost/functional/hash.hpp> // hash_range

#include <ObjexxFCL/formatted.o.hh>
using namespace ObjexxFCL;

#include <iostream>

namespace protocols {
namespace genetic_algorithm {

template <typename T>
class Entity : public utility::pointer::ReferenceCount {

public:
	typedef utility::pointer::owning_ptr< Entity<T> > OP;
	typedef utility::pointer::owning_ptr< Entity<T> const > COP;
	typedef utility::pointer::access_ptr< Entity<T> const > CAP;
	typedef utility::vector1< COP > COPs;
	typedef utility::vector1< CAP > CAPs;

	Entity() : utility::pointer::ReferenceCount(), fitness_(0.), fitness_valid_(false) {}
	virtual ~Entity(){}

	////@brief construct Entity from std::string (e.g. from file)
	Entity( std::string const & line );

	////@brief construct a duplicate Entity from another entity
	Entity( Entity const & entity );

	virtual OP clone() const { return new Entity(*this); }

	virtual void set_traits_size( core::Size size ) { traits_.resize(size); fitness_valid_ = false; }
	virtual void set_traits( utility::vector1<T> const & traits ) { traits_ = traits; fitness_valid_ = false; }
	virtual utility::vector1<T> const & traits() const { return traits_; }
	virtual void set_fitness( core::Real val ) { fitness_ = val; fitness_valid_ = true; }
	virtual core::Real fitness() const { return fitness_; }
	// safer to have this handled automatically for now
	//virtual void set_fitness_valid( bool fitness_valid ) { fitness_valid_ = fitness_valid; }
	virtual bool fitness_valid() const { return fitness_valid_; }

	virtual bool operator == ( Entity<T> const & other ) const
	{
		return ( traits_ == other.traits() );
	}

	virtual bool operator < ( Entity<T> const & other ) const
	{
		return ( fitness_ < other.fitness() );
	}

	virtual void show( std::ostream & os ) const
	{
		os << "Entity with traits:";
		utility::vector1<T> const & seq( this->traits() );
		for ( typename utility::vector1<T>::const_iterator it( seq.begin() ), end( seq.end() );
					it != end; ++it ) {
			os << " " << *it;
		}
		os << " and fitness " << fmt::F(6,3,this->fitness());
	}

	virtual void write_checkpoint( std::ostream & os ) const;
	virtual bool read_checkpoint( std::istream & is );

private:
	utility::vector1<T> traits_;
	core::Real fitness_;
	bool fitness_valid_;
};

template <typename T>
Entity<T>::Entity( std::string const & line )
{
	std::istringstream linestream( line );
	if ( !read_checkpoint(linestream) ) utility_exit_with_message( "invalid string " + line );
}

template <typename T>
Entity<T>::Entity( Entity const & entity ) :
	utility::pointer::ReferenceCount(),
	traits_(entity.traits()),
	fitness_(entity.fitness()),
	fitness_valid_(entity.fitness_valid())
{}

template <typename T>
void
Entity<T>::write_checkpoint(
	std::ostream & os
) const
{
	os << "traits";
	for ( typename utility::vector1<T>::const_iterator key( traits_.begin() ),
				end( traits_.end() ); key != end; ++key ) {
		os << " " << *key;
	}
	os << " fitness " << fitness_;
}

template <typename T>
bool
Entity<T>::read_checkpoint(
	std::istream & is
)
{
	std::string word;
	if (!(is >> word)) return false;
	if ( word != "traits" ) return false;
	while ( is >> word ) {
		if ( word == "fitness" ) break;
		traits_.push_back( T(word) );
	}
	if (is >> fitness_) {
		fitness_valid_ = true;
		return true;
	}
	return false;
}

template <typename T>
std::ostream & operator << ( std::ostream & os, Entity<T> const & entity )
{
	entity.show(os);
	return os;
}

///@brief for sorting owning pointers by that to which they point
template <typename T>
bool lt_OP_deref(
	utility::pointer::owning_ptr<T> const & a,
	utility::pointer::owning_ptr<T> const & b
)
{
	return *a < *b;
}

///@brief for assessing equality between owning pointers by that to which they point
template <typename T>
bool eq_OP_deref(
	utility::pointer::owning_ptr<T> const & a,
	utility::pointer::owning_ptr<T> const & b
)
{
	return *a == *b;
}

template <typename T>
struct
Vec1Hash {
	std::size_t operator() ( utility::vector1<T> const & vec1 ) const {
		return boost::hash_range( vec1.begin(), vec1.end() );
	}
};

} // namespace genetic_algorithm
} // namespace protocols

#endif
