// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/idealize/idealize.cc
/// @brief
/// @author

// Unit Headers
#include <protocols/idealize/idealize.hh>


// // Rosetta Headers
#include <core/types.hh>

#include <core/conformation/ResidueFactory.hh>
#include <core/conformation/util.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>

#include <core/pose/Pose.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/AtomPairConstraint.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/options/option.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>


#include <core/util/basic.hh>
#include <core/util/Tracer.hh> // tracer output

// //Utility Headers
#include <numeric/random/random.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// // C++ Headers

namespace protocols {
namespace idealize {

using namespace core;

static util::Tracer TR( "protocols.idealize" );

static numeric::random::RandomGenerator RG(76223);


/// @brief Add the idealize constraints to the pose's constraint set
void
IdealizeMover::setup_idealize_constraints( pose::Pose & pose )
{
	using namespace scoring::constraints;
	using namespace conformation;
	using namespace id;

	Real const heavyatom_dis2_threshold( 5.5 * 5.5 );
	Real const polarH_dis2_threshold( 2.5 * 2.5 );

	Real const atom_pair_sdev( 0.25 );
	Real const coord_sdev( 0.1 );


	Size const nres( pose.total_residue() );

	runtime_assert( pose.residue( nres ).aa() == core::chemical::aa_vrt ); // should already have setup for using coordinate constraints

	Size total_atompairs( 0 );

	if ( atom_pair_constraint_weight_ != 0.0 ) {
		for ( Size i=1; i<= nres-1; ++i ) {
			Residue const & i_rsd( pose.residue(i) );

			for ( Size j=i+1; j<= nres-1; ++j ) {
				Residue const & j_rsd( pose.residue(j) );

				for ( Size ii = 1; ii<= i_rsd.natoms(); ++ii ) {
					chemical::AtomType const & it( i_rsd.atom_type( ii ) );

					for ( Size jj = 1; jj<= j_rsd.natoms(); ++jj ) {
						chemical::AtomType const & jt( j_rsd.atom_type( jj ) );

						Real const dis2( i_rsd.xyz( ii ).distance_squared( j_rsd.xyz( jj ) ) );

						if ( ( it.is_polar_hydrogen() && jt.is_acceptor()  && dis2 <    polarH_dis2_threshold ) ||
								 ( jt.is_polar_hydrogen() && it.is_acceptor()  && dis2 <    polarH_dis2_threshold ) ||
								 ( it.is_heavyatom()      && jt.is_heavyatom() && dis2 < heavyatom_dis2_threshold ) ) {

							pose.add_constraint( new AtomPairConstraint( AtomID(ii,i), AtomID(jj,j),
																	 new HarmonicFunc( std::sqrt( dis2 ), atom_pair_sdev ))
							);
							++total_atompairs;
						}
					} // jj
				} // ii
			} // j>=i
		} // i
	}

	TR.Info << "total atompairs: " << total_atompairs << std::endl;

	if ( coordinate_constraint_weight_ != 0.0 ) {
		for ( Size i=1; i<= nres-1; ++i ) {
			Residue const & i_rsd( pose.residue(i) );
			for ( Size ii = 1; ii<= i_rsd.natoms(); ++ii ) {
				pose.add_constraint( new CoordinateConstraint( AtomID(ii,i), AtomID(1,nres), i_rsd.xyz( ii ),
															new HarmonicFunc( 0.0, coord_sdev ) )
				);
			}
		}
	} // coordinate_constraint_weight_ != 0
} // setup_idealize_constraints

/// helper
void
dihedral_distance(
	pose::Pose const & pose1,
	pose::Pose const & pose2,
	utility::vector1< bool > const & use_pos,
	Real & avg_bb_angle_dev,
	Real & max_bb_angle_dev,
	Real & avg_chi_angle_dev,
	Real & max_chi_angle_dev
)
{
	using util::subtract_degree_angles;

	avg_bb_angle_dev = 0.0;
	max_bb_angle_dev = 0.0;

	avg_chi_angle_dev = 0.0;
	max_chi_angle_dev = 0.0;

	Size nchi_dihedrals(0), nbb_dihedrals(0);

	for ( Size pos = 1; pos <= pose1.total_residue(); ++pos ) {
		if ( ! use_pos[ pos ] ) continue;

		conformation::Residue const & rsd1( pose1.residue( pos ) );
		conformation::Residue const & rsd2( pose2.residue( pos ) );

		if ( !rsd1.is_polymer() ) continue;

		Size const nbb ( rsd1.n_mainchain_atoms() );
		Size const nchi( rsd1.nchi() );
		runtime_assert( rsd2.is_polymer() && rsd2.nchi() ==  nchi && rsd2.n_mainchain_atoms() == nbb );

		// first the bb dev's
		for ( Size i=1; i<= nbb; ++i ) {
			if ( ( i ==     1 && rsd1.is_lower_terminus() ) ||
					 ( i >= nbb-1 && rsd1.is_upper_terminus() ) ) {
				continue;
			}
			Real const dev( std::abs( subtract_degree_angles( rsd1.mainchain_torsion( i ), rsd2.mainchain_torsion(i) ) ) );
			//if ( dev > 0.01 ) std::cout << "bbdev: " << pos << ' ' << pose1.residue(pos).name1() << ' ' << i << ' ' <<
			//										dev << ' ' << rsd1.mainchain_torsion( i ) << ' ' <<  rsd2.mainchain_torsion(i) << std::endl;
			avg_bb_angle_dev += dev;
			++nbb_dihedrals;
			max_bb_angle_dev = std::max( max_bb_angle_dev, dev );
		}

		for ( Size i=1; i<= nchi; ++i ) {
			Real const dev( std::abs( subtract_degree_angles( rsd1.chi( i ), rsd2.chi( i ) ) ) );
			avg_chi_angle_dev += dev;
			++nchi_dihedrals;
			max_chi_angle_dev = std::max( max_chi_angle_dev, dev );
		}
	}

	avg_bb_angle_dev  /=  nbb_dihedrals;
	avg_chi_angle_dev /= nchi_dihedrals;
}

// positions within window of the idealized positions will move during minimization
void
basic_idealize(
	pose::Pose & pose,
	utility::vector1< Size > pos_list, // local copy
	scoring::ScoreFunction const & scorefxn,
	bool const fast
)
{
	using namespace optimization;
	using namespace id;
	using namespace ObjexxFCL::fmt;
	using scoring::all_atom_rmsd;

	Size const window_width( 3 ); // window:  from pos-window_width to pos+window_width

	pose::Pose const start_pose( pose );
	Size const nres ( pose.total_residue() );
	Size const njump( pose.fold_tree().num_jump() );

	// setup the minimizer options
	MinimizerOptions options( "dfpmin", 0.001, true /*use_nblist*/, false /*deriv_check*/ );
	kinematics::MoveMap final_mm;

	bool const lastjumpmin (
		pose.residue( nres ).aa() == core::chemical::aa_vrt &&
		pose.fold_tree().upstream_jump_residue( njump ) == int(nres)
	);

	TR.Info << "lastjumpmin: " << lastjumpmin << std::endl;
	utility::vector1< bool > idealized( nres, false );

	while ( !pos_list.empty() ) {
		Size const seqpos( pos_list[ static_cast< Size >( RG.uniform() * pos_list.size() + 1 ) ] );
		pos_list.erase( std::find( pos_list.begin(), pos_list.end(), seqpos ) );

		// idealize the mainchain + sidechain
		//pose.dump_pdb( "pre_idl_"+right_string_of(seqpos,4,'0')+".pdb" );
		if( seqpos > (Size)pose.conformation().size() ){ continue; }
		conformation::idealize_position( seqpos, pose.conformation() );
		//pose.dump_pdb( "post_idl_"+right_string_of(seqpos,4,'0')+".pdb" );
		idealized[ seqpos ] = true;

		// setup the window of positions to minimize, also records flexible positions for the final minimize
		utility::vector1< Size > window;
		for ( Size i=seqpos; i >= seqpos-window_width; --i ) {
			window.push_back( i );
			if ( i == 1 || pose.residue(i).is_lower_terminus() ) break;
		}
		for ( Size i=seqpos; i <= seqpos+window_width; ++i ) {
			window.push_back( i );
			if ( i == nres || pose.residue(i).is_upper_terminus() ) break;
		}


		kinematics::MoveMap local_mm;
		local_mm.set_chi( seqpos, true );
		final_mm.set_chi( seqpos, true );
		for ( Size ii=1; ii<= window.size(); ++ii ) {
			Size const i( window[ ii ] );
			local_mm.set_bb( i, true );
			final_mm.set_bb( i, true );
			// disallow proline PHI
			if ( pose.residue(i).aa() == chemical::aa_pro ) local_mm.set( TorsionID( phi_torsion, BB, i ), false );
			if ( pose.residue(i).aa() == chemical::aa_pro ) final_mm.set( TorsionID( phi_torsion, BB, i ), false );
		}

		// if jumpmin
		if ( lastjumpmin ) local_mm.set_jump( pose.fold_tree().num_jump(), true );

		// dont minimize or calculate stats after each idealization in fast mode
		if ( fast ) {
			TR.Info << "forced ideal geometry on seqpos " << seqpos << std::endl;
			continue;
		}

		Real max_bb_angle_dev, avg_bb_angle_dev, max_chi_angle_dev, avg_chi_angle_dev;

		dihedral_distance( pose, start_pose, idealized, avg_bb_angle_dev, max_bb_angle_dev,
											 avg_chi_angle_dev, max_chi_angle_dev );

		TR.Info << "premin:  (pos,rmsd,avg-bb,max-bb,avg-chi,max-chi,score) " <<
			I( 4, seqpos ) << ' ' << pose.residue(seqpos).name1() << F( 9, 3, all_atom_rmsd( pose, start_pose ) ) <<
			F(9,3,avg_bb_angle_dev) << F(9,3,max_bb_angle_dev) <<
			F(9,3,avg_chi_angle_dev) << F(9,3,max_chi_angle_dev) <<
			F(12,3,scorefxn( pose ) ) << std::endl;

		AtomTreeMinimizer().run( pose, local_mm, scorefxn, options );
		//pose.dump_pdb( "post_min_"+right_string_of(seqpos,4,'0')+".pdb" );

		dihedral_distance( pose, start_pose, idealized, avg_bb_angle_dev, max_bb_angle_dev,
											 avg_chi_angle_dev, max_chi_angle_dev );

		TR.Info << "postmin: (pos,rmsd,avg-bb,max-bb,avg-chi,max-chi,score) " <<
			I(4,seqpos) << ' ' << pose.residue(seqpos).name1() << F(9,3,all_atom_rmsd(pose,start_pose)) <<
			F(9,3,avg_bb_angle_dev) << F(9,3,max_bb_angle_dev) <<
			F(9,3,avg_chi_angle_dev) << F(9,3,max_chi_angle_dev) <<
			F(12,3,scorefxn( pose ) ) << std::endl;
		scorefxn.show( TR, pose );
	}

	if ( lastjumpmin ) final_mm.set_jump( pose.fold_tree().num_jump(), true );

	// final minimize
	Real max_bb_angle_dev, avg_bb_angle_dev, max_chi_angle_dev, avg_chi_angle_dev;
	dihedral_distance( pose, start_pose, idealized, avg_bb_angle_dev, max_bb_angle_dev,
										 avg_chi_angle_dev, max_chi_angle_dev );

	TR.Info << "pre-finalmin: (pos,rmsd,avg-bb,max-bb,avg-chi,max-chi,score) " <<
		F(9,3,all_atom_rmsd(pose,start_pose)) <<
		F(9,3,avg_bb_angle_dev) << F(9,3,max_bb_angle_dev) <<
		F(9,3,avg_chi_angle_dev) << F(9,3,max_chi_angle_dev) <<
		F(12,3,scorefxn( pose ) ) << std::endl;

	AtomTreeMinimizer().run( pose, final_mm, scorefxn, options );

	dihedral_distance( pose, start_pose, idealized, avg_bb_angle_dev, max_bb_angle_dev,
										 avg_chi_angle_dev, max_chi_angle_dev );

	TR.Info << "post-finalmin: (pos,rmsd,avg-bb,max-bb,avg-chi,max-chi,score) " <<
		F(9,3,all_atom_rmsd(pose,start_pose)) <<
		F(9,3,avg_bb_angle_dev) << F(9,3,max_bb_angle_dev) <<
		F(9,3,avg_chi_angle_dev) << F(9,3,max_chi_angle_dev) <<
		F(12,3,scorefxn( pose ) ) << std::endl;

	scorefxn.show( TR.Info, pose );
} // basic_idealize


void
IdealizeMover::apply( pose::Pose & pose )
{
	using namespace scoring;
	using namespace core::options;
	using namespace core::options::OptionKeys;

	pose::Pose original_pose = pose;
	pose::Pose unmodified_pose = pose;
	//pose.dump_pdb("idl_initial.pdb");

	// save a copy of the pose's constraints
	scoring::constraints::ConstraintSetOP original_cst_set( pose.constraint_set()->clone() );
	pose.constraint_set( NULL );
	// add virtual residue at the end
	//Size const old_root( pose.fold_tree().root() );
	if ( pose.residue( pose.total_residue() ).aa() != core::chemical::aa_vrt ) {
		pose.append_residue_by_jump(
			*conformation::ResidueFactory::create_residue( pose.residue(1).residue_type_set().name_map( "VRT" ) ),
				pose.total_residue()/2
		);
		original_pose.append_residue_by_jump(
			*conformation::ResidueFactory::create_residue( original_pose.residue(1).residue_type_set().name_map( "VRT" ) ),
			original_pose.total_residue()/2
		);
	}

	Size const nres( pose.total_residue() ); // includes pseudo-rsd

	{
		kinematics::FoldTree f( pose.fold_tree() );
		f.reorder( nres );
		pose.fold_tree( f );
	}

	// setup scorefunction
	scoring::ScoreFunction scorefxn;
	scorefxn.set_weight( atom_pair_constraint,  atom_pair_constraint_weight_ );
	scorefxn.set_weight( coordinate_constraint, coordinate_constraint_weight_ );
	scorefxn.set_weight( dslf_ss_dst, 0.5 );
	// keep prolines closed during idealizations. This is analogous to the above
	// line keeping disulphides together.
	if ( pose.is_fullatom() ) {
		scorefxn.set_weight( pro_close, 0.5 );
	}

	// setup constraints
	setup_idealize_constraints( pose );

	// by default idealize everything
	if ( pos_list_.size() == 0 ) {
		for ( Size i = 1; i <= nres-1; ++i ) {
			pos_list_.push_back( i );
		}
	}


	if( !option[ core::options::OptionKeys::run::dry_run ]() ){
		basic_idealize( pose, pos_list_, scorefxn, fast_ );
	}

	// remove that virtual residue now!
	pose::Pose final_pose = unmodified_pose;
	for ( Size ii = 1; ii <= unmodified_pose.total_residue(); ++ii ) {
		final_pose.replace_residue( ii, pose.residue( ii ), false );
	}
	pose = final_pose;

	// restore the original constraint set
	pose.constraint_set( original_cst_set );

	/// Pose must be rescored after the original constraint set is restored.
	scorefxn( pose );

	TR.Info << "RMS between original pose and idealised pose: "
	        << core::scoring::CA_rmsd( unmodified_pose, pose ) << " CA RMSD, "
	        << core::scoring::all_atom_rmsd( unmodified_pose, pose ) << " All-Atom RMSD, "
					<< std::endl;
} // apply

} // namespace idealize
} // namespace protocols
