// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/jd2/BOINCJobDistributor.cc
/// @brief  implementation of BOINCJobDistributor
/// @author Mike Tyka

// Project headers
// This has to come before boinc.hh or we get this error on VC++
// '_read' : is not a member of 'std::basic_istream<_Elem,_Traits>'
#include <utility/io/izstream.hh>

#ifdef BOINC
	#include <protocols/boinc/boinc.hh>
#endif // BOINC

#include <core/io/silent/util.hh>

#ifdef BOINC
#ifdef USEMPI
Throw a compiler error because MPI and BOINC cannot be used together!
If you got this message, something is wrong with your build settings.
#endif
#endif

// Unit headers
#include <protocols/jd2/BOINCJobDistributor.hh>
// Package headers
#include <protocols/jd2/JobOutputter.hh>
#include <protocols/jd2/Job.hh>

#include <protocols/moves/Mover.hh>

// Utility headers
#include <core/util/Tracer.hh>
#include <core/options/option.hh>
#include <utility/exit.hh>
#include <numeric/random/random.hh>

// Option headers
#include <core/options/keys/out.OptionKeys.gen.hh>


// C++ headers
#include <string>

static core::util::Tracer TR("protocols.jd2.BOINCJobDistributor");


namespace protocols {
namespace jd2 {

///@details constructor.  Notice it calls the parent class!  It also builds some internal variables for determining
///which processor it is in MPI land.
BOINCJobDistributor::BOINCJobDistributor() :
  ShuffleFileSystemJobDistributor()
{
#ifdef BOINC
	protocols::boinc::Boinc::worker_startup();
#endif

}

///@brief dtor
///WARNING WARNING!  SINGLETONS' DESTRUCTORS ARE NEVER CALLED IN MINI!  DO NOT TRY TO PUT THINGS IN THIS FUNCTION!
///here's a nice link explaining why: http://www.research.ibm.com/designpatterns/pubs/ph-jun96.txt
BOINCJobDistributor::~BOINCJobDistributor()
{ }


void
BOINCJobDistributor::go( protocols::moves::MoverOP mover )
{

	ShuffleFileSystemJobDistributor::go( mover );

	// gzip the output silent files.
	core::io::silent::gzip();

	// ideally these would be called in the dtor but the way we have the singleton pattern set up the dtors don't get
	// called
#ifdef BOINC
	Jobs const & jobs( get_jobs() );
	JobOutputterOP outputter = job_outputter();

	// Count completed jobs:
	core::Size count=0;
	for( core::Size i = 1; i <= jobs.size(); i ++ ){
		if( outputter->job_has_completed( jobs[ i ] ) ) count ++;
	}
	protocols::boinc::Boinc::worker_finish_summary( count + 1, count + 1, jobs.size() );
	protocols::boinc::Boinc::worker_shutdown(); // Does not return.
#endif


}


///@brief dummy for master/slave version
core::Size
BOINCJobDistributor::get_new_job_id()
{
#ifdef BOINC
	if( next_random_job() > 0 ){
		if (protocols::boinc::Boinc::worker_is_finished( next_random_job() - 1 )) return 0; // we're done no matter what nstruct syays
	}
#endif
	return ShuffleFileSystemJobDistributor::get_new_job_id();
}

void
BOINCJobDistributor::mark_current_job_id_for_repetition()
{
	// do nothing - no repetitions allowed. Behave as if job had succeeded (handled by job_failed)
}

void BOINCJobDistributor::job_failed( core::pose::Pose& pose, bool /*will_retry*/ ) {
	current_job()->set_status_prefix("C");
	job_succeeded( pose );	//i.e., job_outputter_->final_pose( current_job(), pose );
}
void BOINCJobDistributor::begin_critical_section() {
#ifdef BOINC
	boinc_begin_critical_section();
#endif // BOINC
}

void BOINCJobDistributor::end_critical_section() {
#ifdef BOINC
	boinc_end_critical_section();
#endif // BOINC
}


}//jd2
}//protocols


