// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/jd2/Parser.cc
/// @brief  August 2008 job distributor as planned at RosettaCon08 - Interface base class Parser
/// @author Sarel Fleishman sarelf@u.washington.edu

#include <protocols/jd2/DockDesignParser.hh>
#include <protocols/jd2/Job.hh>

#include <core/types.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/pose/Pose.hh>

#include <core/io/database/open.hh>
#include <ObjexxFCL/string.functions.hh>
#include <core/pack/task/operation/TaskOperationFactory.hh>
#include <core/options/option.hh>
// Utility Headers
#include <utility/vector1.hh>
#include <protocols/ProteinInterfaceDesign/DockDesignFilterFactory.hh>
#include <protocols/filters/Filter.hh>
#include <utility/Tag/Tag.hh>
#include <core/scoring/ScoreType.hh>

// movers
#include <protocols/moves/ConstraintSetMover.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/moves/MoverFactory.hh>
#include <protocols/moves/MutateResidue.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/ScoreMover.hh>
#include <protocols/moves/SwitchResidueTypeSetMover.hh>
#include <protocols/ProteinInterfaceDesign/DockDesign.hh>
#include <protocols/ProteinInterfaceDesign/movers/BackrubDDMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/BestHotspotCstMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/BuildAlaPose.hh>
#include <protocols/ProteinInterfaceDesign/movers/DesignMinimizeHbonds.hh>
#include <protocols/ProteinInterfaceDesign/movers/DisulfideMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/DockAndRetrieveSidechains.hh>
#include <protocols/ProteinInterfaceDesign/movers/DomainAssembly.hh>
#include <protocols/ProteinInterfaceDesign/movers/DumpPdb.hh>
#include <protocols/ProteinInterfaceDesign/movers/FavorNativeResiduePreCycle.hh>
#include <protocols/ProteinInterfaceDesign/movers/HotspotHasherMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/LoopOver.hh>
#include <protocols/ProteinInterfaceDesign/movers/NullDesignMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/PeptideStapleDesignMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/PrepackMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/RepackMinimize.hh>
#include <protocols/ProteinInterfaceDesign/movers/SaveAndRetrieveSidechains.hh>
#include <protocols/ProteinInterfaceDesign/movers/SetAtomTree.hh>
#include <protocols/ProteinInterfaceDesign/movers/SetupHotspotConstraintsMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/TryRotamers.hh>
#include <protocols/ProteinInterfaceDesign/movers/InterfaceRecapitulationMover.hh>
#include <protocols/toolbox/PoseMetricCalculators/SasaCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/InterfaceSasaDefinitionCalculator.hh>
// filters
#include <protocols/ProteinInterfaceDesign/filters/DisulfideFilter.hh>
#include <protocols/ProteinInterfaceDesign/filters/RmsdFilter.hh>
#include <protocols/ProteinInterfaceDesign/dock_design_filters.hh>
#include <protocols/ProteinInterfaceDesign/filters/InterfaceHolesFilter.hh>
#include <protocols/filters/AtomicContactFilter.hh>

// DNA interface TaskOperations
#include <protocols/dna/RestrictDesignToProteinDNAInterface.hh>
// DNA interface Movers
#include <protocols/dna/DnaInterfaceMinMover.hh>
#include <protocols/dna/DnaInterfacePacker.hh>
#include <protocols/dna/DnaInterfaceMultiStateDesign.hh>
#include <protocols/dna/DesignProteinBackboneAroundDNA.hh>

// constraint types
#include <core/scoring/constraints/ConstraintIO.hh>
#include <protocols/constraints_additional/SequenceProfileConstraint.hh>
#include <core/sequence/SequenceProfile.hh> // to pass NULL pointer for constraint prototype

// C++ headers
#include <iostream>
#include <sstream>
#include <fstream>
#include <map>

// option key includes

#include <core/options/keys/parser.OptionKeys.gen.hh>
#include <protocols/moves/DataMap.hh>

namespace protocols {
namespace jd2 {

using namespace core;
	using namespace options;
	using namespace pack;
	using namespace scoring;
using namespace moves;

static util::Tracer TR( "protocols.jd2.DockDesignParser" );

DockDesignParser::DockDesignParser()
	: Parser(),
		taskop_factory_( new TaskOperationFactory ),
		mover_factory_( new MoverFactory ),
		ddfilter_factory_( new DockDesignFilterFactory )
{
	register_factory_prototypes();
}

DockDesignParser::~DockDesignParser(){}

typedef utility::Tag::TagPtr TagPtr;
typedef std::vector< TagPtr > TagPtrs;

/// @details Uses the Tag interface to the xml reader library in boost to parse an xml file that contains design
/// protocol information. A sample protocol file can be found in src/pilot/apps/sarel/dock_design.protocol.
/// SCOREFXNS provides a way to define scorefunctions as they are defined in the rosetta database, using the
/// weights/patch convenctions. Several default scorefunctions are preset and can be used without defining them
/// explicitly.
/// FILTERS defines a set of filters that can be used together with the dockdesign movers to prune out poses that
/// don't meet certain criteria
/// MOVERS defines the movers that will be used
/// PROTOCOLS is the only order-sensitive section where subsequent movers and filters are expected to be defined.
/// These movers and filters were defined in the previous two sections. The order in which the protocols are
/// specified by the user will be maintained by the DockDesign mover.
/// APPLY_TO_POSE This section allows for certain movers to be applied to the pose prior to starting the DockDesign
/// protocol. For instance, these movers could set constraints, such as favor_native_residue. In this case, for
/// example, the weights of res_type_constraint in all of the scorefunctions that are defined in SCOREFXNS or by
/// default are set to 1.0, but can be changed by the user globally (in the definition of the weight on the constraint),
/// or in particular for each of the scorefunctions by changing the relevant term (which is set by default to the
/// global value).
/// Notice that the order of the sections by which the protocol is written doesn't matter, BUT the order of the
/// mover-filter pairs in PROTOCOLS section does matter.
bool
DockDesignParser::generate_mover_from_pose( JobCOP job, Pose & pose, MoverOP & in_mover, bool new_input ){

	using namespace ProteinInterfaceDesign;

	bool modified_pose( false );

	if( !new_input ) return modified_pose;

	in_mover = new ProteinInterfaceDesign::DockDesign;

	std::string const dock_design_filename( option[ OptionKeys::parser::protocol ] );
	TR << "dock_design_filename=" << dock_design_filename << '\n';
	utility::io::izstream fin;
	fin.open(dock_design_filename.c_str() );
	runtime_assert( fin.good() );
	TagPtr tag = utility::Tag::Tag::create(fin);
	fin.close();
	TR << tag << '\n';
	runtime_assert( tag->getName() == "dock_design" );

	Movers_map movers;
	protocols::filters::Filters_map filters;
	DataMap data; // abstract objects, such as scorefunctions, to be used by filter and movers

	MoverOP mover;
	ScoreFunctionOP scorefxn;

	typedef std::pair< std::string const, MoverOP > StringMover_pair;
	typedef std::pair< std::string const, protocols::filters::FilterCOP > StringFilter_pair;
	typedef std::pair< std::string const, ScoreFunctionOP > StringScorefxn_pair;
	typedef std::pair< std::string const, StringScorefxn_pair > ScorefxnObjects_pair;

	Real favor_nat( 0.0 ); // the weight on the res_type_constraint score type
	Real hotspot_hash( 0.0 ); // the weight on the backbone_stub_constraint score type

///// APPLY TO POSE
	if ( tag->hasTag("APPLY_TO_POSE") ) { // section is not mandatory
		hotspot_hashing::HotspotStubSetOP hotspot_stub_set;
	/// apply to pose may affect all of the scorefxn definitions below, so it is called first.
		TagPtrs const apply_tags( tag->getTag("APPLY_TO_POSE")->getTags() );
		bool has_profile( false ), has_fnr( false ); // to see that the user hasn't turned both on by mistake

		for( TagPtrs::const_iterator apply_tag_it=apply_tags.begin(); apply_tag_it!=apply_tags.end();
			 ++apply_tag_it ) {
			TagPtr const apply_tag_ptr = *apply_tag_it;

			if( apply_tag_ptr->getName() == "profile" ){
					using namespace core::sequence;
					using namespace protocols::constraints_additional;

					has_profile = true;
					Real const profile_wgt( apply_tag_ptr->getOption< Real >( "weight", 0.25 ));
					std::string const input_file_name( job->input_tag() );
					core::Size const wheres_period( input_file_name.find_first_of( "." ) );

					std::string const dflt_cst_file_name( input_file_name.substr(0, wheres_period ) + ".cst" );
					std::string const cst_file_name( apply_tag_ptr->getOption< std::string >( "file_name", dflt_cst_file_name ) );

					SequenceProfileOP profile = new SequenceProfile;
					profile->read_from_checkpoint( cst_file_name );
					for( core::Size seqpos( 1 ), end( pose.total_residue() ); seqpos <= end; ++seqpos )
						pose.add_constraint( new SequenceProfileConstraint( pose, seqpos, profile ) );

					TR<<"Added sequence profile constraints specified in file "<<cst_file_name<<". Changed all scorefxns to have profile weights of "<<profile_wgt<<std::endl;
					favor_nat = profile_wgt; // sequence profile constraint uses the same score as favor native residue
					modified_pose = true;
			}
			if( apply_tag_ptr->getName() == "favor_native_residue" ) {
				has_fnr = true;
				Real const fnr_weight( apply_tag_ptr->getOption<Real>( "bonus", 1.5 ) );
				mover  = new movers::FavorNativeResiduePreCycle( fnr_weight );
				mover->apply( pose );
				TR<<"applying favor native residue to pose with weight: "<<fnr_weight<<'\n';
				favor_nat = 1.0;
				modified_pose = true;
			}
			if( has_profile && has_fnr ){
				TR<<"The xml file contains both favor native residue and profile constraints. That is not allowed"<<std::endl;
				runtime_assert( has_profile && has_fnr );
			}
			if( apply_tag_ptr->getName() == "hashing_constraints" ) {
				std::string const scorefxn( apply_tag_ptr->getOption<std::string>( "scorefxn", "score12" ) );
				std::string const hotspot_fname( apply_tag_ptr->getOption<std::string>( "stubfile", "stubs.pdb" ) );
				Size const chain_to_redesign( apply_tag_ptr->getOption<Size>( "redesign_chain", 2 ) );

				Real const cb_force( apply_tag_ptr->getOption<Real>( "cb_force", 0.5 ) );
				Real const worst_allowed_stub_bonus( apply_tag_ptr->getOption<Real>( "worst_allowed_stub_bonus", 0 ) );
				bool const apply_self_energies( apply_tag_ptr->getOption<bool>( "apply_stub_self_energies", 0 ) );
				Real const bump_cutoff( apply_tag_ptr->getOption<Real>( "apply_stub_bump_cutoff", 10. ) );
				bool const apply_ambiguous_constraints( apply_tag_ptr->getOption<bool>( "pick_best_energy_constraint", 1 ) );
				bool const colonyE( apply_tag_ptr->getOption<bool>( "colonyE", 0 ) );

				hotspot_stub_set = new hotspot_hashing::HotspotStubSet;
				hotspot_stub_set->read( hotspot_fname );

				MoverOP add_cst = new movers::SetupHotspotConstraintsMover( //hotspot_hash_packer_taskCOP,
												*hotspot_stub_set, chain_to_redesign, cb_force, worst_allowed_stub_bonus,
												apply_self_energies, bump_cutoff, apply_ambiguous_constraints, colonyE );
				add_cst->apply( pose );
				hotspot_hash = 1.0;
				TR<<"applying hotspot hashing constraints top pose with stubs file "<<hotspot_fname<<" cb_force weight of "<<cb_force<<
					", apply ambiguous constraints set to "<<apply_ambiguous_constraints<< " and colonyE set to " << colonyE << "\n";
				data.add( "constraints" , "hotspot_stubset", hotspot_stub_set );
				modified_pose = true;
			}
		}
	}

//setting up some defaults
	protocols::filters::FilterOP true_filter = new protocols::filters::TrueFilter;
	protocols::filters::FilterOP false_filter = new protocols::filters::FalseFilter;
	filters.insert( StringFilter_pair( "true_filter", true_filter ) );
	filters.insert( StringFilter_pair( "false_filter", false_filter ) );
	MoverOP null_mover = new movers::NullDesignMover();
	movers.insert( StringMover_pair( "null", null_mover) );
// default scorefxns
	ScoreFunctionOP score12 = ScoreFunctionFactory::create_score_function( STANDARD_WTS, SCORE12_PATCH );
	ScoreFunctionOP docking_score = ScoreFunctionFactory::create_score_function( STANDARD_WTS, DOCK_PATCH );
	ScoreFunctionOP soft_rep = ScoreFunctionFactory::create_score_function( SOFT_REP_DESIGN_WTS );
	ScoreFunctionOP docking_score_low = ScoreFunctionFactory::create_score_function( "interchain_cen" );
	ScoreFunctionOP score4L = ScoreFunctionFactory::create_score_function( "cen_std", "score4L" );

	data.add("scorefxns","score12", score12 );
	data.add( "scorefxns","score_docking", docking_score );
	data.add( "scorefxns" , "soft_rep", soft_rep );
	data.add( "scorefxns" , "score_docking_low", docking_score_low );
	data.add("scorefxns","score4L", score4L );

	if( favor_nat || hotspot_hash ) {
		using namespace utility::pointer;
		for( std::map< std::string, ReferenceCountOP >::const_iterator it=(data)[ "scorefxns" ].begin(); it!=(data)[ "scorefxns" ].end(); ++it ){
			ScoreFunctionOP scorefxn( *data.get< ScoreFunction * >( "scorefxns", it->first ) );
			scorefxn->set_weight( res_type_constraint, favor_nat );
			scorefxn->set_weight( backbone_stub_constraint, hotspot_hash );
			TR<<"setting "<<it->first<<" res_type_constraint to "<<favor_nat<<" and backbone_stub_constraint to "<<hotspot_hash<<"\n";
		}
	}
 //default scorefxns end

	TagPtrs const scorefxn_tags( tag->getTag( "SCOREFXNS" )->getTags() );
	for( TagPtrs::const_iterator scorefxn_tag_it=scorefxn_tags.begin(); scorefxn_tag_it!=scorefxn_tags.end();
		 ++scorefxn_tag_it ) {//read user defined scorefxns
		TagPtr const scorefxn_tag = *scorefxn_tag_it;
		using namespace core::scoring;

		ScoreFunctionOP in_scorefxn;
		std::string const scorefxn_name( scorefxn_tag->getName() );
		std::string const scorefxn_weights( scorefxn_tag->getOption<std::string>( "weights", "standard" ) );
    if(  scorefxn_tag->hasOption( "weights" ) && scorefxn_tag->hasOption( "patch" ) ) {
			std::string const scorefxn_patch( scorefxn_tag->getOption<std::string>( "patch" ) );
			in_scorefxn = ScoreFunctionFactory::create_score_function( scorefxn_weights, scorefxn_patch);
			TR << "defined score function \"" << scorefxn_name << "\" with weights \""
				<< scorefxn_weights << "\" and patch \"" << scorefxn_patch << "\"\n";
		}
		else if( scorefxn_tag->hasOption( "weights" ) ) {
			in_scorefxn = ScoreFunctionFactory::create_score_function( scorefxn_weights );
			TR << "defined score function \"" << scorefxn_name << "\" with weights \""
				<< scorefxn_weights << "\"\n";
		}
		else {
			in_scorefxn = new ScoreFunction;
			in_scorefxn->reset();
			TR << "Defined " << scorefxn_name << " with all-zero weights.\n";
		}
		std::vector< TagPtr > const scorefxn_mod_tags( scorefxn_tag->getTags() );
		for( std::vector< TagPtr >::const_iterator mod_it=scorefxn_mod_tags.begin(); mod_it!=scorefxn_mod_tags.end(); ++mod_it ) {
			TagPtr const mod_tag = *mod_it;
			if( mod_tag->getName() == "Reweight" ) {
				std::string const scoretype_name( mod_tag->getOption<std::string>( "scoretype" ) );
				core::Real const weight( mod_tag->getOption<core::Real>( "weight" ) );
				TR<<"setting "<<scorefxn_name<<" weight " << scoretype_name << " to " << weight<<'\n';
				core::scoring::ScoreType const type = score_type_from_name( scoretype_name );
				in_scorefxn->set_weight( type, weight );
			}
		}

		// weights for arbitrary ScoreFunctions should be tampered with only as a consequence of user input--NEVER by default

		// favor native residue
		if ( scorefxn_tag->hasOption("fnr") ) {
			Real const fnr( scorefxn_tag->getOption<Real>( "fnr", favor_nat ) );
			TR<<"setting "<<scorefxn_name<<" res_type_constraint to "<<fnr<<'\n';
			in_scorefxn->set_weight( res_type_constraint, fnr );
		}
		// hotspot hash constraint
		if ( scorefxn_tag->hasOption("hs_hash") ) {
			Real const hs_hash( scorefxn_tag->getOption<Real>( "hs_hash", hotspot_hash ) );
			TR<<"setting "<<scorefxn_name<<" backbone_stub_constraint to "<<hs_hash<<'\n';
			in_scorefxn->set_weight( backbone_stub_constraint, hs_hash );
		}

		data.add( "scorefxns" , scorefxn_name, in_scorefxn );
	}//end user-defined scorefxns
	TR.flush();

////// TaskOperations
	if ( tag->hasTag( "TASKOPERATIONS" ) ) {

		TagPtrs const TO_tags( tag->getTag( "TASKOPERATIONS" )->getTags() );
		for( TagPtrs::const_iterator tp( TO_tags.begin() ), tp_e( TO_tags.end() ); tp != tp_e; ++tp ) {
			std::string const type( (*tp)->getName() );
			if ( ! (*tp)->hasOption("name") )
				utility_exit_with_message( "Can't create unnamed TaskOperation (type: " + type + ")" );
			std::string const name( (*tp)->getOption<std::string>("name") );
			if ( data.has( "task_operations", name ) ) {
				TR << "WARNING WARNING TaskOperation of name \"" << name
					<< "\" (with type " << type << ") already exists. Skipping\n" << tag << std::endl;
				continue;
			}
			task::operation::TaskOperationOP new_t_o( taskop_factory_->newTaskOperation( type, *tp ) );
			runtime_assert( new_t_o );
			data.add("task_operations", name, new_t_o );
			TR << "Defined TaskOperation named \"" << name << "\" of type " << type << std::endl;
		}
	}

////// Filters
	if ( tag->hasTag( "FILTERS" ) ) {

		TagPtrs const TO_tags( tag->getTag( "FILTERS" )->getTags() );
		for( TagPtrs::const_iterator tp( TO_tags.begin() ), tp_e( TO_tags.end() ); tp != tp_e; ++tp ) {
			std::string const type( (*tp)->getName() );
			if ( ! (*tp)->hasOption("name") )
				utility_exit_with_message("Can't define unnamed Filter of type " + type );
			std::string const user_defined_name( (*tp)->getOption<std::string>("name") );
			bool const name_exists( filters.find( user_defined_name ) != filters.end() );
			if ( name_exists ) {
				TR << "WARNING WARNING filter of name \"" << user_defined_name << "\" (with type " << type
					<< ") already exists. Skipping\n" << tag << std::endl;
				continue;
			}
			protocols::filters::FilterOP new_ddf( ddfilter_factory_->newFilter( *tp, data, filters, movers, pose ) );
			runtime_assert( new_ddf );
			filters.insert( std::make_pair( user_defined_name, new_ddf ) );
			TR << "Defined filter named \"" << user_defined_name << "\" of type " << type << std::endl;
		}
	}

////// MOVERS
	if ( ! tag->hasTag("MOVERS") )
			utility_exit_with_message("parser::protocol file must specify MOVERS section");

	TagPtrs const TO_tags( tag->getTag("MOVERS")->getTags() );
	for( TagPtrs::const_iterator tp( TO_tags.begin() ), tp_e( TO_tags.end() ); tp != tp_e; ++tp ) {
		std::string const type( (*tp)->getName() );
		if ( ! (*tp)->hasOption("name") )
			utility_exit_with_message("Can't define unnamed Mover of type " + type );
		std::string const user_defined_name( (*tp)->getOption<std::string>("name") );
		bool const name_exists( movers.find( user_defined_name ) != movers.end() );
		if ( name_exists ) {
			TR << "WARNING WARNING mover of name \"" << user_defined_name << "\" (with type " << type
				<< ") already exists. Skipping\n" << tag << std::endl;
			continue;
		}
		MoverOP new_mover( mover_factory_->newMover( *tp, data, filters, movers, pose ) );
		runtime_assert( new_mover );
		movers.insert( std::make_pair( user_defined_name, new_mover ) );
		TR << "Defined mover named \"" << user_defined_name << "\" of type " << type << std::endl;
	}

	TR.flush();
////// ADD MOVER FILTER PAIRS
	if ( !tag->hasTag("PROTOCOLS") )
			utility_exit_with_message("parser::protocol file must specify PROTOCOLS section");

	TagPtr const protocols_tag( tag->getTag("PROTOCOLS") );
	in_mover->parse_my_tag( protocols_tag, data, filters, movers, pose );
	TR.flush();
	return modified_pose;
}

///@brief Factories avoid requiring compiler dependencies for all possible constructible derived classes, by allowing runtime registration of derived class prototypes. However, this requires pre-registration of a derived type with the factory prior to asking the factory to return an instance of that type. This method registers those additional derived classes that are available for construction in the DockDesignParser context.
/// TO-DO: replace this manual factory registration system with a load-time-initialized singleton scheme (see r32404 for example)
void
DockDesignParser::register_factory_prototypes()
{
	if ( ! taskop_factory_ ) taskop_factory_ = new TaskOperationFactory;

		// note: by default TaskOperationFactory should already be aware of any TaskOperations in namespace core::pack::task::operation
		taskop_factory_->add_type( new dna::RestrictDesignToProteinDNAInterface );

	if ( ! mover_factory_ ) mover_factory_ = new MoverFactory;

	using namespace ProteinInterfaceDesign;
	mover_factory_->add_type( new movers::BackrubDDMover );
	mover_factory_->add_type( new movers::BestHotspotCstMover );
	mover_factory_->add_type( new movers::BuildAlaPose );
	mover_factory_->add_type( new movers::DesignMinimizeHbonds );
	mover_factory_->add_type( new movers::DisulfideMover );
	mover_factory_->add_type( new movers::DockAndRetrieveSidechains );
	mover_factory_->add_type( new movers::DomainAssembly );
	mover_factory_->add_type( new movers::DumpPdb );
	mover_factory_->add_type( new movers::HotspotHasherMover );
	mover_factory_->add_type( new movers::LoopOver );
	mover_factory_->add_type( new movers::PeptideStapleDesignMover );
	mover_factory_->add_type( new movers::PrepackMover );
	mover_factory_->add_type( new movers::RepackMinimize );
	mover_factory_->add_type( new movers::SaveAndRetrieveSidechains );
	mover_factory_->add_type( new movers::SetAtomTree );
	mover_factory_->add_type( new movers::TryRotamers );
	mover_factory_->add_type( new movers::InterfaceRecapitulationMover );
	mover_factory_->add_type( new ConstraintSetMover );
	mover_factory_->add_type( new DockDesign );
	mover_factory_->add_type( new PackRotamersMover );
	mover_factory_->add_type( new MinMover );
	mover_factory_->add_type( new MutateResidue );
	mover_factory_->add_type( new ScoreMover );
	mover_factory_->add_type( new SwitchResidueTypeSetMover );
	mover_factory_->add_type( new dna::DnaInterfaceMinMover );
	mover_factory_->add_type( new dna::DnaInterfacePacker );
	mover_factory_->add_type( new dna::DesignProteinBackboneAroundDNA );
	mover_factory_->add_type( new dna::DnaInterfaceMultiStateDesign );

	if ( ! ddfilter_factory_ ) ddfilter_factory_ = new DockDesignFilterFactory;

	ddfilter_factory_->add_type( new protocols::filters::CompoundFilter );
	ddfilter_factory_->add_type( new BuriedUnsatHbondFilter );
	ddfilter_factory_->add_type( new DdgFilter );
	ddfilter_factory_->add_type( new EnergyPerResidueFilter );
	ddfilter_factory_->add_type( new HbondsToResidueFilter );
	ddfilter_factory_->add_type( new InterfaceSasaFilter );
	ddfilter_factory_->add_type( new NeighborTypeFilter );
	ddfilter_factory_->add_type( new ResidueBurialFilter );
	ddfilter_factory_->add_type( new ResidueDistanceFilter );
	ddfilter_factory_->add_type( new protocols::filters::StochasticFilter );
	ddfilter_factory_->add_type( new ResiduesInInterfaceFilter );
	ddfilter_factory_->add_type( new ScoreTypeFilter );
	ddfilter_factory_->add_type( new TerminusDistanceFilter );
	ddfilter_factory_->add_type( new AlaScan );
	ddfilter_factory_->add_type( new protocols::filters::FalseFilter );
	ddfilter_factory_->add_type( new protocols::filters::AtomicContactFilter );
	ddfilter_factory_->add_type( new protocols::ProteinInterfaceDesign::filters::DisulfideFilter );
	ddfilter_factory_->add_type( new protocols::ProteinInterfaceDesign::filters::InterfaceHolesFilter );
	ddfilter_factory_->add_type( new protocols::ProteinInterfaceDesign::filters::RmsdFilter );

	// also register some constraint types with the ConstraintFactory (global singleton class)
	// this allows derived non-core constraints to be constructed from string definitions in constraints files
	using namespace core::scoring::constraints;
	ConstraintFactory & cstf( ConstraintIO::get_cst_factory() );
	cstf.add_type( new constraints_additional::SequenceProfileConstraint(
		Size(), utility::vector1< id::AtomID >(), NULL ) );

	// register calculators
	core::Size const chain1( 1 ), chain2( 2 );
	core::pose::metrics::PoseMetricCalculatorOP int_sasa_calculator = new protocols::toolbox::PoseMetricCalculators::InterfaceSasaDefinitionCalculator( chain1, chain2 );
	core::pose::metrics::CalculatorFactory::Instance().register_calculator( "sasa_interface", int_sasa_calculator );
}

}//jd2
}//protocols
