// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/jd2/GenericJobInputter.cc
/// @brief  August 2008 job distributor as planned at RosettaCon08 - Base class GenericJobInputter
/// @author Oliver Lange

///Unit headers
#include <protocols/jd2/GenericJobInputter.hh>
#include <protocols/jd2/Job.hh>

///Project headers
#include <core/io/pdb/pose_io.hh>
#include <core/pose/Pose.hh>

///Utility headers
#include <core/util/Tracer.hh>
#include <core/options/option.hh>
#include <core/options/util.hh>
#include <utility/vector1.hh>

///C++ headers
#include <string>

// option key includes
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/jd2.OptionKeys.gen.hh>

static core::util::Tracer tr("protocols.jd2.GenericJobInputter");

namespace protocols {
namespace jd2 {

 protocols::jd2::GenericJobInputter::GenericJobInputter(){
 	tr.Debug << "Instantiate GenericJobInputter" << std::endl;
 }

/// @details This function will first see if the pose already exists in the Job.
/// If not, it will read it into the pose reference, and hand a COP cloned from
/// that pose to the Job. If the pose pre-exists it just copies the COP's pose
/// into it.
void protocols::jd2::GenericJobInputter::pose_from_job( core::pose::Pose& pose, JobOP job){
	// what should be the generic behaviour ? -->create empty pose ? or pose from fasta ?
	// I think more generic is the empty pose
	// WRONG! More generic is Pose from Fasta. That way you can divorce the idea
	// of a protocol from its method of creating Poses. If you don't have a protocol
	// and its method for creating a Pose independent, then there's little point in
	// having a Job Distributor.
	// How about a FastaJobInputter ... i think it is a valid request, but less generic this one.
	// Generic really shouldn't mean "let the protocol decide where the Pose
	// comes from." More generic than having protocols instantiate Poses is to
	// have the protocol not care about the source of the Pose. That way the
	// protocol is smaller and there's better separation of the concepts of Pose
	// creation and Pose manipulation ideas. This separation is critical so that
	// we can string together protocols inside of minirosetta, otherwise someone
	// will have to refactor whatever mess you leave behind.
	tr.Debug << "GenericJobInputter::pose_from_job" << std::endl;
	if( !job->inner_job()->get_pose() ){
		return ; // generic doesn't do this ...
// 		//		option[ in::file::fasta ]()[1] )[1].sequence();
// 	std::string sequence = core::sequence::read_fasta_file( job->inner_job()->input_tag() );
// 	chemical::make_pose_from_sequence(
// 		 pose,
// 		 sequence,
// 		 *( chemical::ChemicalManager::get_instance()->residue_type_set( chemical::CENTROID ))
// 	 );
// 	job->inner_job_nonconst()->set_pose( pose );

		//tr.Debug << "filling pose from Generic " << job->inner_job()->input_tag() << std::endl;
	} else {
		pose = *(job->inner_job()->get_pose());
		tr.Debug << "filling pose from saved copy " << job->inner_job()->input_tag() << std::endl;
	}
}

/// @details this function determines what jobs exist from -s/-l
void protocols::jd2::GenericJobInputter::fill_jobs( Jobs & jobs ){
	tr.Debug << "GenericJobInputter::fill_jobs" << std::endl;

	jobs.clear(); //should already be empty anyway

	core::Size const nstruct( get_nstruct () );

	//note that we are not really using the second and third fields in this implementation
	using core::options::OptionKeys::jd2::generic_job_name; //This option defaults to 'S' for original behavior
	InnerJobOP ijob( new InnerJob( core::options::option[ generic_job_name ].value() , nstruct ) );

	for( core::Size index = 1; index <= nstruct; ++index) {
		jobs.push_back( JobOP( new Job( ijob, index ) ) );
		tr.Trace << "create job index " << index << std::endl;
	}//loop over nstruct
}//fill_jobs

/// @brief Return the type of input source that the GenericJobInputter is currently
///  using.
/// @return Always <em>POSE</em>.
JobInputterInputSource::Enum GenericJobInputter::input_source() const {
	return JobInputterInputSource::POSE;
}

}//jd2
}//protocols
