// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/jd2/Job.cc
/// @brief  August 2008 job distributor as planned at RosettaCon08 - Job classes
/// @author Steven Lewis smlewi@gmail.com

///Unit headers
#include <protocols/jd2/Job.hh>

///Project headers
#include <protocols/jobdist/Jobs.hh>

#include <core/pose/Pose.hh>

///Utility headers
#include <core/util/Tracer.hh>
#include <core/options/option.hh>
#include <core/options/util.hh>
#include <utility/vector1.hh>

///C++ headers
#include <string>

static core::util::Tracer TR("protocols.jd2.Job");

namespace protocols {
namespace jd2 {


/////////////////////////////InnerJob/////////////////////////////
InnerJob::InnerJob( std::string const & input_tag, core::Size nstruct_max )
	: input_tag_(input_tag), nstruct_max_(nstruct_max), pose_(NULL), bad_( false )
{
	//commented these out... somehow they don't respond to -mute
	//TR.Trace << "Using InnerJob (base class) for JobDistributor" << std::endl;
}

InnerJob::InnerJob( core::pose::PoseCOP pose, std::string const & input_tag, core::Size nstruct_max )
	: input_tag_(input_tag), nstruct_max_(nstruct_max), pose_( pose ), bad_( false )
{
	//TR.Trace << "Using InnerJob (base class) for JobDistributor" << std::endl;
}

InnerJob::~InnerJob(){}

std::string const & InnerJob::input_tag() const { return input_tag_; }

core::Size InnerJob::nstruct_max() const { return nstruct_max_; }

core::pose::PoseCOP InnerJob::get_pose() const { return pose_; }

void InnerJob::set_pose( core::pose::PoseCOP pose ) { pose_ = pose; }


////////////////////////////Job/////////////////////////////
Job::Job( InnerJobOP inner_job, core::Size nstruct_index )
	: inner_job_(inner_job),
		nstruct_index_(nstruct_index),
		status_prefix_( "" ),
		completed_(false)
{
	//TR.Trace << "Using Job (base class) for JobDistributor" << std::endl;
}

Job::~Job(){}

///@brief returns a copy of this object whose "output fields" are zeroed out.  Used by the JobDistributor in cases where the job fails and must be retried to prevent accumulation of Job state after a failure.  This implementation was chosen over a clear_all_output function to prevent mover A from deleting mover B's hard work!  You probably should not be trying to call this function.
JobOP Job::copy_without_output() const{
	return new Job(inner_job_, nstruct_index_);
}

InnerJobCOP Job::inner_job() const { return inner_job_; }

InnerJobOP Job::inner_job_nonconst() { return inner_job_; }

core::Size Job::nstruct_index() const { return nstruct_index_; }

//functions for loading output info into the job
///@brief add an output string
void Job::add_string( std::string const & string_in ){
	long_strings_.push_back(string_in);
}

///@brief adds output strings
void Job::add_strings( Strings const & strings )
{
	long_strings_.insert( long_strings_.end(), strings.begin(), strings.end() );
}

///@brief add a string/string pair
void Job::add_string_string_pair( std::string const & string1, std::string const & string2 ){
	string_string_pairs_.push_back(std::make_pair(string1, string2));
}

///@brief add a string/real pair
void Job::add_string_real_pair( std::string const & string_in, core::Real const real_in ){
	string_real_pairs_.push_back(std::make_pair(string_in, real_in));
}

//functions for returning output info from the job.  You get iterators so that this interface can stay constant as the underlying implementation changes
Job::Strings::const_iterator Job::output_strings_begin() const
{ return long_strings_.begin(); }

Job::Strings::const_iterator Job::output_strings_end() const
{ return long_strings_.end(); }

Job::StringStringPairs::const_iterator Job::output_string_string_pairs_begin() const
{ return string_string_pairs_.begin(); }

Job::StringStringPairs::const_iterator Job::output_string_string_pairs_end() const
{ return string_string_pairs_.end(); }

Job::StringRealPairs::const_iterator Job::output_string_real_pairs_begin() const
{ return string_real_pairs_.begin(); }

Job::StringRealPairs::const_iterator Job::output_string_real_pairs_end() const
{ return string_real_pairs_.end(); }


JobOP const JD2_BOGUS_JOB( new Job( (new InnerJob("EMPTY_JOB_use_jd2", 0)), 0) );

}//jd2
}//protocols
